/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icommoneventobservers.h"


#include "icontrolmodule.h"
#include "idatareader.h"
#include "ierror.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "iparallel.h"
#include "ipicker.h"
#include "iscript.h"
#include "ishellfactory.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkPointWidget.h>
#include <vtkProcessObject.h>
#include <vtkRenderWindow.h>
#include <vtkTimerLog.h>

//
//  Templates
//
#include "iarraytemplate.h"


bool iAbortRenderEventObserver::mTheseAreBlocked = false;


//
//  iProgressEventObserver class
//
iProgressEventObserver* iProgressEventObserver::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return iRequiredCast<iProgressEventObserver>(INFO,iShellFactory::CreateEventObserver("Progress",vm));
}


iProgressEventObserver::iProgressEventObserver(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm)
{
	mAborted = false;
	mMode = _Default;
	mScript = 0;
	mProgressValue = 0.0;
}


void iProgressEventObserver::AttachScript(iScript *s)
{
	mScript = s;
}


void iProgressEventObserver::ExecuteBody(vtkObject *, unsigned long event, void *data)
{
	switch(event)
	{
	case vtkCommand::StartEvent:
		{
			this->Started(mMode);
			break;
		}
	case vtkCommand::ProgressEvent:
		{
			float f;
			if(data == 0) f = 0.0; else f = static_cast<double*>(data)[0];
			this->SetProgress(f);
			break;
		}
	case vtkCommand::EndEvent:
		{
			mScript = 0;
			this->Finished();
			break;
		}
	}
}


void iProgressEventObserver::Started(ProgressMode m)
{
	mMode = m;
	mAborted = false;
	mProgressValue = 0.0;
	this->ChildStarted();
}


void iProgressEventObserver::SetProgress(float f)
{
	mAborted = this->CheckAbort();
	if(mScript!=0 && mScript->CheckAbort()) mAborted = true;

	if(!mAborted && f>mProgressValue+0.0009)
	{
		mProgressValue = f;
		this->SetProgressBody(f);
	}
}


void iProgressEventObserver::Finished()
{
	this->ChildFinished();
}


void iProgressEventObserver::SetMode(ProgressMode m)
{
	mMode = m;
}


const char* iProgressEventObserver::GetLabel() const
{
	switch(mMode)
	{
	case _Reading:
		{
			return "Reading...";
		}
	case _Projecting:
		{
			return "Projecting...";
		}
	case _Rendering:
		{
			return "Rendering...";
		}
	case _Writing:
		{
			return "Writing...";
		}
	case _Operating:
		{
			return "Operating...";
		}
	case _Formatting:
		{
			return "Formatting...";
		}
	case _Shifting:
		{
			return "Shifting...";
		}
	case _Computing:
		{
			return "Computing...";
		}
	default:
		{
			return "";
		}
	}
}


//
//  iAbortRenderEventObserver class
//
iAbortRenderEventObserver* iAbortRenderEventObserver::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return iRequiredCast<iAbortRenderEventObserver>(INFO,iShellFactory::CreateEventObserver("AbortRender",vm));
}


iAbortRenderEventObserver::iAbortRenderEventObserver(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm)
{
	mCancelled = mInProgress = false;
	mRenderTimer = vtkTimerLog::New(); IERROR_ASSERT(mRenderTimer);
	mRenderTime = 0.0f;
	mInteractive = true;
}


iAbortRenderEventObserver::~iAbortRenderEventObserver()
{
	mRenderTimer->Delete();
}


void iAbortRenderEventObserver::ExecuteBody(vtkObject *caller, unsigned long event, void *)
{
	if(mTheseAreBlocked) return;

	switch(event)
	{
	case vtkCommand::StartEvent:
		{
			if(!mInProgress)
			{
				mInProgress = true;
				this->Started();
				mCancelled = false;
				mRenderTimer->StartTimer();
			}
			break;
		}
	case vtkCommand::AbortCheckEvent:
	case vtkCommand::ProgressEvent:
		{
			if(mInProgress)
			{
				mCancelled = this->CheckAbort();
				if(mCancelled)
				{
					vtkRenderWindow *rw = vtkRenderWindow::SafeDownCast(caller);
					if(rw != 0) rw->SetAbortRender(1);
					vtkProcessObject *po = vtkProcessObject::SafeDownCast(caller);
					if(po != 0) po->SetAbortExecute(1);
				}
			}
			break;
		}
	case vtkCommand::EndEvent:
		{
			if(mInProgress)
			{
				mRenderTimer->StopTimer();
				mRenderTime = mRenderTimer->GetElapsedTime();
				mInProgress = false;
				this->Finished();
				if(mCancelled)
				{
					//
					//  If rendering was cancelled, then different parts of the pipeline
					//  remain in different stages of completion, which confuses VTK.
					//  To avoid that we modify the time stamp of the data, so that next
					//  rendering starts from scratch. It is not an elegant solution,
					//  but it works.
					//
					this->GetViewModule()->GetReader()->ResetPipeline();
				}
			}
			break;
		}
	case vtkCommand::UserEvent:
		{
			this->PostFinished();
			break;
		}
	}
}


//
//  iSlaveAbortRenderEventObserver class
//
iSlaveAbortRenderEventObserver* iSlaveAbortRenderEventObserver::New(iAbortRenderEventObserver *master)
{
	IERROR_ASSERT(master);
	return new iSlaveAbortRenderEventObserver(master);
}


iSlaveAbortRenderEventObserver::iSlaveAbortRenderEventObserver(iAbortRenderEventObserver *m) : iEventObserver(m->GetViewModule()->GetControlModule()->GetShell()), mViewModule(m->GetViewModule())
{
	mMaster = m;
	mMaster->Register(this);
}


iSlaveAbortRenderEventObserver::~iSlaveAbortRenderEventObserver()
{
	mMaster->UnRegister();
}


void iSlaveAbortRenderEventObserver::ExecuteBody(vtkObject *caller, unsigned long event, void *)
{
	if(mMaster->AbortRenderEventObserversBlocked()) return;

	switch(event)
	{
	case vtkCommand::AbortCheckEvent:
	case vtkCommand::ProgressEvent:
		{
			if(mMaster!=0 && mMaster->IsCancelled())
			{
				vtkProcessObject *po = vtkProcessObject::SafeDownCast(caller);
				if(po != 0) po->SetAbortExecute(1);
			}
			break;
		}
	}
}


//
//  iPickEventObserver class
//
iPickEventObserver* iPickEventObserver::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return iRequiredCast<iPickEventObserver>(INFO,iShellFactory::CreateEventObserver("Pick",vm));
}


iPickEventObserver::iPickEventObserver(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm)
{
}


void iPickEventObserver::ExecuteBody(vtkObject *caller, unsigned long eventId, void *)
{
	switch (eventId)
	{
	case vtkCommand::StartPickEvent: 
		{ 
			this->Started();
			break; 
		}
	case vtkCommand::EndPickEvent: 
		{ 
			this->Finished();
			break; 
		}
	}
}


//
//  iParallelUpdateEventObserver class
//
iParallelUpdateEventObserver* iParallelUpdateEventObserver::New(iControlModule *cm)
{
	IERROR_ASSERT(cm);
	return iRequiredCast<iParallelUpdateEventObserver>(INFO,iShellFactory::CreateEventObserver("ParallelUpdate",cm));
}


iParallelUpdateEventObserver::iParallelUpdateEventObserver(iControlModule *cm) : iEventObserver(cm->GetShell()), mControlModule(cm)
{
}


void iParallelUpdateEventObserver::ExecuteBody(vtkObject *, unsigned long eventId, void *)
{
	if(eventId == iParallel::InformationEvent) this->UpdateInformation(); 
}


//
//  iAnimatorEventObserver class
//
iAnimatorEventObserver* iAnimatorEventObserver::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return iRequiredCast<iAnimatorEventObserver>(INFO,iShellFactory::CreateEventObserver("Animator",vm));
}


iAnimatorEventObserver::iAnimatorEventObserver(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm)
{
}


void iAnimatorEventObserver::ExecuteBody(vtkObject *, unsigned long eventId, void *)
{
	if(eventId == 0UL) this->OnCameraPath(0);
}


//
//  iMarkerEventObserver class
//
iMarkerEventObserver* iMarkerEventObserver::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iMarkerEventObserver(vm);
}


iMarkerEventObserver::iMarkerEventObserver(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm)
{
}


void iMarkerEventObserver::ExecuteBody(vtkObject *caller, unsigned long eventId, void *)
{
	vtkPointWidget *pw = vtkPointWidget::SafeDownCast(caller);
	if(pw == 0) return;

	switch (eventId)
	{
	case vtkCommand::InteractionEvent:
		{
			double *x = pw->GetPosition();
			iPosition p(this->GetViewModule());
			p = x;
			this->GetViewModule()->GetMarkerFamily()->GetCurrentMember()->Move(p);
			break;
		}
	case vtkCommand::EndInteractionEvent:
		{
			this->GetViewModule()->GetMarkerFamily()->GetCurrentMember()->FinishMoving();
		}
	}
}

