/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#include <sqlite3.h>

#define PLUGIN_NAME "DB IMSpector filter plugin"
#define PLUGIN_SHORT_NAME "DB"

#define SQLITE_SOCKET "/tmp/.imspectorsqlite"

#define CREATE_TABLE "CREATE TABLE IF NOT EXISTS lists ( " \
	"id integer PRIMARY KEY AUTOINCREMENT, " \
	"localid text, " \
	"remoteid text, " \
	"action integer NOT NULL, " \
	"type integer NOT NULL, " \
	"timestamp integer NOT NULL );" \

#define MATCH_ACTION_STATEMENT "SELECT COUNT(*) FROM lists WHERE " \
	"(localid=? OR localid IS NULL) AND " \
	"(remoteid=? OR remoteid IS NULL) AND action=?"
	
#define ADD_AWL_STATEMENT "INSERT INTO lists " \
	"(id, localid, remoteid, action, type, timestamp)  " \
	"VALUES (NULL, ?, ?, ?, ?, ?)"

#define ACTION_ACCEPT 1
#define ACTION_BLOCK 2
#define ACTION_AWLABLE 3

#define TYPE_MANUAL 1
#define TYPE_AUTO 2

/* This is a handy struct to pass about. */
struct dbinfo
{
	sqlite3 *db;
	sqlite3_stmt *matchactionstatement;
	sqlite3_stmt *addawlstatement;
};

bool localdebugmode;

extern "C"
{
	bool initfilterplugin(struct filterplugininfo &filterplugininfo,
		class Options &options, bool debugmode);
	void closefilterplugin(void);
	bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent);
};

bool initdb(struct dbinfo &dbinfo, std::string filename);
int matchaction(std::string localid, std::string remoteid, int action);
int addawl(std::string localid, std::string remoteid);
int dbclient(std::string commandline);
bool dbserver(struct dbinfo &dbinfo, std::string filename);
int processcommand(struct dbinfo &dbinfo, std::string command, std::vector<std::string> args, int argc);

bool initfilterplugin(struct filterplugininfo &filterplugininfo,
	class Options &options, bool debugmode)
{
	std::string filename = options["db_filter_filename"];
	
	if (filename.empty()) return false;

	localdebugmode = debugmode;
	
	filterplugininfo.pluginname = PLUGIN_NAME;
	
	struct dbinfo dbinfo;
	
	if (!initdb(dbinfo, filename)) return false;

	/* Fork off the  server process. */
	switch (fork())
	{
		/* An error occured. */
		case -1:
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Error: Fork failed: %s", strerror(errno));
			return false;
		
		/* In the child. */
		case 0:
			dbserver(dbinfo, filename);
			debugprint(localdebugmode,  PLUGIN_SHORT_NAME ": Error: We should not come here");
			exit(0);
	
		/* In the parent. */
		default:
			break;
	}		
	
	return true;
}

void closefilterplugin(void)
{
	return;
}

/* The main plugin function. See filterplugin.cpp. */
bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent)
{
	std::string localid = imevent.localid;
	std::string remoteid = imevent.remoteid;
	bool outgoing = imevent.outgoing;

	/* First match for ACCEPT. This includes things added by the AWL. */
	if (matchaction(localid, remoteid, ACTION_ACCEPT) > 0)
		return false;
	
	if (outgoing)
	{
		/* If it's an outgoing message, AND the local/remote IDs are AWLABLE, then
		 * add the AWL entry. */
		if (matchaction(localid, remoteid, ACTION_AWLABLE) > 0)
		{
			addawl(localid, remoteid);
			return false;
		}
	}

	/* See if we are blocking. */
	if (matchaction(localid, remoteid, ACTION_BLOCK) > 0)
		return true;
	
	/* Otherwise, when nothing else matches, don't block. */
	return false;
}

/* Simple stuff: inits SQLite and makes the statements etc. */
bool initdb(struct dbinfo &dbinfo, std::string filename)
{
	int rc = sqlite3_open(filename.c_str(), &dbinfo.db);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't open DB, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}

	rc = sqlite3_exec(dbinfo.db, CREATE_TABLE, NULL, NULL, NULL);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't create table, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}
		
	rc = sqlite3_prepare(dbinfo.db, MATCH_ACTION_STATEMENT, -1, &dbinfo.matchactionstatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre() MATCH_ACTION_STATEMENT, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}

	rc = sqlite3_prepare(dbinfo.db, ADD_AWL_STATEMENT, -1, &dbinfo.addawlstatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre() ADD_AWL_STATEMENT, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}
	
	return true;
}

/* Returns the number of rows for a particular localid, remoteid, and action. */
int matchaction(std::string localid, std::string remoteid, int action)
{
	return dbclient(stringprintf("MATCH_ACTION %s %s %d", localid.c_str(),
		remoteid.c_str(), action));
}

/* Adds an AWL entry. */
int addawl(std::string localid, std::string remoteid)
{
	return dbclient(stringprintf("ADD_AWL %s %s", localid.c_str(), remoteid.c_str()));
}

/* Client for the DB. Returns whatever the DB server gives it, which will always
 * be a number or -1 for an error. */
int dbclient(std::string commandline)
{
	class Socket sqlsock(AF_UNIX, SOCK_STREAM);
	
	/* Complete the connection. */
	if (!(sqlsock.connectsocket(SQLITE_SOCKET, ""))) return -1;
	
	/* Add on a CR as the server needs these for end of line. */
	std::string commandlinecr = commandline + "\n";
	
	if (!sqlsock.sendalldata(commandlinecr.c_str(), commandlinecr.length())) return -1;
	
	char buffer[BUFFER_SIZE];
	
	memset(buffer, 0, BUFFER_SIZE);
	
	if (sqlsock.recvline(buffer, BUFFER_SIZE) < 0)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get command line from SQL client");
		return -1;
	}
		
	stripnewline(buffer);
	
	sqlsock.closesocket();
	
	return (atol(buffer));
}

bool dbserver(struct dbinfo &dbinfo, std::string filename)
{
	class Socket sqlsock(AF_UNIX, SOCK_STREAM);
	
	if (!sqlsock.listensocket(SQLITE_SOCKET))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Error: Couldn't bind to SQL socket");
		return false;
	}

	/* This loop has no exit, except when the parent kills it off. */
	while (true)
	{
		std::string clientaddress;
		class Socket clientsock(AF_UNIX, SOCK_STREAM);
		char buffer[BUFFER_SIZE];
		
		if (!sqlsock.awaitconnection(clientsock, clientaddress)) continue;

		memset(buffer, 0, BUFFER_SIZE);
		if (clientsock.recvline(buffer, BUFFER_SIZE) < 0)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get command line from SQL client");
			continue;
		}
		
		stripnewline(buffer);
		
		std::string command;
		std::vector<std::string> args;
		int argc;
		
		chopline(buffer, command, args, argc);
				
		int result = processcommand(dbinfo, command, args, argc);
		std::string resultstring = stringprintf("%d\n", result);
				
		if (clientsock.sendline(resultstring.c_str(), resultstring.length()) < 0)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't send result to SQL client");
			continue;
		}

		clientsock.closesocket();
	}
	
	return true;
}

/* Carry out a command on the DB. */
int processcommand(struct dbinfo &dbinfo, std::string command, std::vector<std::string> args, int argc)
{
	if (argc < 2) return -1;
	
	std::string localid = args[0];
	std::string remoteid = args[1];
	
	int action = 0;
	int result = 0;

	/* 3rd argument is optional. */	
	if (argc >= 3)
		action = atol(args[2].c_str());
	
	sqlite3_stmt *statement = NULL;
	
	if (command == "MATCH_ACTION")
		statement = dbinfo.matchactionstatement;
	else if (command == "ADD_AWL")
		statement = dbinfo.addawlstatement;
	else
		return -1;
		
	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Command: %s localid: %s remoteid: %s action: %d",
		command.c_str(), localid.c_str(), remoteid.c_str(), action);
	
	/* localid and remoteid are common to both queries. */
	if (sqlite3_bind_text(statement, 1, localid.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind localid");
		return -1;
	}
	if (sqlite3_bind_text(statement, 2, remoteid.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind remoteid");
		return -1;
	}
	
	if (statement == dbinfo.addawlstatement)
	{
		/* This is ADD_AWL.  3 extra parementers: action, type and timestamp. */
		if (sqlite3_bind_int(statement, 3, ACTION_ACCEPT) != SQLITE_OK)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind action");
			return -1;
		}
		if (sqlite3_bind_int(statement, 4, TYPE_AUTO) != SQLITE_OK)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind type");
			return -1;
		}
		if (sqlite3_bind_int(statement, 5, (int) time(NULL)) != SQLITE_OK)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind timestamp");
			return -1;
		}
		
		while (sqlite3_step(statement) == SQLITE_ROW) result++;
	}
	else
	{
		/* And this is MATCH_ACTION. Bind to the selected action. */
		if (sqlite3_bind_int(statement, 3, action) != SQLITE_OK)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind action");
			return -1;
		}			
	
		/* And fetch out the fist column. It will be the COUNT(*). */
		if (sqlite3_step(statement) == SQLITE_ROW)
			result = sqlite3_column_int(statement, 0);
	}
	
	sqlite3_reset(statement);

	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Result: %d", result);

	return result;
}
