/*===================== begin_copyright_notice ==================================

Copyright (c) 2017 Intel Corporation

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


======================= end_copyright_notice ==================================*/
#pragma once
#include "usc.h"
namespace USC
{

/*****************************************************************************\
MACRO: BIT
\*****************************************************************************/
#ifndef BIT
#define BIT( n )    ( 1 << (n) )
#endif

/*****************************************************************************\
ENUM: SHADER_USAGE
\*****************************************************************************/
enum SHADER_USAGE
{
    SHADER_USAGE_UNKNOWN,
    SHADER_USAGE_POSITION,
    SHADER_USAGE_CLIPVERTEX,
    SHADER_USAGE_DEPTH,
    SHADER_USAGE_DEPTH_BIAS,
    SHADER_USAGE_BLENDWEIGHT,
    SHADER_USAGE_BLENDINDICES,
    SHADER_USAGE_NORMAL,
    SHADER_USAGE_POINTSIZE,
    SHADER_USAGE_TEXTURECOORDINATE,
    SHADER_USAGE_TANGENT,
    SHADER_USAGE_BINORMAL,
    SHADER_USAGE_TESSFACTOR,
    SHADER_USAGE_PRIMARYCOLOR,
    SHADER_USAGE_SECONDARYCOLOR,
    SHADER_USAGE_FOGCOORDINATE,
    SHADER_USAGE_FOGFACTOR,
    SHADER_USAGE_EDGEFLAG,
    SHADER_USAGE_BACKPRIMARYCOLOR,
    SHADER_USAGE_BACKSECONDARYCOLOR,
    SHADER_USAGE_CLIPDISTANCE,
    SHADER_USAGE_VFACE,
    SHADER_USAGE_CULLDISTANCE,
    SHADER_USAGE_RENDER_TARGET_ARRAY_INDEX,
    SHADER_USAGE_VIEWPORT_ARRAY_INDEX,
    SHADER_USAGE_VERTEX_ID,
    SHADER_USAGE_PRIMITIVE_ID,
    SHADER_USAGE_INSTANCE_ID,
    SHADER_USAGE_SAMPLEINDEX,
    SHADER_USAGE_SAMPLEPOS,
    SHADER_USAGE_SAMPLEMASK,
    SHADER_USAGE_QUAD_U0_EDGE_TESS_FACTOR,
    SHADER_USAGE_QUAD_U1_EDGE_TESS_FACTOR,
    SHADER_USAGE_QUAD_V0_EDGE_TESS_FACTOR,
    SHADER_USAGE_QUAD_V1_EDGE_TESS_FACTOR,
    SHADER_USAGE_QUAD_U_INSIDE_TESS_FACTOR,
    SHADER_USAGE_QUAD_V_INSIDE_TESS_FACTOR,
    SHADER_USAGE_TRI_U_EDGE_TESS_FACTOR,
    SHADER_USAGE_TRI_V_EDGE_TESS_FACTOR,
    SHADER_USAGE_TRI_W_EDGE_TESS_FACTOR,
    SHADER_USAGE_TRI_INSIDE_TESS_FACTOR,
    SHADER_USAGE_LINE_DENSITY_TESS_FACTOR,
    SHADER_USAGE_LINE_DETAIL_TESS_FACTOR,
    SHADER_USAGE_THREAD_ID,
    SHADER_USAGE_THREAD_GROUPS_COUNT, // OpenGL 4.3 gl_NumWorkGroups GLSL input
    SHADER_USAGE_THREAD_GROUP_ID,
    SHADER_USAGE_THREAD_ID_IN_GROUP,
    SHADER_USAGE_THREAD_ID_IN_GROUP_FLATTENED,
    SHADER_USAGE_DEPTH_GREATER_EQUAL,
    SHADER_USAGE_DEPTH_LESS_EQUAL,
    SHADER_USAGE_OUTPUT_STENCIL,
    SHADER_USAGE_HELPER_INVOCATION,
    SHADER_USAGE_REQUESTED_COARSE_FRAGMENT_SIZE, // gl_RequestedCoarseFragmentSize
    SHADER_USAGE_ACTUAL_COARSE_FRAGMENT_SIZE, // gl_ActualCoarseFragmentSize, gl_ActualCoarseFragmentCount
    SHADER_USAGE_COARSE_FRAGMENT_MASK, // gl_FragmentMaskIn[]
    SHADER_USAGE_BASEVERTEXARB,         // gl_BaseVertexARB
    SHADER_USAGE_BASEINSTANCEARB,       // gl_BaseInstanceARB
    SHADER_USAGE_DRAWIDARB,             // gl_DrawIDARB
    SHADER_USAGE_INNER_TESSFACTOR_IN,   // gl_TessLevelInner
    SHADER_USAGE_OUTER_TESSFACTOR_IN,   // gl_TessLevelOuter
    NUM_SHADER_USAGES
};

/*****************************************************************************\
ENUM: SHADER_INTERPOLATION_MODE
\*****************************************************************************/
enum SHADER_INTERPOLATION_MODE
{
    SHADER_INTERPOLATION_UNKNOWN,
    SHADER_INTERPOLATION_DEFAULT,
    SHADER_INTERPOLATION_CONSTANT,
    SHADER_INTERPOLATION_LINEAR,
    SHADER_INTERPOLATION_LINEARCENTROID,
    SHADER_INTERPOLATION_LINEARNOPERSPECTIVE,
    SHADER_INTERPOLATION_LINEARNOPERSPECTIVECENTROID,
    SHADER_INTERPOLATION_LINEARSAMPLE,
    SHADER_INTERPOLATION_LINEARNOPERSPECTIVESAMPLE,
    NUM_SHADER_INTERPOLATION_MODES
};

/*****************************************************************************\
ENUM: SHADER_RESOURCE_TYPE
\*****************************************************************************/
enum SHADER_RESOURCE_TYPE
{
    SHADER_RESOURCE_INVALID,
    SHADER_RESOURCE_BUFFER,
    SHADER_RESOURCE_1D,
    SHADER_RESOURCE_1D_ARRAY,
    SHADER_RESOURCE_2D,
    SHADER_RESOURCE_2D_ARRAY,
    SHADER_RESOURCE_3D,
    SHADER_RESOURCE_CUBE,
    SHADER_RESOURCE_CUBE_ARRAY,
    SHADER_RESOURCE_2D_MEDIA,
    SHADER_RESOURCE_2D_MEDIA_BLOCK,
    NUM_SHADER_RESOURCE_TYPES
};

/*****************************************************************************\
ENUM: SHADER_RESOURCE_RETURN_TYPE
\*****************************************************************************/
enum SHADER_RESOURCE_RETURN_TYPE
{
    SHADER_RESOURCE_RETURN_UNKNOWN = 0,
    SHADER_RESOURCE_RETURN_FLOAT,
    SHADER_RESOURCE_RETURN_UNORM,
    SHADER_RESOURCE_RETURN_SNORM,
    SHADER_RESOURCE_RETURN_UINT,
    SHADER_RESOURCE_RETURN_SINT,
    SHADER_RESOURCE_RETURN_RAW,
    SHADER_RESOURCE_RETURN_STRUCTURED,
    NUM_SHADER_RESOURCE_RETURN_TYPES
};

/*****************************************************************************\
ENUM: SHADER_SAMPLER_TYPE
\*****************************************************************************/
enum SHADER_SAMPLER_TYPE
{
    SHADER_SAMPLER_TYPE_TEXTURE,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_2DCONVOLVE,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_MINMAXFILTER,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_DILATE,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_ERODE,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_MOTION_ESTIMATION,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_MINMAX,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_CENTROID,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_BOOL_CENTROID,
    SHADER_SAMPLER_TYPE_VIDEO_ANALYTICS_FUNCTION_BOOL_SUM,
    SHADER_SAMPLER_TYPE_TEXTURE_EXTERNAL_2D_Y2Y_EXT,
    NUM_SHADER_SAMPLER_TYPES
};

/*****************************************************************************\
ENUM: SHADER_UAV_ACCESS_MODE
\*****************************************************************************/
enum SHADER_UAV_ACCESS_MODE
{
    SHADER_UAV_ACCESS_MODE_UNKNOWN = 0,
    SHADER_UAV_ACCESS_MODE_RW,
    SHADER_UAV_ACCESS_MODE_RO,
    SHADER_UAV_ACCESS_MODE_WO,
    NUM_SHADER_UAV_ACCESS_MODES
};

/*****************************************************************************\
ENUM: SHADER_OPCODE
\*****************************************************************************/
enum SHADER_OPCODE
{
    SHADER_OPCODE_NOP,

    // Move Instructions
    SHADER_OPCODE_MOV,
    SHADER_OPCODE_IMOV,
    SHADER_OPCODE_UMOV,
    SHADER_OPCODE_FMOV,
    SHADER_OPCODE_MOV_FCMP,
    SHADER_OPCODE_MOV_ICMP,
    SHADER_OPCODE_MOV_UCMP,
    SHADER_OPCODE_MOV_CND,
    SHADER_OPCODE_MOV_SWZ,
    SHADER_OPCODE_DMOV,
    SHADER_OPCODE_DMOVC,
    SHADER_OPCODE_SWAPC,

    // Floating-Point Arithmetic Instructions
    SHADER_OPCODE_FADD,
    SHADER_OPCODE_FACOS,
    SHADER_OPCODE_FASIN,
    SHADER_OPCODE_FATAN,
    SHADER_OPCODE_FCLAMP,
    SHADER_OPCODE_FCOS,
    SHADER_OPCODE_FCRS,
    SHADER_OPCODE_FDIV,
    SHADER_OPCODE_FDP2,
    SHADER_OPCODE_FDP3,
    SHADER_OPCODE_FDP4,
    SHADER_OPCODE_FDPH,
    SHADER_OPCODE_FEXP2,
    SHADER_OPCODE_FEXPE,
    SHADER_OPCODE_FFRC,
    SHADER_OPCODE_FLERP,
    SHADER_OPCODE_FLOG2,
    SHADER_OPCODE_FLOGE,
    SHADER_OPCODE_FM3X2,
    SHADER_OPCODE_FM3X3,
    SHADER_OPCODE_FM3X4,
    SHADER_OPCODE_FM4X3,
    SHADER_OPCODE_FM4X4,
    SHADER_OPCODE_FMAD,
    SHADER_OPCODE_FMAX,
    SHADER_OPCODE_FMIN,
    SHADER_OPCODE_FMUL,
    SHADER_OPCODE_FNRM2,
    SHADER_OPCODE_FNRM3,
    SHADER_OPCODE_FNRM4,
    SHADER_OPCODE_FPOW,
    SHADER_OPCODE_FRCP,
    SHADER_OPCODE_FRND_NE,
    SHADER_OPCODE_FRND_NI,
    SHADER_OPCODE_FRND_PI,
    SHADER_OPCODE_FRND_Z,
    SHADER_OPCODE_FRSQ,
    SHADER_OPCODE_FSGN,
    SHADER_OPCODE_FSIN,
    SHADER_OPCODE_FSINCOS,
    SHADER_OPCODE_FSINCOS_VECTOR,
    SHADER_OPCODE_FSQRT,
    SHADER_OPCODE_FTAN,
    SHADER_OPCODE_GRADIENT_X,
    SHADER_OPCODE_GRADIENT_Y,

    // Floating-Point Special Opcode for correctly rounded division and sqrt
    SHADER_OPCODE_FDIV_IEEE,
    SHADER_OPCODE_FSQRT_IEEE,

    // Double-Precision Floating Point Arithmetic Instructions
    SHADER_OPCODE_DADD,
    SHADER_OPCODE_DMUL,
    SHADER_OPCODE_DMAD,
    SHADER_OPCODE_DDIV,
    SHADER_OPCODE_DRCP,
    SHADER_OPCODE_DMIN,
    SHADER_OPCODE_DMAX,
    SHADER_OPCODE_DCLAMP,    // dbl-precision clamping of a number to the range
    SHADER_OPCODE_DFRC,      // dbl-precision fractional part of a number
    SHADER_OPCODE_DSQRT,     // dbl-precision square root function
    SHADER_OPCODE_DRSQ,      // dbl-precision inverse of the square root
    SHADER_OPCODE_DRND_NE,   // dbl-precision rounding to nearest even
    SHADER_OPCODE_DRND_NI,   // dbl-precision rounding towards -inf
    SHADER_OPCODE_DRND_PI,   // dbl-precision rounding towards +inf
    SHADER_OPCODE_DRND_Z,    // dbl-precision rounding towards zero
    SHADER_OPCODE_DFREXP,    // extract exponent and significand from double
    SHADER_OPCODE_DLDEXP,    // combine exponent and significand into a double
    SHADER_OPCODE_DSGN,      // dbl-precision signum function


    // Single-Precision Arithmetic Conversion Instructions
    SHADER_OPCODE_F32TO16,
    SHADER_OPCODE_F32TO16_NE,
    SHADER_OPCODE_F32TO16_PI,
    SHADER_OPCODE_F32TO16_NI,

    SHADER_OPCODE_FMOV_Z,
    SHADER_OPCODE_FMOV_NE,
    SHADER_OPCODE_FMOV_PI,
    SHADER_OPCODE_FMOV_NI,

    SHADER_OPCODE_F16TO32,

    // Integer Arithmetic Instructions
    SHADER_OPCODE_IADD,
    SHADER_OPCODE_IDIV,
    SHADER_OPCODE_IMAD,
    SHADER_OPCODE_IMAX,
    SHADER_OPCODE_IMIN,
    SHADER_OPCODE_ICLAMP,
    SHADER_OPCODE_IMUL,
    SHADER_OPCODE_INEG,
    SHADER_OPCODE_UADD,
    SHADER_OPCODE_USUB,
    SHADER_OPCODE_UDIV,
    SHADER_OPCODE_UMAD,
    SHADER_OPCODE_UMAX,
    SHADER_OPCODE_UMIN,
    SHADER_OPCODE_UCLAMP,
    SHADER_OPCODE_UMUL,
    SHADER_OPCODE_UADDC,
    SHADER_OPCODE_USUBB,

    // Bitwise Instructions
    SHADER_OPCODE_FIRSTBIT_HI,
    SHADER_OPCODE_FIRSTBIT_LO,
    SHADER_OPCODE_FIRSTBIT_SHI,
    SHADER_OPCODE_UCLZ,
    SHADER_OPCODE_USCLZ,
    SHADER_OPCODE_UBCLZ,
    SHADER_OPCODE_UCTZ,
    SHADER_OPCODE_USCTZ,
    SHADER_OPCODE_UBCTZ,
    SHADER_OPCODE_ISHL,
    SHADER_OPCODE_ISHR,
    SHADER_OPCODE_UBFE,
    SHADER_OPCODE_IBFE,
    SHADER_OPCODE_UBCOUNTBITS,
    SHADER_OPCODE_USCOUNTBITS,
    SHADER_OPCODE_COUNTBITS,
    SHADER_OPCODE_BFREV,
    SHADER_OPCODE_BFI,

    // Byte Arithmetic Instructions
    SHADER_OPCODE_BADD,
    SHADER_OPCODE_BDIV,
    SHADER_OPCODE_BMAD,
    SHADER_OPCODE_BMAX,
    SHADER_OPCODE_BMIN,
    SHADER_OPCODE_BMUL,
    SHADER_OPCODE_BNEG,
    SHADER_OPCODE_UBADD,
    SHADER_OPCODE_UBSUB,
    SHADER_OPCODE_UBDIV,
    SHADER_OPCODE_UBMAD,
    SHADER_OPCODE_UBMAX,
    SHADER_OPCODE_UBMIN,
    SHADER_OPCODE_UBMUL,

    // Short Arithmetic Instructions
    SHADER_OPCODE_SADD,
    SHADER_OPCODE_SDIV,
    SHADER_OPCODE_SMAD,
    SHADER_OPCODE_SMAX,
    SHADER_OPCODE_SMIN,
    SHADER_OPCODE_SMUL,
    SHADER_OPCODE_SNEG,
    SHADER_OPCODE_USADD,
    SHADER_OPCODE_USSUB,
    SHADER_OPCODE_USDIV,
    SHADER_OPCODE_USMAD,
    SHADER_OPCODE_USMAX,
    SHADER_OPCODE_USMIN,
    SHADER_OPCODE_USMUL,

    // Type Conversion Instructions
    SHADER_OPCODE_FTOI,
    SHADER_OPCODE_FTOU,
    SHADER_OPCODE_FTOS,
    SHADER_OPCODE_FTOUS,
    SHADER_OPCODE_FTOB,
    SHADER_OPCODE_FTOUB,

    SHADER_OPCODE_ITOF,
    SHADER_OPCODE_ITOF_NI,
    SHADER_OPCODE_ITOF_PI,
    SHADER_OPCODE_ITOF_Z,
    SHADER_OPCODE_ITOU,
    SHADER_OPCODE_ITOS,
    SHADER_OPCODE_ITOUS,
    SHADER_OPCODE_ITOB,
    SHADER_OPCODE_ITOUB,

    SHADER_OPCODE_UTOF,
    SHADER_OPCODE_UTOF_NI,
    SHADER_OPCODE_UTOF_PI,
    SHADER_OPCODE_UTOF_Z,
    SHADER_OPCODE_UTOI,
    SHADER_OPCODE_UTOS,
    SHADER_OPCODE_UTOUS,
    SHADER_OPCODE_UTOB,
    SHADER_OPCODE_UTOUB,

    SHADER_OPCODE_STOF,
    SHADER_OPCODE_STOI,
    SHADER_OPCODE_STOU,
    SHADER_OPCODE_STOUS,
    SHADER_OPCODE_STOB,
    SHADER_OPCODE_STOUB,

    SHADER_OPCODE_USTOF,
    SHADER_OPCODE_USTOI,
    SHADER_OPCODE_USTOU,
    SHADER_OPCODE_USTOS,
    SHADER_OPCODE_USTOB,
    SHADER_OPCODE_USTOUB,

    SHADER_OPCODE_BTOF,
    SHADER_OPCODE_BTOI,
    SHADER_OPCODE_BTOU,
    SHADER_OPCODE_BTOS,
    SHADER_OPCODE_BTOUS,
    SHADER_OPCODE_BTOUB,

    SHADER_OPCODE_UBTOF,
    SHADER_OPCODE_UBTOI,
    SHADER_OPCODE_UBTOU,
    SHADER_OPCODE_UBTOS,
    SHADER_OPCODE_UBTOUS,
    SHADER_OPCODE_UBTOB,

    SHADER_OPCODE_DTOF,
    SHADER_OPCODE_DTOU,
    SHADER_OPCODE_DTOI,
    SHADER_OPCODE_DTOS,
    SHADER_OPCODE_DTOUS,
    SHADER_OPCODE_FTOD,
    SHADER_OPCODE_UTOD,
    SHADER_OPCODE_ITOD,

    SHADER_OPCODE_USTOD,
    SHADER_OPCODE_STOD,

    SHADER_OPCODE_DTOB,
    SHADER_OPCODE_DTOUB,
    SHADER_OPCODE_BTOD,
    SHADER_OPCODE_UBTOD,

    SHADER_OPCODE_IBITCAST,
    SHADER_OPCODE_UBITCAST,
    SHADER_OPCODE_FBITCAST,

    // Bitwise Instructions
    SHADER_OPCODE_AND,
    SHADER_OPCODE_NOT,
    SHADER_OPCODE_OR,
    SHADER_OPCODE_SAR,
    SHADER_OPCODE_SHL,
    SHADER_OPCODE_SHR,
    SHADER_OPCODE_XOR,

    // Conditional Computing Instructions
    SHADER_OPCODE_FCMP,
    SHADER_OPCODE_ICMP,
    SHADER_OPCODE_SCMP,
    SHADER_OPCODE_BCMP,
    SHADER_OPCODE_UCMP,
    SHADER_OPCODE_USCMP,
    SHADER_OPCODE_UBCMP,
    SHADER_OPCODE_DCMP,

    // Control-Flow Instructions
    SHADER_OPCODE_IF_FCMP,
    SHADER_OPCODE_IF_ICMP,
    SHADER_OPCODE_IF_UCMP,
    SHADER_OPCODE_IF_CND,
    SHADER_OPCODE_ELSE,
    SHADER_OPCODE_ENDIF,
    SHADER_OPCODE_GOTO,
    SHADER_OPCODE_GOTO_CND,
    SHADER_OPCODE_JOIN,
    SHADER_OPCODE_LOOP,
    SHADER_OPCODE_ENDLOOP,
    SHADER_OPCODE_CONTINUE,
    SHADER_OPCODE_CONTINUE_CND,
    SHADER_OPCODE_BREAK,
    SHADER_OPCODE_BREAK_CND,
    SHADER_OPCODE_BREAK_FCMP,
    SHADER_OPCODE_CALL,
    SHADER_OPCODE_CALL_CND,
    SHADER_OPCODE_FUNC,
    SHADER_OPCODE_RET,
    SHADER_OPCODE_RET_CND,
    SHADER_OPCODE_SWITCH,
    SHADER_OPCODE_CASE,
    SHADER_OPCODE_DEFAULT,
    SHADER_OPCODE_ENDSWITCH,
    SHADER_OPCODE_WHILE_CND,
    SHADER_OPCODE_ENDWHILE,
    SHADER_OPCODE_REPEAT,
    SHADER_OPCODE_UNTIL_CND,
    SHADER_OPCODE_FOR,
    SHADER_OPCODE_ENDFOR,
    SHADER_OPCODE_INTERFACE_CALL,
    SHADER_OPCODE_FUNC_BODY,
    SHADER_OPCODE_FUNC_BODY_RET,
    SHADER_OPCODE_FUNC_BODY_RET_CND,
    SHADER_OPCODE_SUBROUTINE_BODY,
    SHADER_OPCODE_SUBROUTINE_CALL,
    SHADER_OPCODE_SUBROUTINE_RET,

    // Resource Access Instructions
    // don't change the order, make sure the order is matched to
    // GEN6 sampler CSIL instructions order in kpo_inst_g4_cs.h
    SHADER_OPCODE_LOAD,
    SHADER_OPCODE_SAMPLE,
    SHADER_OPCODE_SAMPLE_CL,
    SHADER_OPCODE_SAMPLE_B,
    SHADER_OPCODE_SAMPLE_B_CL,
    SHADER_OPCODE_SAMPLE_G,
    SHADER_OPCODE_SAMPLE_G_CL,
    SHADER_OPCODE_SAMPLE_L,
    SHADER_OPCODE_SAMPLE_P,
    SHADER_OPCODE_SAMPLE_C,
    SHADER_OPCODE_SAMPLE_C_CL,
    SHADER_OPCODE_SAMPLE_C_LZ,
    SHADER_OPCODE_SAMPLE_BC,
    SHADER_OPCODE_SAMPLE_LC,
    SHADER_OPCODE_SAMPLE_PC,
    SHADER_OPCODE_SAMPLE_PB,
    SHADER_OPCODE_SAMPLE_PL,
    SHADER_OPCODE_SAMPLE_PBC,
    SHADER_OPCODE_SAMPLE_PLC,
    SHADER_OPCODE_SAMPLE_DISCARD,
    SHADER_OPCODE_RESINFO_FLOAT,
    SHADER_OPCODE_RESINFO_UINT,
    SHADER_OPCODE_RESINFO_RCPFLOAT,
    SHADER_OPCODE_SAMPLEINFO_FLOAT,
    SHADER_OPCODE_SAMPLEINFO_UINT,
    SHADER_OPCODE_SAMPLEPOS,
    SHADER_OPCODE_LOAD2DMS,
    SHADER_OPCODE_LOAD2DMS_LOD,
    SHADER_OPCODE_GATHER4,
    SHADER_OPCODE_LOD,

    // New Gen7 sample instruction opcodes
    SHADER_OPCODE_BUFINFO,
    SHADER_OPCODE_SAMPLE_G_C,

    // New Gen7 deriv instructions
    SHADER_OPCODE_GRADIENT_X_COARSE,
    SHADER_OPCODE_GRADIENT_X_FINE,
    SHADER_OPCODE_GRADIENT_Y_COARSE,
    SHADER_OPCODE_GRADIENT_Y_FINE,

    // VME instructions
    SHADER_OPCODE_VME_MB_QUERY,
    SHADER_OPCODE_VME_MB_MULTI_QUERY_8,
    SHADER_OPCODE_VME_MB_MULTI_QUERY_4,
    SHADER_OPCODE_VME_MB_MULTI_CHECK_16x16,
    SHADER_OPCODE_VME_MB_MULTI_BIDIR_CHECK_16x16,
    SHADER_OPCODE_VME_MB_MULTI_CHECK_8x8,
    SHADER_OPCODE_VME_MB_MULTI_BIDIR_CHECK_8x8,

    SHADER_OPCODE_MCE_SET_MV_COST_FUNCTION,
    SHADER_OPCODE_MCE_SET_INTRA_LUMA_MODE_COST_FUNCTION,
    SHADER_OPCODE_MCE_SET_INTRA_CHROMA_MODE_COST_FUNCTION,
    SHADER_OPCODE_MCE_SET_INTER_MULTI_REF_PENALTY,
    SHADER_OPCODE_MCE_SET_INTER_SHAPE_PENALTY,
    SHADER_OPCODE_MCE_SET_INTRA_LUMA_SHAPE_PENALTY,
    SHADER_OPCODE_MCE_SET_INTER_DIRECTION_PENALTY,
    SHADER_OPCODE_MCE_SET_AC_ONLY_HAAR,
    SHADER_OPCODE_MCE_SET_SRC_INTERLACED_FIELD_POLARITY,
    SHADER_OPCODE_MCE_SET_REF_PIXEL_BIAS,
    SHADER_OPCODE_MCE_SET_SINGLE_REF_ID,
    SHADER_OPCODE_MCE_SET_DUAL_REF_ID,
    SHADER_OPCODE_MCE_SET_REF_ID_RAW,
    SHADER_OPCODE_MCE_SET_REF_ID_POLARITIES_RAW,
    SHADER_OPCODE_MCE_SET_BILINEAR_FILTER_ENABLE,
    SHADER_OPCODE_MCE_GET_MOTION_VECTORS,
    SHADER_OPCODE_MCE_GET_INTER_DISTORTIONS,
    SHADER_OPCODE_MCE_GET_BEST_INTER_DISTORTION,
    SHADER_OPCODE_MCE_GET_INTER_MAJOR_SHAPE,
    SHADER_OPCODE_MCE_GET_INTER_MINOR_SHAPES,
    SHADER_OPCODE_MCE_GET_INTER_DIRECTIONS,
    SHADER_OPCODE_MCE_GET_INTER_REFERENCE_IDS,
    SHADER_OPCODE_MCE_GET_INTER_MOTION_VECTOR_COUNT,
    SHADER_OPCODE_MCE_GET_8x8_TRANSFORM_RECOMMENDATION,

    SHADER_OPCODE_IME_INITIALIZE,
    SHADER_OPCODE_IME_SET_SINGLE_REF,
    SHADER_OPCODE_IME_SET_MULTI_REF,
    SHADER_OPCODE_IME_SET_MAX_MOTION_VECTOR_COUNT,
    SHADER_OPCODE_IME_EVALUATE_WITH_SINGLE_REF,
    SHADER_OPCODE_IME_EVALUATE_WITH_MULTI_REF,
    SHADER_OPCODE_IME_EVALUATE_WITH_SINGLE_REF_SOUT,
    SHADER_OPCODE_IME_EVALUATE_WITH_MULTI_REF_SOUT,
    SHADER_OPCODE_IME_EVALUATE_WITH_SINGLE_REF_SIN,
    SHADER_OPCODE_IME_EVALUATE_WITH_MULTI_REF_SIN,
    SHADER_OPCODE_IME_EVALUATE_WITH_SINGLE_REF_SINOUT,
    SHADER_OPCODE_IME_EVALUATE_WITH_MULTI_REF_SINOUT,
    SHADER_OPCODE_IME_GET_SINGLE_REF_SIN,
    SHADER_OPCODE_IME_GET_MULTI_REF_SIN,
    SHADER_OPCODE_IME_GET_SOUT_MOTION_VECTORS,
    SHADER_OPCODE_IME_GET_SOUT_DISTORTIONS,
    SHADER_OPCODE_IME_GET_SOUT_REFERENCE_IDS,
    SHADER_OPCODE_IME_SET_UNIDIR_MIX_DISABLE,
    SHADER_OPCODE_IME_SET_WEIGHTED_SAD,
    SHADER_OPCODE_IME_GET_BORDER_REACHED,
    SHADER_OPCODE_IME_GET_WEIGTHING_PATTERN_MIN_MOTION_VECTOR,
    SHADER_OPCODE_IME_GET_WEIGTHING_PATTERN_MIN_DISTORTION,
    SHADER_OPCODE_IME_GET_EARLY_TERMINATION,
    SHADER_OPCODE_IME_SET_EARLY_TERMINATION_THRESHOLD,
    SHADER_OPCODE_IME_GET_TRUNCATED_SEARCH_INDICATION,

    SHADER_OPCODE_FME_INITIALIZE,
    SHADER_OPCODE_BME_INITIALIZE,
    SHADER_OPCODE_REF_SET_BIDIR_MIX_DISABLE,
    SHADER_OPCODE_REF_EVALUATE_WITH_SINGLE_REF,
    SHADER_OPCODE_REF_EVALUATE_WITH_MULTI_REF,
    SHADER_OPCODE_REF_EVALUATE_WITH_MULTI_REF_ID,

    SHADER_OPCODE_SIC_INITIALIZE,
    SHADER_OPCODE_SIC_CONFIGURE_IPE,
    SHADER_OPCODE_SIC_CONFIGURE_IPE_WITH_CHROMA,
    SHADER_OPCODE_SIC_CONFIGURE_SKC,
    SHADER_OPCODE_SIC_SET_SKC_FWD_TRANSFORM_ENABLE,
    SHADER_OPCODE_SIC_SET_SKC_BLOCK_BASED_RAW_SKIP_SAD,
    SHADER_OPCODE_SIC_EVALUATE_IPE,
    SHADER_OPCODE_SIC_EVALUATE_WITH_SINGLE_REF,
    SHADER_OPCODE_SIC_EVALUATE_WITH_MULTI_REF,
    SHADER_OPCODE_SIC_EVALUATE_WITH_MULTI_REF_ID,
    SHADER_OPCODE_SIC_GET_IPE_LUMA_SHAPE,
    SHADER_OPCODE_SIC_GET_BEST_IPE_LUMA_DISTORTION,
    SHADER_OPCODE_SIC_GET_BEST_IPE_CHROMA_DISTORTION,
    SHADER_OPCODE_SIC_GET_PACKED_IPE_LUMA_MODES,
    SHADER_OPCODE_SIC_GET_IPE_CHROMA_MODE,
    SHADER_OPCODE_SIC_GET_INTER_RAW_SAD,
    SHADER_OPCODE_SIC_GET_PACKED_SKC_LUMA_COUNT_THRESHOLD,
    SHADER_OPCODE_SIC_GET_PACKED_SKC_LUMA_SUM_THRESHOLD,

    // Raster Instructions
    SHADER_OPCODE_DISCARD_FCMP,
    SHADER_OPCODE_DISCARD_ICMP,
    SHADER_OPCODE_DISCARD_UCMP,
    SHADER_OPCODE_DISCARD_CND,
    SHADER_OPCODE_TEXKILL,

    // Topology Instructions
    SHADER_OPCODE_CUT,
    SHADER_OPCODE_CUT_STREAM,
    SHADER_OPCODE_EMIT,
    SHADER_OPCODE_EMIT_STREAM,
    SHADER_OPCODE_EMIT_THEN_CUT,
    SHADER_OPCODE_EMIT_THEN_CUT_STREAM,

    // DX11 Instructions
    SHADER_OPCODE_LD_UAV_TYPED,
    SHADER_OPCODE_STORE_UAV_TYPED,
    SHADER_OPCODE_LD_RAW,
    SHADER_OPCODE_LD_RAW_SCRATCH,
    SHADER_OPCODE_LD_STRUCTURED,
    SHADER_OPCODE_STORE_RAW,
    SHADER_OPCODE_STORE_RAW_SCRATCH,
    SHADER_OPCODE_STORE_STRUCTURED,
    SHADER_OPCODE_ATOMIC_AND,
    SHADER_OPCODE_ATOMIC_OR,
    SHADER_OPCODE_ATOMIC_XOR,
    SHADER_OPCODE_ATOMIC_CMP_STORE,
    SHADER_OPCODE_ATOMIC_IADD,
    SHADER_OPCODE_ATOMIC_IMAX,
    SHADER_OPCODE_ATOMIC_IMIN,
    SHADER_OPCODE_ATOMIC_UMAX,
    SHADER_OPCODE_ATOMIC_UMIN,
    SHADER_OPCODE_IMM_ATOMIC_ALLOC,
    SHADER_OPCODE_IMM_ATOMIC_CONSUME,
    SHADER_OPCODE_IMM_ATOMIC_IADD,
    SHADER_OPCODE_IMM_ATOMIC_PREDEC,
    SHADER_OPCODE_IMM_ATOMIC_AND,
    SHADER_OPCODE_IMM_ATOMIC_OR,
    SHADER_OPCODE_IMM_ATOMIC_XOR,
    SHADER_OPCODE_IMM_ATOMIC_EXCH,
    SHADER_OPCODE_IMM_ATOMIC_CMP_EXCH,
    SHADER_OPCODE_IMM_ATOMIC_IMAX,
    SHADER_OPCODE_IMM_ATOMIC_IMIN,
    SHADER_OPCODE_IMM_ATOMIC_UMAX,
    SHADER_OPCODE_IMM_ATOMIC_UMIN,
    SHADER_OPCODE_GATHER4_C,
    SHADER_OPCODE_GATHER4_PO,
    SHADER_OPCODE_GATHER4_PO_C,
    SHADER_OPCODE_SAMPLE2DMS,

    // OCL 2.0 Instruction for read_write images
    SHADER_OPCODE_LD_UAV_TYPED_SAMPLER,

    // DX11 Eval Instructions
    SHADER_OPCODE_EVAL_SNAPPED,
    SHADER_OPCODE_EVAL_SAMPLEINDEX,
    SHADER_OPCODE_EVAL_CENTROID,

    // OGL4.2 MSAA LD/STORE UAV Instructions
    SHADER_OPCODE_RESINFO_UINT_MSAA,
    SHADER_OPCODE_LD_UAV_TYPED_MSAA,
    SHADER_OPCODE_STORE_UAV_TYPED_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_AND_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_OR_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_XOR_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_CMP_EXCH_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_IADD_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_IMAX_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_IMIN_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_UMAX_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_UMIN_MSAA,
    SHADER_OPCODE_IMM_ATOMIC_EXCH_MSAA,

    // Intel Vendor Extension instructions
    SHADER_OPCODE_SPAN_OFFSET,
    SHADER_OPCODE_SPAN_SWIZZLE,
    SHADER_OPCODE_LANE_SHIFT,
    SHADER_OPCODE_BARYCENTRIC_COORD,
    SHADER_OPCODE_VERTEX_DATA,

    // Masked Sum of Absolute Differences
    SHADER_OPCODE_MSAD,

    // Sync Instruction
    SHADER_OPCODE_SYNC,
    SHADER_OPCODE_SYNC_COUNT,

    // Pointer Instructions
    SHADER_OPCODE_ADDRESS_OF,
    SHADER_OPCODE_NULLCMP,
    SHADER_OPCODE_NULLMOV,
    SHADER_OPCODE_PDIFF,
    SHADER_OPCODE_PADD,
    SHADER_OPCODE_PCMP,
    SHADER_OPCODE_BSTORE_RAW_PTR,
    SHADER_OPCODE_BSTORE_RAW_PTR_SIB,
    SHADER_OPCODE_SSTORE_RAW_PTR,
    SHADER_OPCODE_SSTORE_RAW_PTR_SIB,
    SHADER_OPCODE_STORE_RAW_PTR,
    SHADER_OPCODE_STORE_RAW_PTR_SIB,
    SHADER_OPCODE_BLD_RAW_PTR,
    SHADER_OPCODE_BLD_RAW_PTR_SIB,
    SHADER_OPCODE_SLD_RAW_PTR,
    SHADER_OPCODE_SLD_RAW_PTR_SIB,
    SHADER_OPCODE_LD_RAW_PTR,
    SHADER_OPCODE_LD_RAW_PTR_SIB,

    // Packing and unpacking instructions
    SHADER_OPCODE_BPACK,
    SHADER_OPCODE_SPACK,
    SHADER_OPCODE_BUNPACK,
    SHADER_OPCODE_SUNPACK,

    // 64-bit Integer Instructions
    SHADER_OPCODE_LADD,
    SHADER_OPCODE_LCMP,
    SHADER_OPCODE_ULSUB,
    SHADER_OPCODE_ULCMP,

    SHADER_OPCODE_LMOV,
    SHADER_OPCODE_LMOVC,

    // 64-bit Integer Arithmetic Instructions
    SHADER_OPCODE_LMAX,
    SHADER_OPCODE_LMIN,
    SHADER_OPCODE_ULADD,
    SHADER_OPCODE_ULMAX,
    SHADER_OPCODE_ULMIN,

    // 64-bit Bitwise Instructions
    SHADER_OPCODE_LAND,
    SHADER_OPCODE_LNOT,
    SHADER_OPCODE_LOR,
    SHADER_OPCODE_LSAR,
    SHADER_OPCODE_LSHL,
    SHADER_OPCODE_LSHR,
    SHADER_OPCODE_LXOR,

    // 64-bit Type Conversion Instructions
    SHADER_OPCODE_ITOL,
    SHADER_OPCODE_UTOL,
    SHADER_OPCODE_STOL,
    SHADER_OPCODE_USTOL,
    SHADER_OPCODE_BTOL,
    SHADER_OPCODE_UBTOL,
    SHADER_OPCODE_ITOUL,
    SHADER_OPCODE_UTOUL,
    SHADER_OPCODE_STOUL,
    SHADER_OPCODE_USTOUL,
    SHADER_OPCODE_BTOUL,
    SHADER_OPCODE_UBTOUL,

    SHADER_OPCODE_LTOI,
    SHADER_OPCODE_LTOU,
    SHADER_OPCODE_LTOS,
    SHADER_OPCODE_LTOUS,
    SHADER_OPCODE_LTOB,
    SHADER_OPCODE_LTOUB,
    SHADER_OPCODE_ULTOI,
    SHADER_OPCODE_ULTOU,
    SHADER_OPCODE_ULTOS,
    SHADER_OPCODE_ULTOUS,
    SHADER_OPCODE_ULTOB,
    SHADER_OPCODE_ULTOUB,

    SHADER_OPCODE_ULTOL,
    SHADER_OPCODE_LTOUL,

    SHADER_OPCODE_FTOL,
    SHADER_OPCODE_FTOUL,
    SHADER_OPCODE_LTOF,
    SHADER_OPCODE_ULTOF,
    SHADER_OPCODE_DTOL,
    SHADER_OPCODE_DTOUL,
    SHADER_OPCODE_LTOD,
    SHADER_OPCODE_ULTOD,

    // Pointer Immediate Atomic Instructions
    SHADER_OPCODE_PTR_IMM_ATOMIC_IADD,
    SHADER_OPCODE_PTR_IMM_ATOMIC_XCHG,
    SHADER_OPCODE_PTR_IMM_ATOMIC_CMP_XCHG,
    SHADER_OPCODE_PTR_IMM_ATOMIC_IMIN,
    SHADER_OPCODE_PTR_IMM_ATOMIC_UMIN,
    SHADER_OPCODE_PTR_IMM_ATOMIC_IMAX,
    SHADER_OPCODE_PTR_IMM_ATOMIC_UMAX,
    SHADER_OPCODE_PTR_IMM_ATOMIC_AND,
    SHADER_OPCODE_PTR_IMM_ATOMIC_OR,
    SHADER_OPCODE_PTR_IMM_ATOMIC_XOR,

    //Gen 9 atomic instructions
    SHADER_OPCODE_PTR_IMM_ATOMIC_FMIN,
    SHADER_OPCODE_PTR_IMM_ATOMIC_FMAX,
    SHADER_OPCODE_PTR_IMM_ATOMIC_FCMP_XCHG,

    // Printf Instruction
    SHADER_OPCODE_PRINTF,

    // SIMD Instructions
    SHADER_OPCODE_SIMD_SIZE,
    SHADER_OPCODE_SIMD_ID,
    SHADER_OPCODE_SIMD_ANY,

    SHADER_OPCODE_SIMD_SHUFFLE,
    SHADER_OPCODE_SIMD_SHUFFLE_DOWN,
    SHADER_OPCODE_SSIMD_SHUFFLE,
    SHADER_OPCODE_SSIMD_SHUFFLE_DOWN,
    SHADER_OPCODE_BSIMD_SHUFFLE,
    SHADER_OPCODE_BSIMD_SHUFFLE_DOWN,

    SHADER_OPCODE_SIMD_MEDIA_BLOCK_READ_4,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_READ_8,
    SHADER_OPCODE_SIMD_BLOCK_READ_4,
    SHADER_OPCODE_SIMD_BLOCK_READ_8,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_WRITE_4,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_WRITE_8,
    SHADER_OPCODE_SIMD_BLOCK_WRITE_4,
    SHADER_OPCODE_SIMD_BLOCK_WRITE_8,

    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_READ_4,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_READ_8,
    SHADER_OPCODE_SSIMD_BLOCK_READ_4,
    SHADER_OPCODE_SSIMD_BLOCK_READ_8,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_WRITE_4,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_WRITE_8,
    SHADER_OPCODE_SSIMD_BLOCK_WRITE_4,
    SHADER_OPCODE_SSIMD_BLOCK_WRITE_8,

    SHADER_OPCODE_SIMD_V2HADD,
    SHADER_OPCODE_SIMD_V2HMUL,
    SHADER_OPCODE_SIMD_V2HMAD,
    SHADER_OPCODE_SIMD_V2HCMP,

    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_READ_DIM_4,
    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_READ_DIM_8,
    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_READ_DIM_16,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_READ_DIM_4,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_READ_DIM_8,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_READ_DIM_16,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_READ_DIM_4,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_READ_DIM_8,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_READ_DIM_16,

    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_WRITE_DIM_4,
    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_WRITE_DIM_8,
    SHADER_OPCODE_BSIMD_MEDIA_BLOCK_WRITE_DIM_16,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_WRITE_DIM_4,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_WRITE_DIM_8,
    SHADER_OPCODE_SSIMD_MEDIA_BLOCK_WRITE_DIM_16,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_WRITE_DIM_4,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_WRITE_DIM_8,
    SHADER_OPCODE_SIMD_MEDIA_BLOCK_WRITE_DIM_16,

    // End of Program
    SHADER_OPCODE_END,

    // Profile Instruction
    SHADER_OPCODE_PROFILE,
    SHADER_OPCODE_PROFILE_AGGREGATED,
    SHADER_OPCODE_PROFILE_PM0,
    SHADER_OPCODE_PROFILE_EXECUTION_MASK,
    SHADER_OPCODE_PROFILE_HIT_COUNTER,
    SHADER_OPCODE_ATOMIC_FMAX,
    SHADER_OPCODE_ATOMIC_FMIN,
    SHADER_OPCODE_ATOMIC_FCMP_STORE,

    // Video Analytic Instructions
    SHADER_OPCODE_VA_2DCONVOLVE_SLM,
    SHADER_OPCODE_VA_2DCONVOLVE_16x1,
    SHADER_OPCODE_VA_2DCONVOLVE_16x4,
    SHADER_OPCODE_VA_MINMAXFILTER,
    SHADER_OPCODE_VA_MINFILTER_16x1_UB,
    SHADER_OPCODE_VA_MINFILTER_16x4_UB,
    SHADER_OPCODE_VA_MAXFILTER_16x1_UB,
    SHADER_OPCODE_VA_MAXFILTER_16x4_UB,
    SHADER_OPCODE_VA_DILATE,
    SHADER_OPCODE_VA_ERODE,
    SHADER_OPCODE_VA_MINMAX,
    SHADER_OPCODE_VA_CENTROID,
    SHADER_OPCODE_VA_BOOLCENTROID,
    SHADER_OPCODE_VA_BOOLSUM,

    // GL_INTEL_fragment_shader_ordering, GL_ARB_fragment_shader_interlock
    SHADER_OPCODE_IMG_SERIALIZE_ALL,

    // DX_ - GEN9 instruction
    SHADER_OPCODE_RENDERTARGET_READ,
    SHADER_OPCODE_RENDERTARGET_READ_AT_SAMPLE,
    SHADER_OPCODE_CHECK_ACCESS_FULLY_MAPPED,

    // Address Space Qualifiers
    SHADER_OPCODE_IS_UAV,
    SHADER_OPCODE_IS_TGSM,
    SHADER_OPCODE_IS_TPM,

    // Internal / Debug Instruction
    SHADER_OPCODE_ACTIVECHANNELMASK,
    SHADER_OPCODE_MOVREG,
    SHADER_OPCODE_MOVFLAG,
    SHADER_OPCODE_MOVCR,
    SHADER_OPCODE_HW_THREAD_ID,
    SHADER_OPCODE_SLICE_ID,
    SHADER_OPCODE_SUBSLICE_ID,
    SHADER_OPCODE_EU_ID,
    SHADER_OPCODE_EU_THREAD_ID,

    // OCL - Store with LOD
    SHADER_OPCODE_STORE_UAV_TYPED_L,

    // GL_EXT_YUV_target
    SHADER_OPCODE_RGB_TO_YUV,
    SHADER_OPCODE_YUV_TO_RGB,

    NUM_SHADER_OPCODES
};

/*****************************************************************************\
ENUM: SHADER_OPCODE_TYPE
\*****************************************************************************/
enum SHADER_OPCODE_TYPE
{
    SHADER_OPCODE_TYPE_NULL,
    SHADER_OPCODE_TYPE_FLOAT,
    SHADER_OPCODE_TYPE_INT,
    SHADER_OPCODE_TYPE_UINT,
    SHADER_OPCODE_TYPE_SHORT,
    SHADER_OPCODE_TYPE_USHORT,
    SHADER_OPCODE_TYPE_BYTE,
    SHADER_OPCODE_TYPE_UBYTE,
    SHADER_OPCODE_TYPE_DOUBLE,
    SHADER_OPCODE_TYPE_LONG,
    SHADER_OPCODE_TYPE_ULONG,
    // Float if modifiers are used (negate, absolute), otherwise raw binary data.
    SHADER_OPCODE_TYPE_RAW_OR_FLOAT,
    NUM_SHADER_OPCODE_TYPES
};

/*****************************************************************************\
ENUM: SHADER_CHANNEL
\*****************************************************************************/
enum SHADER_CHANNEL
{
    SHADER_CHANNEL_X = 0,
    SHADER_CHANNEL_Y = 1,
    SHADER_CHANNEL_Z = 2,
    SHADER_CHANNEL_W = 3,
    NUM_SHADER_CHANNELS = 4,
    INVALID_CHANNEL = 4,
    BITS_FOR_SHADER_CHANNEL = 2
};

inline SHADER_CHANNEL& operator ++(SHADER_CHANNEL& rChannel)
{
    unsigned int& a = (unsigned int&)rChannel;
    ++a;
    return rChannel;
}

inline SHADER_CHANNEL operator ++(SHADER_CHANNEL& rChannel,int)
{
    SHADER_CHANNEL a = rChannel;
    ++rChannel;
    return a;
}

/*****************************************************************************\
ENUM: EXT_TEXTURE_SWIZZLE
    Extended swizzle augmented by the possibility of setting a channel
    to either zero or one.
    Used for texture swizzling (cf. OpenGL GL_EXT_texture_swizzle extension).
\*****************************************************************************/
enum EXT_TEXTURE_SWIZZLE
{
    EXT_TEXTURE_SWIZZLE_RED = 0,
    EXT_TEXTURE_SWIZZLE_GREEN,
    EXT_TEXTURE_SWIZZLE_BLUE,
    EXT_TEXTURE_SWIZZLE_ALPHA,
    EXT_TEXTURE_SWIZZLE_ZERO,
    EXT_TEXTURE_SWIZZLE_ONE,
    NUM_EXT_TEXTURE_SWIZZLES
};

/*****************************************************************************\
ENUM: SHADER_MASK
\*****************************************************************************/
enum SHADER_MASK
{
    SHADER_MASK_EMPTY = 0,
    SHADER_MASK_X       = BIT(SHADER_CHANNEL_X),
    SHADER_MASK_Y       = BIT(SHADER_CHANNEL_Y),
    SHADER_MASK_XY      = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Y),
    SHADER_MASK_Z       = BIT(SHADER_CHANNEL_Z),
    SHADER_MASK_XZ      = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Z),
    SHADER_MASK_YZ      = BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_Z),
    SHADER_MASK_XYZ     = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_Z),
    SHADER_MASK_W       = BIT(SHADER_CHANNEL_W),
    SHADER_MASK_XW      = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_YW      = BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_XYW     = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_ZW      = BIT(SHADER_CHANNEL_Z) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_XZW     = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Z) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_YZW     = BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_Z) | BIT(SHADER_CHANNEL_W),
    SHADER_MASK_XYZW    = BIT(SHADER_CHANNEL_X) | BIT(SHADER_CHANNEL_Y) | BIT(SHADER_CHANNEL_Z) | BIT(SHADER_CHANNEL_W),
    NUM_SHADER_MASKS
};

inline SHADER_MASK& operator +=(SHADER_MASK& rMask,unsigned int a)
{
    rMask = (SHADER_MASK)((unsigned int)rMask + a);
    return rMask;
}

inline SHADER_MASK& operator |=(SHADER_MASK& rMask,unsigned int a)
{
    rMask = (SHADER_MASK)((unsigned int)rMask | a);
    return rMask;
}

/*****************************************************************************\
ENUM: SHADER_COMPARISON
\*****************************************************************************/
enum SHADER_COMPARISON
{                           // < = >
    SHADER_COMPARISON_NONE, // 0 0 0
    SHADER_COMPARISON_GT,   // 0 0 1
    SHADER_COMPARISON_EQ,   // 0 1 0
    SHADER_COMPARISON_GE,   // 0 1 1
    SHADER_COMPARISON_LT,   // 1 0 0
    SHADER_COMPARISON_NE,   // 1 0 1
    SHADER_COMPARISON_LE,   // 1 1 0
    SHADER_COMPARISON_TO_PATCH,   // 1 1 1
    NUM_SHADER_COMPARISONS
};

/*****************************************************************************\
ENUM: SHADER_CONDITIONAL
\*****************************************************************************/
enum SHADER_CONDITIONAL
{
    SHADER_CONDITIONAL_NONE,
    SHADER_CONDITIONAL_Z,   // _z
    SHADER_CONDITIONAL_NZ,  // _nz
    NUM_SHADER_CONDITIONALS
};

/*****************************************************************************\
ENUM: SHADER_QTR_CTRL
\*****************************************************************************/
enum SHADER_QTR_CTRL
{
    SHADER_QTR_CTRL_1Q = 0,
    SHADER_QTR_CTRL_2Q = 1,
    SHADER_QTR_CTRL_3Q = 2,
    SHADER_QTR_CTRL_4Q = 3,
    SHADER_QTR_CTRL_1H = 0,
    SHADER_QTR_CTRL_2H = 3
};

/*****************************************************************************\
CONST:
    const unsigned int numberOfShaderSyncBits

    Determines number of bits used for SHADER_SYNC

\*****************************************************************************/
const unsigned int cNumberOfShaderSyncBits = 6;

/*****************************************************************************\
UNION: SHADER_SYNC
\*****************************************************************************/
union SHADER_SYNC
{
    struct _BasicType
    {
       unsigned int COMPUTE_SHADER_SYNC_THREADS_IN_GROUP : 1; // Thread group sync
       unsigned int COMPUTE_SHADER_SYNC_THREAD_GROUP_SHARED_MEMORY : 1; // g# (Thread Group Shared Memory) fence
       unsigned int COMPUTE_SHADER_SYNC_UNORDERED_ACCESS_VIEW_MEMORY_GROUP : 1; // Thread group scope u# (UAV) memory fence
       unsigned int SHADER_SYNC_UNORDERED_ACCESS_VIEW_MEMORY_GLOBAL : 1; // Global u# (UAV) memory fence (the only onle allowed in PS)
       unsigned int PIXEL_SHADER_SYNC_PIXEL_BARRIER : 1;
       unsigned int COMPUTE_SHADER_SYNC_IMAGE : 1;
    } ShaderSyncFields;

    int unsigned AllShaderSyncFields : cNumberOfShaderSyncBits;

    bool operator == ( const SHADER_SYNC& toCheck ) const
    {
        return ( AllShaderSyncFields == toCheck.AllShaderSyncFields );
    }

    bool operator != ( const SHADER_SYNC& toCheck ) const
    {
        return !( AllShaderSyncFields == toCheck.AllShaderSyncFields );
    }

    SHADER_SYNC operator | ( const SHADER_SYNC& toCheck ) const
    {
        SHADER_SYNC ret;
        ret.AllShaderSyncFields = ( AllShaderSyncFields | toCheck.AllShaderSyncFields );
        return ret;
    }

    operator unsigned int( ) const
    {
        return AllShaderSyncFields;
    }

    static SHADER_SYNC Create( unsigned int allShaderSyncFields )
    {
        SHADER_SYNC ret;
        ret.AllShaderSyncFields = allShaderSyncFields;
        return ret;
    }
};

const SHADER_SYNC SHADER_SYNC_NO_FLAGS =                                    { { 0, 0, 0, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_THREADS_IN_GROUP =                    { { 1, 0, 0, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_THREAD_GROUP_SHARED_MEMORY =          { { 0, 1, 0, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_G_T =                                 { { 1, 1, 0, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_UNORDERED_ACCESS_VIEW_MEMORY_GROUP =  { { 0, 0, 1, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_UGROUP_T =                            { { 1, 0, 1, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_UGROUP_G =                            { { 0, 1, 1, 0, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_UGROUP_G_T =                          { { 1, 1, 1, 0, 0, 0 } };
const SHADER_SYNC SHADER_SYNC_UNORDERED_ACCESS_VIEW_MEMORY_GLOBAL =         { { 0, 0, 0, 1, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SHADER_SYNC_UGLOBAL_T =                    { { 1, 0, 0, 1, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SHADER_SYNC_UGLOBAL_G =                    { { 0, 1, 0, 1, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SHADER_SYNC_UGLOBAL_G_T =                  { { 1, 1, 0, 1, 0, 0 } };
const SHADER_SYNC PIXEL_SHADER_SYNC_PIXEL_BARRIER =                         { { 0, 0, 0, 0, 1, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_COUNT_ANY =                           { { 1, 0, 1, 1, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_COUNT_ALL =                           { { 0, 1, 1, 1, 0, 0 } };
const SHADER_SYNC COMPUTE_SHADER_SYNC_IMAGE =                               { { 0, 0, 0, 0, 0, 1 } };
const SHADER_SYNC NUM_SHADER_SYNCS_INIT =                                   { { 1, 1, 1, 1, 1, 1 } };
const unsigned int NUM_SHADER_SYNCS = NUM_SHADER_SYNCS_INIT.AllShaderSyncFields + 1;

/*****************************************************************************\
ENUM: SHADER_REGISTER_FILE
\*****************************************************************************/
enum SHADER_REGISTER_FILE
{
    SHADER_REGFILE_INVALID,
    SHADER_REGFILE_TEMP,                // r#, x#[]
    SHADER_REGFILE_INPUT,               // i#
    SHADER_REGFILE_OUTPUT,              // o#
    SHADER_REGFILE_SAMPLER,             // s#
    SHADER_REGFILE_RESOURCE,            // t#
    SHADER_REGFILE_CONSTANT_BUFFER,     // C#[]
    SHADER_REGFILE_CONSTANT_REGISTER,   // c#
    SHADER_REGFILE_PREDICATE,           // p#
    SHADER_REGFILE_IMMEDIATE,           // {}
    SHADER_REGFILE_PRIMITIVEID,         // pid
    SHADER_REGFILE_NULL,                // null
    SHADER_REGFILE_RASTERIZER,          // rast
    SHADER_REGFILE_PATCH_CONSTANT_INPUT,       // ipc#
    SHADER_REGFILE_OUTPUT_CONTROL_POINT_INPUT, // iocp#
    SHADER_REGFILE_CONTROL_POINT_ID,           // cpid
    SHADER_REGFILE_FORK_INSTANCE_ID,           // fiid
    SHADER_REGFILE_JOIN_INSTANCE_ID,           // jiid
    SHADER_REGFILE_DOMAIN,                     // dom
    SHADER_REGFILE_UAV,                 // u#
    SHADER_REGFILE_TGSM,                // g#
    SHADER_REGFILE_TPM,                 // m#
    SHADER_REGFILE_GENERIC_MG,          // mg,
    SHADER_REGFILE_GENERIC_UMG,         // umg
    SHADER_REGFILE_GENERIC_UG,          // ug
    SHADER_REGFILE_GENERIC_UM,          // um
    SHADER_REGFILE_THREAD_ID,           // vThreadID
    SHADER_REGFILE_THREAD_GROUP_ID,     // vThreadGroupID
    SHADER_REGFILE_THREAD_ID_IN_GROUP,  // vThreadIDInGroup
    SHADER_REGFILE_THREAD_ID_IN_GROUP_FLATTENED,  // vThreadIDInGroupFlattened
    SHADER_REGFILE_POINTER,             // ptr#
    SHADER_REGFILE_GSINSTANCEID,        // vGSInstanceID
    SHADER_REGFILE_ADDRESSABLESO_STREAMID, // vGSAddressableSOStreamID
    SHADER_REGFILE_CYCLE_COUNTER,       // vCycleCounter
    SHADER_REGFILE_PATCH_CONSTANT,      //opc#
    SHADER_REGFILE_JOURNAL,             // journal buffer for kernel instrumentation
    SHADER_REGFILE_STRING,              // str#
    SHADER_REGFILE_INPUT_COVERAGE_MASK, // vCoverage
    SHADER_REGFILE_OUTPUT_STENCIL,      // oStencil
    SHADER_REGFILE_INPUT_CONTROL_POINT_COUNT, // icpc OpenGL 4.3 gl_PatchVerticesIn
    SHADER_REGFILE_PS2SS,               // ps#
    SHADER_REGFILE_INNER_OUTER_TESSFACTOR_INOUT,
    NUM_SHADER_REGISTER_FILES
};

/*****************************************************************************\
CONST: Indicates that printf data size was not explicitly set.
\*****************************************************************************/
const long  SHADER_PRINTF_DATA_SIZE_NOT_SET = -1;

/*****************************************************************************\
ENUM: SHADER_PRINTF_TYPE
\*****************************************************************************/
enum SHADER_PRINTF_TYPE
{
    SHADER_PRINTF_INVALID,
    SHADER_PRINTF_BYTE,
    SHADER_PRINTF_SHORT,
    SHADER_PRINTF_INT,
    SHADER_PRINTF_FLOAT,
    SHADER_PRINTF_STRING_LITERAL,
    SHADER_PRINTF_LONG,
    SHADER_PRINTF_POINTER,
    SHADER_PRINTF_DOUBLE,
    NUM_SHADER_PRINTF_TYPES
};

/*****************************************************************************\
ENUM: GSHADER_INPUT_PRIMITIVE_TYPE
\*****************************************************************************/
enum GSHADER_INPUT_PRIMITIVE_TYPE
{
    GSHADER_INPUT_POINT,
    GSHADER_INPUT_LINE,
    GSHADER_INPUT_TRIANGLE,
    GSHADER_INPUT_LINE_ADJ,
    GSHADER_INPUT_TRIANGLE_ADJ,
    GSHADER_INPUT_QUAD,
    GSHADER_INPUT_RECTANGLE,
    GSHADER_INPUT_PATCHLIST_1,
    GSHADER_INPUT_PATCHLIST_2,
    GSHADER_INPUT_PATCHLIST_3,
    GSHADER_INPUT_PATCHLIST_4,
    GSHADER_INPUT_PATCHLIST_5,
    GSHADER_INPUT_PATCHLIST_6,
    GSHADER_INPUT_PATCHLIST_7,
    GSHADER_INPUT_PATCHLIST_8,
    GSHADER_INPUT_PATCHLIST_9,
    GSHADER_INPUT_PATCHLIST_10,
    GSHADER_INPUT_PATCHLIST_11,
    GSHADER_INPUT_PATCHLIST_12,
    GSHADER_INPUT_PATCHLIST_13,
    GSHADER_INPUT_PATCHLIST_14,
    GSHADER_INPUT_PATCHLIST_15,
    GSHADER_INPUT_PATCHLIST_16,
    GSHADER_INPUT_PATCHLIST_17,
    GSHADER_INPUT_PATCHLIST_18,
    GSHADER_INPUT_PATCHLIST_19,
    GSHADER_INPUT_PATCHLIST_20,
    GSHADER_INPUT_PATCHLIST_21,
    GSHADER_INPUT_PATCHLIST_22,
    GSHADER_INPUT_PATCHLIST_23,
    GSHADER_INPUT_PATCHLIST_24,
    GSHADER_INPUT_PATCHLIST_25,
    GSHADER_INPUT_PATCHLIST_26,
    GSHADER_INPUT_PATCHLIST_27,
    GSHADER_INPUT_PATCHLIST_28,
    GSHADER_INPUT_PATCHLIST_29,
    GSHADER_INPUT_PATCHLIST_30,
    GSHADER_INPUT_PATCHLIST_31,
    GSHADER_INPUT_PATCHLIST_32,
    NUM_GSHADER_INPUT_PRIMITIVE_TYPES
};

/*****************************************************************************\
ENUM: PRIMITIVE_TOPOLOGY_TYPE
\*****************************************************************************/
enum PRIMITIVE_TOPOLOGY_TYPE
{
    PRIMITIVE_TOPOLOGY_POINTLIST,
    PRIMITIVE_TOPOLOGY_POINTLIST_BACKFACE,
    PRIMITIVE_TOPOLOGY_LINELIST,
    PRIMITIVE_TOPOLOGY_LINELIST_ADJACENCY,
    PRIMITIVE_TOPOLOGY_LINESTRIP,
    PRIMITIVE_TOPOLOGY_LINESTRIP_CONTINUOUS,
    PRIMITIVE_TOPOLOGY_LINESTRIP_BACKFACE,
    PRIMITIVE_TOPOLOGY_LINESTRIP_CONTINUOUS_BACKFACE,
    PRIMITIVE_TOPOLOGY_LINESTRIP_ADJACENCY,
    PRIMITIVE_TOPOLOGY_LINELOOP,
    PRIMITIVE_TOPOLOGY_TRIANGLELIST,
    PRIMITIVE_TOPOLOGY_TRIANGLELIST_ADJACENCY,
    PRIMITIVE_TOPOLOGY_TRIANGLESTRIP,
    PRIMITIVE_TOPOLOGY_TRIANGLESTRIP_ADJACENCY,
    PRIMITIVE_TOPOLOGY_TRIANGLESTRIP_REVERSE,
    PRIMITIVE_TOPOLOGY_TRIANGLEFAN,
    PRIMITIVE_TOPOLOGY_TRIANGLEFAN_NOSTIPPLE,
    PRIMITIVE_TOPOLOGY_QUADLIST,
    PRIMITIVE_TOPOLOGY_QUADSTRIP,
    PRIMITIVE_TOPOLOGY_POLYGON,
    PRIMITIVE_TOPOLOGY_RECTLIST,
    PRIMITIVE_TOPOLOGY_CLEAR,
    PRIMITIVE_TOPOLOGY_PATCHLIST_1,
    PRIMITIVE_TOPOLOGY_PATCHLIST_2,
    PRIMITIVE_TOPOLOGY_PATCHLIST_3,
    PRIMITIVE_TOPOLOGY_PATCHLIST_4,
    PRIMITIVE_TOPOLOGY_PATCHLIST_5,
    PRIMITIVE_TOPOLOGY_PATCHLIST_6,
    PRIMITIVE_TOPOLOGY_PATCHLIST_7,
    PRIMITIVE_TOPOLOGY_PATCHLIST_8,
    PRIMITIVE_TOPOLOGY_PATCHLIST_9,
    PRIMITIVE_TOPOLOGY_PATCHLIST_10,
    PRIMITIVE_TOPOLOGY_PATCHLIST_11,
    PRIMITIVE_TOPOLOGY_PATCHLIST_12,
    PRIMITIVE_TOPOLOGY_PATCHLIST_13,
    PRIMITIVE_TOPOLOGY_PATCHLIST_14,
    PRIMITIVE_TOPOLOGY_PATCHLIST_15,
    PRIMITIVE_TOPOLOGY_PATCHLIST_16,
    PRIMITIVE_TOPOLOGY_PATCHLIST_17,
    PRIMITIVE_TOPOLOGY_PATCHLIST_18,
    PRIMITIVE_TOPOLOGY_PATCHLIST_19,
    PRIMITIVE_TOPOLOGY_PATCHLIST_20,
    PRIMITIVE_TOPOLOGY_PATCHLIST_21,
    PRIMITIVE_TOPOLOGY_PATCHLIST_22,
    PRIMITIVE_TOPOLOGY_PATCHLIST_23,
    PRIMITIVE_TOPOLOGY_PATCHLIST_24,
    PRIMITIVE_TOPOLOGY_PATCHLIST_25,
    PRIMITIVE_TOPOLOGY_PATCHLIST_26,
    PRIMITIVE_TOPOLOGY_PATCHLIST_27,
    PRIMITIVE_TOPOLOGY_PATCHLIST_28,
    PRIMITIVE_TOPOLOGY_PATCHLIST_29,
    PRIMITIVE_TOPOLOGY_PATCHLIST_30,
    PRIMITIVE_TOPOLOGY_PATCHLIST_31,
    PRIMITIVE_TOPOLOGY_PATCHLIST_32,
    NUM_PRIMITIVE_TOPOLOGY_TYPES
};

/*****************************************************************************\
ENUM: CLEAR_PS
\*****************************************************************************/
enum CLEAR_PS
{
    NO_REPLICATEDATA_CLEAR,     // for cases where replicate data can not be used
    REPLICATEDATA_CLEAR,        // for color buffer clears where replicate data can be used
    REPLICATEDATA_FASTCLEAR,    // for fast clear of MCS surfaces
    NUM_CLEAR_PS
};

/*****************************************************************************\
ENUM: SHADER_VERSION_TYPE
\*****************************************************************************/
enum SHADER_VERSION_TYPE
{
    SHADER_VERSION_1_0,
    SHADER_VERSION_2_0,
    SHADER_VERSION_2_1,
    SHADER_VERSION_3_0,
    SHADER_VERSION_3_1,
    SHADER_VERSION_4_0,
    NUM_SHADER_VERSION_TYPES
};

/*****************************************************************************\
ENUM: USC_PLANAR_YUV_FORMAT

  Specifies if a texture is in one of the planar YUV formats.
  Such formats require a special handling and therefore are a NOS.
  Cf. USC_OPENGL_PLANAR_YUV_FORMAT in opengl_usc.h
\*****************************************************************************/
enum USC_PLANAR_YUV_FORMAT
{
    USC_PLANAR_YUV_FORMAT_NONE = 0,
    USC_PLANAR_YUV_FORMAT_YV12,
    USC_PLANAR_YUV_FORMAT_NV12,
    USC_PLANAR_YUV_FORMAT_NV21,
    USC_PLANAR_YUV_FORMAT_P010,
    USC_NUM_PLANAR_YUV_FORMATS
};

/*****************************************************************************\
ENUM: USC_FLOAT_PRECISION
\*****************************************************************************/
enum USC_FLOAT_PRECISION
{
    USC_FLOAT16,
    USC_FLOAT32,
    USC_FLOAT64,
    USC_FLOAT_NUM_PRECISIONS
};

/*****************************************************************************\
ENUM: USC_FLOAT DENORM MODE
\*****************************************************************************/
enum USC_FLOAT_DENORM_MODE
{
    USC_FLOAT_DENORM_DEFAULT = 0,         // left default mode (not defined by the user)
    USC_FLOAT_DENORM_FORCE_FLUSH_TO_ZERO, // forces user defined flushing denorms to zero
    USC_FLOAT_DENORM_FORCE_RETAIN_DENORMS,// forces user defined retaining to denorms
    USC_FLOAT_DENORM_NUM_MODES
};

/*****************************************************************************\
ENUM: USC_EARLY_DEPTH_STENCIL_TEST_MODE
\*****************************************************************************/
enum USC_EARLY_DEPTH_STENCIL_TEST_MODE
{
    USC_EARLY_DEPTH_STENCIL_TEST_MODE_DEFAULT = 0, // not the same as POST, because driver might choose which one is more optimal
    USC_EARLY_DEPTH_STENCIL_TEST_MODE_POST_PS,
    USC_EARLY_DEPTH_STENCIL_TEST_MODE_PRE_PS,
    USC_EARLY_DEPTH_STENCIL_TEST_NUM_MODES
};


/*****************************************************************************\
ENUM: USC_SHADING_RATE
\*****************************************************************************/
enum USC_SHADING_RATE
{
    USC_SHADING_RATE_DEFAULT,
    USC_SHADING_RATE_PIXEL,
    USC_SHADING_RATE_SAMPLE,
    USC_SHADING_RATE_COARSE,
    USC_NUM_SHADING_RATES
};

/*****************************************************************************\
ENUM: USC_BLEND_EQUATION_ADVANCED_MODE
\*****************************************************************************/
enum USC_BLEND_EQUATION_ADVANCED_MODE
{
    USC_BLEND_EQUATION_ADVANCED_NONE = 0,
    USC_BLEND_EQUATION_ADVANCED_MULTIPLY,
    USC_BLEND_EQUATION_ADVANCED_SCREEN,
    USC_BLEND_EQUATION_ADVANCED_OVERLAY,
    USC_BLEND_EQUATION_ADVANCED_DARKEN,
    USC_BLEND_EQUATION_ADVANCED_LIGHTEN,
    USC_BLEND_EQUATION_ADVANCED_COLORDODGE,
    USC_BLEND_EQUATION_ADVANCED_COLORBURN,
    USC_BLEND_EQUATION_ADVANCED_HARDLIGHT,
    USC_BLEND_EQUATION_ADVANCED_SOFTLIGHT,
    USC_BLEND_EQUATION_ADVANCED_DIFFERENCE,
    USC_BLEND_EQUATION_ADVANCED_EXCLUSION,
    USC_BLEND_EQUATION_ADVANCED_HSL_HUE,
    USC_BLEND_EQUATION_ADVANCED_HSL_SATURATION,
    USC_BLEND_EQUATION_ADVANCED_HSL_COLOR,
    USC_BLEND_EQUATION_ADVANCED_HSL_LUMINOSITY,
    USC_NUM_BLEND_EQUATION_ADVANCED_MODES
};

}
