/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.jmeter.threads;

import org.apache.jmeter.samplers.SampleResult;
import org.apache.jmeter.samplers.Sampler;
import org.apache.jmeter.engine.StandardJMeterEngine;

/**
 * Holds context for a thread.
 * Generated by JMeterContextService.
 * 
 * @NotThreadSafe // Only intended for use within a thread
 */
public class JMeterContext {
    private JMeterVariables variables;

    private SampleResult previousResult;

    private Sampler currentSampler;

    private Sampler previousSampler;

    private boolean samplingStarted;

    private StandardJMeterEngine engine;

    private JMeterThread thread;

    private ThreadGroup threadGroup;

    private int threadNum;

    private byte[] readBuffer = null;

    JMeterContext() {
        clear0();
    }

    public void clear() {
        clear0();
    }

    private void clear0() {
        variables = null;
        previousResult = null;
        currentSampler = null;
        previousSampler = null;
        samplingStarted = false;
        threadNum = 0;
        readBuffer = null;
    }

    /**
     * Gives access to the JMeter variables for the current thread.
     * 
     * @return a pointer to the JMeter variables.
     */
    public JMeterVariables getVariables() {
        return variables;
    }

    /**
     * A temporary buffer that can be shared between samplers in a thread.
     * 
     * @return the shared read buffer
     */
    public byte[] getReadBuffer() {
        if (readBuffer == null) {
            readBuffer = new byte[8192];
        }
        return readBuffer;
    }

    public void setVariables(JMeterVariables vars) {
        this.variables = vars;
    }

    public SampleResult getPreviousResult() {
        return previousResult;
    }

    public void setPreviousResult(SampleResult result) {
        this.previousResult = result;
    }

    public Sampler getCurrentSampler() {
        return currentSampler;
    }

    public void setCurrentSampler(Sampler sampler) {
        this.previousSampler = currentSampler;
        this.currentSampler = sampler;
    }

    /**
     * Returns the previousSampler.
     *
     * @return Sampler
     */
    public Sampler getPreviousSampler() {
        return previousSampler;
    }

    /**
     * Returns the threadNum.
     *
     * @return int
     */
    public int getThreadNum() {
        return threadNum;
    }

    /**
     * Sets the threadNum.
     *
     * @param threadNum
     *            the threadNum to set
     */
    public void setThreadNum(int threadNum) {
        this.threadNum = threadNum;
    }

    public JMeterThread getThread() {
        return this.thread;
    }

    public void setThread(JMeterThread thread) {
        this.thread = thread;
    }

    public ThreadGroup getThreadGroup() {
        return this.threadGroup;
    }

    public void setThreadGroup(ThreadGroup threadgrp) {
        this.threadGroup = threadgrp;
    }

    public StandardJMeterEngine getEngine() {
        return engine;
    }

    public void setEngine(StandardJMeterEngine engine) {
        this.engine = engine;
    }

    public boolean isSamplingStarted() {
        return samplingStarted;
    }

    public void setSamplingStarted(boolean b) {
        samplingStarted = b;
    }
}
