# CKString shows the result of binding as string.
#
# == Bindings
# Required attributes: <b>value</b>
#
# <b>value</b>::  Text to be displayed.   
# <b>escape</b>:: Escapes HTML control characters in "value" if the "escape"
#                 is true. The default value is true.
# <b>empty</b>::  Text that is substituted for "value"
#                 when the "value" is nil.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKString < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['value', 'escape', 'empty']
		end
	end

	def run	
		@attr_value  = fetch( 'value' )  # String
		@attr_escape = fetch( 'escape' ) # true or false
		@attr_empty  = fetch( 'empty' )  # String
		if @attr_escape == nil then
			@attr_escape = true
		end

		check_required_attributes( ['value'], ['empty'] )
	end

	def to_s
		to_s = nil

		if @attr_value then
			to_s = @attr_value
		elsif @attr_empty then
			to_s = @attr_empty
		end

		if to_s and (@attr_escape == true) then
			to_s = CKUtilities.escape_html to_s
		end

		to_s
	end
end


# CKHyperlink generates a hypertext link.
#
# == Bindings
# Required attributes: <b>action</b>, <b>href</b> or <b>page</b>.
#
# <b>action</b>::  Method to be invoked when the link is clicked.
# <b>enabled</b>:: Generates a non-active link if the value is false.
# <b>href</b>::    You specify the URL to other web page directly.
#                  This attribute prevails over "action" or "page" attribute.
# <b>page</b>::    Name of component to display when the link is clicked.
# <b>string</b>::  Text of the link. If the body of CKHyperlink tag is
#                  not empty, the body is displayed. For example,
#                  if the template includes "<cgikit name=link>foo</cgikit>",
#                  this element shows "foo" as the link.
# <b>target</b>::  target attribute of HTML's <a> tag. 
# <b>secure</b>::  Appends "https://" to the URL if the value is true.
#                  The default value is false.
# <b>query</b>::   Hash as the query string. The value of "query" attribute is
#                  converted to string. Then, the string is added to append
#                  the URL.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKHyperlink < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['action', 'enabled', 'href', 'page', 'string', 'target',
			 'secure', 'query']
		end
	end

	def run
		@attr_action  = fetch( 'action', false ) # CKComponent
		@attr_enabled = fetch( 'enabled' )       # true or false
		@attr_href    = fetch( 'href' )          # String
		@attr_page    = fetch( 'page' )          # String
		@attr_string  = fetch( 'string' )        # String
		@attr_target  = fetch( 'target' )        # String
		@attr_secure  = fetch( 'secure' )        # true or false
		@attr_query   = fetch( 'query' ) || {}   # Hash

		if @attr_string then
			@attr_string = @attr_string.to_s
		end

		check_required_attributes( ['action'], ['href'], ['page'] )
		check_conflicted_attributes( 'action', 'href', 'page' )
	end

	def to_s
		container = nil
		unless body.empty? then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			container               = parser.parse
		end

		# action
		if    @attr_page   then id = CKElementID.new @attr_page
		elsif @attr_action then id = element_id
		else                    id = nil end
		@attr_action = application.url( id, @attr_query, @attr_secure )

		# to_s
		to_s = '<a'
		if   @attr_href    then to_s << " href=\"#@attr_href\""
		else                    to_s << " href=\"#@attr_action\""   end
		if   @attr_target  then to_s << " target=\"#@attr_target\"" end
		to_s << other_attributes_string
		to_s << '>'
		if    @attr_string then to_s << @attr_string
		elsif container    then to_s << container end
		to_s << '</a>'
		
		if @attr_enabled == false then
			to_s = @attr_string
		end
		if to_s.nil? || to_s.length == 0 then
			to_s = container
		end

		to_s
	end
end


# Creates an image tag.
#
# == Bindings
# Required attributes: <b>file</b>, <b>src</b>, or <b>data</b>.
#
# <b>alt</b>::    Alternative text to the picture.
# <b>border</b>:: Size of image border.
# <b>width</b>::  Width of image.
# <b>height</b>:: Height of image.
# <b>file</b>::   Name of an image file in web server resource directory.
# <b>src</b>::    You specify an image file directly. This value prevails 
#                 over "file" attribute.
# <b>data</b>::   A CKByteData object to display as image.
#                 If you create the object without resource manager,
#                 You must use this with "mime" attribute.
# <b>mime</b>::   MIME type for a resource of "data" attribute.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKImage < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['alt', 'border', 'width', 'height', 'file', 'src', 'data', 'mime']
		end
	end

	def run
		@attr_alt    = fetch( 'alt' )         # String
		@attr_border = fetch( 'border' )      # Integer
		@attr_width  = fetch( 'width' )       # Integer
		@attr_height = fetch( 'height' )      # Integer
		@attr_file   = fetch( 'file' )        # String
		@attr_src    = fetch( 'src' )         # String
		@attr_data   = fetch( 'data', false ) # String

		check_required_attributes( ['file'], ['src'], ['data','mime'] )
		check_conflicted_attributes( 'file', 'src', 'data' )
	end

	def to_s
		if @attr_file then
			@attr_src = application.resource_manager.url @attr_file
		elsif @attr_data then
			@attr_src = application.url element_id
		end

		to_s = "<img src=\"#@attr_src\" alt=\"#@attr_alt\""
		if @attr_border then to_s << " border=\"#@attr_border\"" end
		if @attr_width  then to_s << " width=\"#@attr_width\"" end
		if @attr_height then to_s << " height=\"#@attr_height\"" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# Controls generating HTML.
#
# == Bindings
# Required attribute: <b>condition</b>
#
# <b>condition</b>:: If the value is true and "negate" is false, the body of the 
#                    CKCoditional tag is displayed.
# <b>negate</b>::    Inverts the meaning of the "condition".
#
# == Control table
#   condition  negate  result
#   true       false   show
#   false      false   not show
#   true       true    not show
#   false      false   show
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKConditional < CKElement
	include ElementAttribute, ElementState

	class << self
		def bind_request( component, definition, value, id )
			parsed = CKElement::ElementState.parse( component, value.first )

			# change if the definition is variable
			if component.variable? definition['condition'] then
				component.take_value( definition['condition'], parsed['condition'])
			end
			if component.variable? definition['negate'] then
				component.take_value( definition['negate'], parsed['negate'] )
			end
		end

		def own_attributes
			['condition', 'negate']
		end
	end

	def run
		@attr_condition = fetch( 'condition' )
		@attr_negate    = fetch( 'negate' )

		check_required_attributes( ['condition'] )
	end

	def to_s
		to_s = ''
		if ( ( ( not @attr_negate    ) and @attr_condition ) or \
		     ( ( not @attr_condition ) and @attr_negate    ) ) then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			to_s = parser.parse
		end
		to_s
	end
end


# A CKRepeition object repeats its contents. 
#
# == Bindings
# Required attributes: <b>list</b> and <b>item</b>, or <b>count</b>
#
# <b>count</b>:: CKRepeition repeats its contents this number of times.
#                The attribute conflicts with "list".
# <b>list</b>::  Array which is iterated through.
# <b>item</b>::  Current item when the list is iterated through.
# <b>index</b>:: Index of the current item.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKRepetition < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['count', 'list', 'item', 'index']
		end
	end

	def run
		@attr_count = fetch( 'count' )        # Integer
		@attr_item  = fetch( 'item', false )
		@attr_list  = fetch( 'list' )         # Enumerable
		@attr_index = fetch( 'index', false )

		@repetitions << [ @name, @repetition_index ]

		check_required_attributes( ['list', 'item'], ['count'] )
		check_conflicted_attributes( 'list', 'count' )
	end

 	def to_s
 		to_s  = ''

 		if @attr_list then
			@attr_list.each_with_index do | item, index |
				parent.take_value( @attr_item, item )
				to_s << _iteration( index )
			end
		elsif @attr_count then
			@attr_count.times do | index |
				to_s << _iteration( index )
 			end
 		end
 		to_s
 	end

	private
	def _iteration( index )
		parent.take_value( @attr_index, index ) if @attr_index
		parser                  = CKHTMLParser.new( parent, body )
		parser.repetitions      = @repetitions
		parser.repetition_index = index
		parser.parse
	end
end


# CKContent is used in nested components.
# This element tag in the template is replaced
# with a part of the template of its grandparent component.
#
# == Bindings
# CKContent doesn't have any attribute.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKContent < CKElement
	include ElementAttribute

	def to_s
		unless parent.parent
			return
		end

		parser = CKHTMLParser.new( parent.parent, parent.body )
		parser.parse
	end
end


# CKFrame generates frame tag in HTML.
#
# == Bindings
# Required attributes: <b>page</b>, <b>src</b> or <b>value</b>.
#
# The priorities of "src", "page" and "value" is "src", 
# "page", "value". 
# 
# <b>name</b>::  "name" attribute of HTML's <frame> tag.
# <b>page</b>::  Name of component that supplies the content
#                for the frame.
# <b>src</b>::   You specify the URL or file for the frame. 
# <b>value</b>:: Method that supplies the content. The parent of this element
#                must have the specified method.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKFrame < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['name', 'page', 'src', 'value']
		end
	end

	def run
		@attr_name  = fetch( 'name' )  # String
		@attr_page  = fetch( 'page' )  # String
		@attr_src   = fetch( 'src' )   # String
		@attr_value = fetch( 'value' ) # CKComponent

		check_required_attributes( ['page'], ['src'], ['value'] )
		check_conflicted_attributes( 'page', 'src', 'value' )
	end

	def to_s
		if @attr_src then
		elsif @attr_page then
			id = CKElementID.new @attr_page
			@attr_src = application.url id
		elsif @attr_value then
			@attr_src = application.url element_id
		end

		to_s =  "<frame name=\"#@attr_name\" src=\"#@attr_src\""
		to_s << other_attributes_string
		to_s << ">"
	end
end


# CKGenericElement generates generic HTML tags.
#
# == Bindings
#
# Required attributes: <b>tag</b>
#
# <b>tag</b>::           Name of the HTML tag. If the attribute is nil,
#                        body enclosed by the element or "string"
#                        attribute are displayed.
# <b>enabled</b>::       Enables or disables the tag. If the attribute is false,
#                        body enclosed by the element or "string"
#                        attribute are displayed.
# <b>string</b>::        String to display if body enclosed by the element
#                        isn't exist.
# <b>option</b>::        String to append for the open tag. For example,
#                        "checked" or "selected".
# <b>form_value</b>::    If the element is form, the attribute is setted
#                        form datas as a string.
# <b>form_values</b>::   If the element is form, the attribute is setted
#                        form datas as an array.
# <b>invoke_action</b>:: If the element is executable ( hyperlink, button,
#                        etc. ), the method is called when clicked.
#
# You can define other voluntary attributes.
# The attributes is appended to the tag in format as "attribute=value".
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKGenericElement < CKElement
	include ElementAttribute

	NO_CLOSE_TAGS = [ 'area', 'base', 'basefont', 'br', 'col', 'frame', 'hr',
	                  'img', 'input', 'link', 'map', 'meta', 'param' ]

	class << self
		def bind_request( component, definition, value, id )
			if definition['invoke_action'] then
				component.application.element_id = id
			end

			if action = definition['form_value'] then
				if id.repetitions? then
					list          = component.retrieve_value action
					list_id       = id.repetitions.last.last
					list[list_id] = value.first
				else
					component.take_value( action, value.first )
				end
			end

			if action = definition['form_values'] then
				if id.repetitions? then
					list          = component.retrieve_value action
					list_id       = id.repetitions.last.last
					list[list_id] = value
				else
					component.take_value( action, value )
				end
			end
		end

		def own_attributes
			['tag', 'enabled', 'string', 'option', 'form_value',
			 'form_values', 'invoke_action']
		end
	end

	def run
		@attr_name          = element_id                      # String
		@attr_tag           = fetch( 'tag' )                  # String
		@attr_enabled       = fetch( 'enabled' ) == false ? false : true # true/false
		@attr_string        = fetch( 'string' )               # String
		@attr_value         = fetch( 'value' )                # String
		@attr_form_value    = fetch( 'form_value' )           # String
		@attr_form_values   = fetch( 'form_values' )          # String
		@attr_invoke_action = fetch( 'invoke_action', false ) # String
		@attr_option        = fetch( 'option' )               # String

		check_required_attributes( ['tag'] )
	end

	def to_s
		string = nil
		if @attr_string then
			string = @attr_string
		elsif body.empty? == false then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			container               = parser.parse

			if container then string = container
			else              string = @attr_string end
		end

		to_s = value = href = nil
		if @attr_enabled == true then
			# hyperlink, <form> or <a>
			if    @attr_invoke_action   then href = application.url element_id
			elsif _href = fetch('href') then href = _href end

			# value
			if    @attr_value       then value = @attr_value
			elsif @attr_form_value  then value = @attr_form_value
			elsif @attr_form_values then
				value = @attr_form_values[@repetition_index]
			end

			# radiobutton
			if (@attr_tag.downcase == 'input') and \
				(fetch('type').downcase == 'radio') and (_name = fetch('name')) then
				@attr_name.element = _name
				if value == nil then value = 'on' end
				value = "#{definition['oid']}.#{value}"
			end

			to_s =  "<#@attr_tag name=\"#@attr_name\""
			to_s << " value=\"#{value}\"" if value
			to_s << " href=\"#{href}\""   if href
			to_s << " #@attr_option"      if @attr_option
			to_s << other_attributes_string
			to_s << '>'
			to_s << string if string

			unless NO_CLOSE_TAGS.include? @attr_tag then
				to_s << "</#@attr_tag>"
			end
		else
			to_s = string
		end

		to_s
	end
end


