# CKUtilitis is a module wihch collects utility methods based on cgi.rb.
module CKUtilities
	CR            = "\015"
	LF            = "\012"
	EOL           = CR + LF
	RFC822_DAYS   = %w[ Sun Mon Tue Wed Thu Fri Sat ]
	RFC822_MONTHS = %w[ Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ]

	# Returns an encoded string for URL.
	def escape_url( string )
		string.to_s.gsub( /([^ a-zA-Z0-9_.-]+)/n ) do
			'%' + $1.unpack( 'H2' * $1.size ).join( '%' ).upcase
		end.tr( ' ', '+' )
	end

	# Returns a string decoded from URL.
	def unescape_url( string )
		string.to_s.tr( '+', ' ' ).gsub( /((?:%[0-9a-fA-F]{2})+)/n ) do
			[ $1.delete( '%' ) ].pack( 'H*' )
		end
	end

	# Escapes HTML control characters.
	def escape_html( string )
		string.to_s.gsub(/&/n, '&amp;').
			gsub(/\"/n, '&quot;').
			gsub(/>/n, '&gt;').
			gsub(/</n, '&lt;').
			gsub(/'/n, '&#39;')
	end

	# Unescapes HTML control characters.
	def unescape_html( string )
  		string.to_s.gsub(/&(.*?);/n) do
	    	match = $1.dup
	    	case match
    		when /\Aamp\z/ni           then '&'
			when /\Aquot\z/ni          then '"'
			when /\Agt\z/ni            then '>'
			when /\Alt\z/ni            then '<'
			when /\A#0*(\d+)\z/n       then
				if Integer($1) < 256
					Integer($1).chr
				else
					if Integer($1) < 65536 and \
						($KCODE[0] == ?u or $KCODE[0] == ?U)
						[Integer($1)].pack("U")
					else
						"&##{$1};"
					end
				end
			when /\A#x([0-9a-f]+)\z/ni then
				if $1.hex < 256
					$1.hex.chr
				else
					if $1.hex < 65536 and \
						($KCODE[0] == ?u or $KCODE[0] == ?U)
						[$1.hex].pack("U")
					else
						"&#x#{$1};"
					end
				end
			else
				"&#{match};"
			end
		end
	end

	# Formats Time object in RFC1123.
	# For example, "Sat, 1 Jan 2000 00:00:00 GMT".
	def date( time )
		t = time.clone.gmtime
		return format("%s, %.2d %s %.4d %.2d:%.2d:%.2d GMT",
			RFC822_DAYS[t.wday], t.day, RFC822_MONTHS[t.month-1], t.year,
			t.hour, t.min, t.sec)
	end

	module_function :escape_url
	module_function :unescape_url
	module_function :escape_html
	module_function :unescape_html
	module_function :date
end


# CKLog is a simple logging class.
#
# == Debug level
# DEBUG < INFO < WARN < ERROR < FATAL
class CKLog

	DEBUG          = 1
	INFO           = 2
	WARN           = 3
	ERROR          = 4
	FATAL          = 5
	DEFAULT_LEVEL  = DEBUG

	# Log level. The default level is DEBUG.
	attr_accessor :level

	# Outputter. The default outputter is standard error.
	attr_accessor :out

	# Program name. The default name is "CGIKit" and the version.
	attr_accessor :name

	# Max file size to log. If size of file to output is over the max file size,
	# exception FileSizeError is raised.
	attr_accessor :max_file_size

	class FileSizeError < CKError; end #:nodoc:

	class << self
		# Outputs a debug string to STDERR.
		# A backtrace is sent if "exception" is given an error object.
		def debug( message, exception = nil )
			message = "[#{Time.now}] DEBUG - #{CKApplication.version}: #{message}\n"
			if exception then
				message << exception.backtrace.join << "\n"
			end
			$stderr.write message
		end
	end

	def initialize( options = {} )
		@level         = options['level']  || DEFAULT_LEVEL
		@name          = options['name']   || CKApplication.version
		@max_file_size = options['max_file_size']
		@file          = options['file']
		@out           = options['out'] || $stderr
	end

	def add( level, message = nil, &block )
		if message.nil? and block_given? then
			message = yield
		end

		message = _format_message( level, @name, message )

		if @file then
			if _max_file_size? message then
				raise FileSizeError, "#@file: Can't write log. The file size is limit."
			else
				CKFileLock.exclusive_lock(@file, 'a') do |f|
					f.write message
				end
			end
		else
			@out.write message
		end
	end

	private
	def _format_message( level, name, message )
		case level
		when DEBUG then level_s = 'DEBUG'
		when INFO  then level_s = ' INFO'
		when WARN  then level_s = ' WARN'
		when ERROR then level_s = 'ERROR'
		when FATAL then level_s = 'FATAL'
		end

		"[#{Time.now}] #{level_s} - #{name}: #{message}\n"
	end

	def _max_file_size?( message )
		if @max_file_size and FileTest.exist?(@file) then
			@max_file_size <= (File.size(@file) + message.size)
		end
	end

	public
	def close
		@out.close
	end

	def debug( message = nil, &block )
		add( DEBUG, message, &block ) if debug?
	end

	def info( message = nil, &block )
		add( INFO, message, &block ) if info?
	end

	def warn( message = nil, &block )
		add( WARN, message, &block ) if warn?
	end

	def error( message = nil, &block )
		add( ERROR, message, &block ) if error?
	end

	def fatal( message = nil, &block )
		add( FATAL, message, &block ) if fatal?
	end

	def debug?
		@level <= DEBUG
	end

	def info?
		@level <= INFO
	end

	def warn?
		@level <= WARN
	end

	def error?
		@level <= ERROR
	end

	def fatal?
		@level <= FATAL
	end

end


# CKFileLock is for locking files.
class CKFileLock
	# Creates a shared file lock on a file.
	def self.shared_lock( filename, mode = 'r' )
		File.open( filename, mode ) do | io |
			io.flock File::LOCK_SH
			yield io
			io.flock File::LOCK_UN
		end
	end

	# Creates a exclusive file lock on a file.
	def self.exclusive_lock( filename, mode = 'w' )
		File.open( filename, mode ) do | io |
			io.flock File::LOCK_EX
			yield io
			io.flock File::LOCK_UN
		end
	end
end


