/*
    libmaus2
    Copyright (C) 2009-2019 German Tischler
    Copyright (C) 2011-2013 Genome Research Limited

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#if ! defined(LIBMAUS2_SERVERSOCKET_HPP)
#define LIBMAUS2_SERVERSOCKET_HPP

#include <libmaus2/network/SocketBase.hpp>

namespace libmaus2
{
	namespace network
	{
		struct ServerSocket : public SocketBase
		{
			typedef ServerSocket this_type;
			typedef ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;

			sockaddr_in recadr;

			unsigned short getPort() const
			{
				return ntohs(recadr.sin_port);
			}

			static unique_ptr_type allocateServerSocket(
				unsigned short & port,
				unsigned int const backlog,
				std::string const & hostname,
				unsigned int tries)
			{
				unique_ptr_type ptr(allocateServerSocket(port,backlog,hostname.c_str(),tries));
				return ptr;
			}

			static unique_ptr_type allocateServerSocket(
				unsigned short & port,
				unsigned int const backlog,
				char const * hostname,
				unsigned int tries)
			{
				for ( unsigned int i = 0; i < tries; ++i )
				{
					try
					{
						return unique_ptr_type ( new this_type(port,backlog,hostname) );
					}
					catch(std::exception const & ex)
					{
						// std::cerr << ex.what() << std::endl;
						port++;
					}
				}

				::libmaus2::exception::LibMausException ex;
				ex.getStream() << "Failed to allocate ServerSocket (no ports available)";
				ex.finish();
				throw ex;
			}

			ServerSocket(unsigned short rport, unsigned int backlog, char const * hostname)
			: SocketBase()
			{
				memset(&recadr,0,sizeof(recadr));

				setAddress(hostname, recadr);
				recadr.sin_port = htons(rport);

				while ( ::bind ( getFD(), reinterpret_cast<struct sockaddr *>(&recadr), sizeof(recadr) ) != 0 )
				{
					int const error = errno;

					switch ( error )
					{
						case EINTR:
						case EAGAIN:
							break;
						case EADDRINUSE:
							throw std::runtime_error("bind: address is already in use.");
						default:
						{
							::libmaus2::exception::LibMausException se;
							se.getStream() << "ServerSocket: bind() failed with error " << strerror(error) << std::endl;
							se.finish();
							throw se;
						}
					}
				}

				while ( ::listen ( getFD(), backlog ) != 0 )
				{
					int const error = errno;

					switch ( error )
					{
						case EINTR:
						case EAGAIN:
							break;
						default:
						{
							::libmaus2::exception::LibMausException se;
							se.getStream() << "ServerSocket: listen() failed with error " << strerror(error) << std::endl;
							se.finish();
							throw se;
						}
					}
				}
			}

			~ServerSocket()
			{
			}

			SocketBase::unique_ptr_type accept()
			{
				sockaddr_in aadr;

				#if defined(LIBMAUS2_HAVE_SOCKLEN_T)
				socklen_t
				#else
				int
				#endif
					len = sizeof(aadr);

				int accfd;

				while ( (accfd = ::accept(getFD(),reinterpret_cast<struct sockaddr *>(&aadr),&len)) < 0 )
				{
					int const error = errno;

					switch ( error )
					{
						case EINTR:
						case EAGAIN:
							break;
						default:
						{
							::libmaus2::exception::LibMausException se;
							se.getStream() << "ServerSocket::accept: accept() failed with error " << strerror(errno) << std::endl;
							se.finish();
							throw se;
						}
					}
				}

				try
				{
					SocketBase::unique_ptr_type ptr(new SocketBase(accfd,&aadr));
					return ptr;
				}
				catch(...)
				{
					::close(accfd);
					throw;
				}
			}

			SocketBase::shared_ptr_type acceptShared()
			{
				SocketBase::unique_ptr_type uptr(accept());
				SocketBase * ptr = uptr.release();
				SocketBase::shared_ptr_type sptr(ptr);
				return sptr;
			}

			bool waitForConnection(unsigned int const t)
			{
				int r = -1;

				while ( r < 0 )
				{
					fd_set fds;
					FD_ZERO(&fds);
					FD_SET(getFD(),&fds);
					struct timeval timeout = {
						static_cast<long>(t),
						static_cast<long>(0)
					};

					r = ::select(getFD()+1,&fds,0,0,&timeout);

					if ( r < 0 )
					{
						int const error = errno;

						switch ( error )
						{
							case EINTR:
							case EAGAIN:
								break;
							default:
							{
								::libmaus2::exception::LibMausException se;
								se.getStream() << "ServerSocket::waitForConnection: select failed on socket with error " << strerror(error) << std::endl;
								se.finish();
								throw se;
							}
						}
					}
				}

				if ( r == 0 )
				{
					return false;
				}
				else
				{
					assert ( r > 0 );
					return true;
				}
			}
		};
	}
}
#endif
