/*
    libmaus2
    Copyright (C) 2009-2019 German Tischler
    Copyright (C) 2011-2013 Genome Research Limited

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#if ! defined(LIBMAUS2_SOCKETBASE_HPP)
#define LIBMAUS2_SOCKETBASE_HPP

#include <libmaus2/LibMausWindows.hpp>

#if defined(LIBMAUAS2_HAVE_UNISTD_H)
#include <unistd.h>
#endif

#if defined(LIBMAUS2_HAVE_SYS_TYPES_H)
#include <sys/types.h>
#endif

#if defined(LIBMAUS2_HAVE_SYS_STAT_H)
#include <sys/stat.h>
#endif

#if defined(LIBMAUS2_HAVE_SYS_SOCKET_H)
#include <sys/socket.h>
#endif

#if defined(LIBMAUS2_HAVE_NETINET_IN_H)
#include <netinet/in.h>
#endif

#if defined(LIBMAUS2_HAVE_FCNTL_H)
#include <fcntl.h>
#endif

#if defined(LIBMAUS2_HAVE_SYS_IOCTL_H)
#include <sys/ioctl.h>
#endif

#if defined(LIBMAUS2_HAVE_LIBGEN_H)
#include <libgen.h>
#endif

#if defined(LIBMAUS2_HAVE_NETDB_H)
#include <netdb.h>
#endif

#if defined(LIBMAUS2_HAVE_ARPA_INET_H)
#include <arpa/inet.h>
#endif

#if defined(LIBMAUS2_HAVE_NETINET_TCP_H)
#include <netinet/tcp.h>
#endif

#if defined(LIBMAUS2_HAVE_SYS_RESOURCE_H)
#include <sys/resource.h>
#endif

#include <cerrno>
#include <climits>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <set>
#include <stdexcept>

#include <libmaus2/exception/LibMausException.hpp>
#include <libmaus2/network/GetHostName.hpp>
#include <libmaus2/network/SocketInputOutputInterface.hpp>
#include <libmaus2/posix/PosixFunctions.hpp>
#include <libmaus2/util/unique_ptr.hpp>
#include <libmaus2/util/shared_ptr.hpp>

namespace libmaus2
{
	namespace network
	{
		struct SocketBase : public SocketInputOutputInterface
		{
			public:
			typedef SocketBase this_type;
			typedef ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;
			typedef ::libmaus2::util::shared_ptr<this_type>::type shared_ptr_type;

			private:
			int fd;

			protected:
			struct sockaddr_in remaddr;
			bool remaddrset;

			private:
			void cleanup()
			{
				if ( fd != -1 )
				{
					::close(fd);
					fd = -1;
				}
			}

			std::string getStringAddr()
			{
				if ( remaddrset )
				{
					std::ostringstream ostr;
					uint32_t const rem = ntohl(remaddr.sin_addr.s_addr);
					ostr
						<< ((rem >> 24) & 0xFF) << "."
						<< ((rem >> 16) & 0xFF) << "."
						<< ((rem >>  8) & 0xFF) << "."
						<< ((rem >>  0) & 0xFF);
					return ostr.str();
				}
				else
				{
					return "<unknown addr>";
				}
			}

			public:
			void write(char const * data, size_t len)
			{
				while ( len )
				{
					::libmaus2::ssize_t const wr = ::libmaus2::posix::PosixFunctions::write ( fd, data, len );

					if ( wr < 0 )
					{
						int const error = errno;

						switch ( error )
						{
							case EINTR:
							case EAGAIN:
								break;
							default:
							{
								::libmaus2::exception::LibMausException se;
								se.getStream() << "SocketBase::write() to " << getStringAddr() << " failed: " << strerror(errno);
								se.finish();
								throw se;
							}
						}
					}
					else
					{
						data += wr;
						len -= wr;
					}
				}
			}

			::libmaus2::ssize_t read(char * data, size_t len)
			{
				::libmaus2::ssize_t totalred = 0;

				while ( len )
				{
					::libmaus2::ssize_t red = ::libmaus2::posix::PosixFunctions::read(fd,data,len);

					if ( red > 0 )
					{
						totalred += red;
						data += red;
						len -= red;
					}
					else if ( red == 0 )
					{
						len = 0;
					}
					else
					{
						int const error = errno;

						switch ( error )
						{
							case EINTR:
							case EAGAIN:
								break;
							default:
							{
								::libmaus2::exception::LibMausException se;
								se.getStream() << "SocketBase::read() to " << getStringAddr() << " failed: " << strerror(errno);
								se.finish();
								throw se;
							}
						}
					}
				}

				return totalred;
			}

			::libmaus2::ssize_t readPart(char * data, size_t len)
			{
				::libmaus2::ssize_t totalred = 0;

				while ( (! totalred) && len )
				{
					::libmaus2::ssize_t red = ::libmaus2::posix::PosixFunctions::read(fd,data,len);

					if ( red > 0 )
					{
						totalred += red;
						data += red;
						len -= red;
					}
					else if ( red == 0 )
					{
						len = 0;
					}
					else
					{
						int const error = errno;

						switch ( error )
						{
							case EINTR:
							case EAGAIN:
								break;
							default:
							{
								::libmaus2::exception::LibMausException se;
								se.getStream() << "SocketBase::readPart() to " << getStringAddr() << " failed: " << strerror(errno);
								se.finish();
								throw se;
							}
						}
					}
				}

				return totalred;
			}

			void setKeepAlive(
				int const
					#if defined(LIBMAUS2_HAVE_SETSOCKOPT)
					enable
					#endif
					, // boolean
				int const
					#if defined(__linux__) || defined(__APPLE__)
					timeout // time until first check/ping (seconds)
					#endif
					,
				int const
					#if defined(__linux__)
					num
					#endif
					// number of checks
					,
				int const
					#if defined(__linux__)
					intv
					#endif
					// interval between checks/pings (seconds)
			)
			{
				#if defined(LIBMAUS2_HAVE_SETSOCKOPT) && ! defined(_WIN32)
				setsockopt(getFD(), SOL_SOCKET, SO_KEEPALIVE, &enable, sizeof(enable));

				#if defined(__linux__)
				setsockopt(getFD(), IPPROTO_TCP, TCP_KEEPIDLE, &timeout, sizeof(timeout));
				#elif defined(__APPLE__)
				setsockopt(getFD(), IPPROTO_TCP, TCP_KEEPALIVE, &timeout, sizeof(timeout));
				#endif

				#if defined(__linux__)
				setsockopt(getFD(), IPPROTO_TCP, TCP_KEEPCNT, &num, sizeof(num));
				setsockopt(getFD(), IPPROTO_TCP, TCP_KEEPINTVL, &intv, sizeof(intv));
				#endif
				#endif
			}

			protected:
			static void setAddress(char const * hostname, sockaddr_in & recadr)
			{
				if ( hostname )
				{
					#if defined(LIBMAUS2_HAVE_GETHOSTBYNAME2)
					struct hostent * he = gethostbyname2(hostname,AF_INET);
					#elif defined(LIBMAUS2_HAVE_GETHOSTBYNAME)
					struct hostent * he = gethostbyname(hostname);
					#else
					#error "Neither gethostbyname2 nor gethostbyname available"
					#endif

					if ( ! he )
					{
						::libmaus2::exception::LibMausException se;
						#if defined(LIBMAUS2_HAVE_HSTRERROR)
						se.getStream() << "failed to get address for " << hostname << " via gethostbyname: " << hstrerror(h_errno);
						#else
						se.getStream() << "failed to get address for " << hostname << " via gethostbyname: " << h_errno;
						#endif
						se.finish();
						throw se;
					}

					recadr.sin_family = he->h_addrtype;

					if ( he->h_addr_list[0] == 0 )
					{
						::libmaus2::exception::LibMausException se;
						se.getStream() << "failed to get address for " << hostname << " via gethostbyname (no address returned)";
						se.finish();
						throw se;
					}
					else
					{
						if ( recadr.sin_family == AF_INET )
						{
							memcpy ( &recadr.sin_addr.s_addr, he->h_addr_list[0], he->h_length );
						}
						else
						{
							::libmaus2::exception::LibMausException se;
							se.getStream() << "only IPv4 supported";
							se.finish();
							throw se;
						}

						#if 0
						if ( recadr.sin_family == AF_INET )
						{
							std::cerr << inet_ntoa(recadr.sin_addr) << std::endl;
						}
						#endif
					}
				}
				else
				{
					recadr.sin_family = AF_INET;
					recadr.sin_addr.s_addr = INADDR_ANY;
				}
			}

			public:
			SocketBase()
			: fd(-1), remaddrset(false)
			{
				fd = socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);

				if ( fd < 0 )
				{
					cleanup();
					::libmaus2::exception::LibMausException se;
					se.getStream() << "socket() failed: " << strerror(errno);
					se.finish();
					throw se;
				}
			}

			SocketBase(int rfd, sockaddr_in const * aadr = 0)
			: fd(rfd), remaddrset(aadr != 0)
			{
				if ( aadr )
				{
					memcpy ( &remaddr, aadr, sizeof(sockaddr_in) );
				}
			}

			~SocketBase()
			{
				cleanup();
			}

			int getFD() const
			{
				return fd;
			}

			int releaseFD()
			{
				int const rfd = fd;
				fd = -1;
				return rfd;
			}

		};
	}
}
#endif
