#ifndef OSMIUM_GEOM_MERCATOR_PROJECTION_HPP
#define OSMIUM_GEOM_MERCATOR_PROJECTION_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2016 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cmath>
#include <string>

#include <osmium/geom/coordinates.hpp>
#include <osmium/geom/util.hpp>
#include <osmium/osm/location.hpp>

namespace osmium {

    namespace geom {

        namespace detail {

            constexpr double earth_radius_for_epsg3857 = 6378137.0;
            constexpr double max_coordinate_epsg3857 = 20037508.34;

            constexpr inline double lon_to_x(double lon) {
                return earth_radius_for_epsg3857 * deg_to_rad(lon);
            }

            inline double lat_to_y(double lat) { // not constexpr because math functions aren't
                return earth_radius_for_epsg3857 * std::log(std::tan(osmium::geom::PI/4 + deg_to_rad(lat)/2));
            }

            constexpr inline double x_to_lon(double x) {
                return rad_to_deg(x) / earth_radius_for_epsg3857;
            }

            inline double y_to_lat(double y) { // not constexpr because math functions aren't
                return rad_to_deg(2 * std::atan(std::exp(y / earth_radius_for_epsg3857)) - osmium::geom::PI/2);
            }

        } // namespace detail

        /**
         * The maximum latitude that can be projected with the Web Mercator
         * (EPSG:3857) projection.
         */
        constexpr double MERCATOR_MAX_LAT = 85.0511288;

        inline Coordinates lonlat_to_mercator(const Coordinates& c) {
            return Coordinates(detail::lon_to_x(c.x), detail::lat_to_y(c.y));
        }

        inline Coordinates mercator_to_lonlat(const Coordinates& c) {
            return Coordinates(detail::x_to_lon(c.x), detail::y_to_lat(c.y));
        }

        /**
         * Functor that does projection from WGS84 (EPSG:4326) to "Web
         * Mercator" (EPSG:3857)
         */
        class MercatorProjection {

        public:

            Coordinates operator()(osmium::Location location) const {
                return Coordinates {detail::lon_to_x(location.lon()), detail::lat_to_y(location.lat())};
            }

            int epsg() const noexcept {
                return 3857;
            }

            std::string proj_string() const {
                return "+proj=merc +a=6378137 +b=6378137 +lat_ts=0.0 +lon_0=0.0 +x_0=0.0 +y_0=0 +k=1.0 +units=m +nadgrids=@null +wktext +no_defs";
            }

        }; // class MercatorProjection

    } // namespace geom

} // namespace osmium

#endif // OSMIUM_GEOM_MERCATOR_PROJECTION_HPP
