/*
 * pam_sshauth: PAM module for authentication via a remote ssh server.
 * Copyright (C) 2010 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/*
 * I met a traveller from an antique land
 * Who said: `Two vast and trunkless legs of stone
 * Stand in the desert. Near them, on the sand,
 * Half sunk, a shattered visage lies, whose frown,
 * And wrinkled lip, and sneer of cold command,
 * Tell that its sculptor well those passions read
 * Which yet survive, stamped on these lifeless things,
 * The hand that mocked them and the heart that fed.
 * And on the pedestal these words appear --
 * "My name is Ozymandias, king of kings:
 * Look on my works, ye Mighty, and despair!"
 * Nothing beside remains. Round the decay
 * Of that colossal wreck, boundless and bare
 * The lone and level sands stretch far away.'
 *
 * --Percy Bysshe Shelley
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/types.h>
#include <syslog.h>
#include <libssh/libssh.h>

#include <config.h>

/*
 * PAM_SM_* define.
 */

#define PAM_SM_AUTH             /* supports Authentication */

#include <security/pam_modules.h>

/*
 * Our defines
 */

#define HOST "PAM_SSHAUTH_HOST"
#define PORT "PAM_SSHAUTH_PORT"

#define SYSTEM_KNOWNHOSTS "/etc/ssh/ssh_known_hosts"
#define AUTHTRIES 3             /* Three chances to get password right */

/*
 * Globals.
 */

static int debug;               /* Debug flag */
static int nostrict;            /* nostrict flag */
static int authtries;           /* Number of times we'll try to authenticate */
static int try_first_pass;      /* Try to obtain auth token from pam stack */

/*
 * Send a message through pam's conv stack.
 *
 * for PAM_TEXT_INFO, and PAM_ERROR_MSG messages.
 */

static int
send_pam_msg (pam_handle_t * pamh, int style, const char *format, ...)
{
  char msgbuf[BUFSIZ];
  const struct pam_message mymsg =
    {
      .msg_style = style,
      .msg = msgbuf,
    };

  const struct pam_message *msgp = &mymsg;
  const struct pam_conv *pc;
  struct pam_response *resp;
  int pam_result;
  va_list ap;

  /*
   * Start our varargs list, so we can format our message.
   */

  va_start (ap, format);
  if (vsnprintf (msgbuf, sizeof msgbuf, format, ap) >= sizeof msgbuf)
    {
      /*
       * Message was truncated.  Make sure we're NULL terminated,
       * and log an error.
       */

       msgbuf[(sizeof msgbuf) - 1] = '\0';
       pam_syslog (pamh, LOG_ERR, "send_pam_msg () truncated a message.");
    }

  va_end (ap);

  pam_result = pam_get_item (pamh, PAM_CONV, (const void **) &pc);
  if (pam_result != PAM_SUCCESS)
    {
      return pam_result;
    }

  if (!pc || !pc->conv)
    {
      return PAM_CONV_ERR;
    }

  return pc->conv (1, &msgp, &resp, pc->appdata_ptr);
}

/*
 * auth_kbinit ()
 *
 * conduct an ssh login based authentication
 */

static int
auth_kbdint (pam_handle_t * pamh, ssh_session session)
{
  int ssh_result;
  int i, n;
  const char *message;
  const char *prompt;
  char *response;
  char echo;

  ssh_result = ssh_userauth_kbdint (session, NULL, NULL);

  while (ssh_result == SSH_AUTH_INFO)
    {

      /*
       * Get any interactive prompts the ssh session has generted,
       * and send them to the user via the pam message system.
       */

      message = ssh_userauth_kbdint_getinstruction (session);
      n = ssh_userauth_kbdint_getnprompts (session);
      if (message && *message != '\0')
        {
          send_pam_msg (pamh, PAM_TEXT_INFO, message);
        }

      /*
       * Loop through the prompts that ssh has given us, and ask the
       * user via pam prompts for the answers.
       */

      for (i = 0; i < n; ++i)
        {
          prompt = ssh_userauth_kbdint_getprompt (session, i, &echo);

          if (pam_prompt (pamh, echo ? PAM_PROMPT_ECHO_ON : PAM_PROMPT_ECHO_OFF, &response, prompt) != PAM_SUCCESS)
            {
              return SSH_AUTH_ERROR;
            }

          ssh_userauth_kbdint_setanswer (session, i, response);
        }

      ssh_result = ssh_userauth_kbdint (session, NULL, NULL);
    }

  /*
   * The very last response we've gotten should be the password.  Store it
   * as the AUTHTOK for stacking in other pam modules.
   */

  if (ssh_result == SSH_AUTH_SUCCESS)
    {
      if (pam_set_item (pamh, PAM_AUTHTOK, response) != PAM_SUCCESS)
        {
          return SSH_AUTH_ERROR;
        }
    }

  return ssh_result;
}

/*
 * auth_pw ()
 *
 * conduct an ssh simple password based authentication
 */

static int
auth_pw (pam_handle_t * pamh, ssh_session session)
{
  int ssh_result;
  const void *password = NULL;

  /*
   * try_first_pass works with simple password authentication.
   */

  if (try_first_pass)
    {
      if (pam_get_item (pamh, PAM_AUTHTOK, &password) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't obtain PAM_AUTHTOK from the pam stack.");
          password = NULL;
        }
    }
    
  if (password == NULL)
    {
      if (pam_prompt (pamh, PAM_PROMPT_ECHO_OFF, &password, "Password:") != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't obtain password from pam_prompt.");
          return SSH_AUTH_ERROR;
        }
    }

  ssh_result = ssh_userauth_password (session, NULL, password);
  if (ssh_result == SSH_AUTH_SUCCESS)
    {
      /*
       * The very last response we've gotten should be the password.  Store it
       * as the AUTHTOK
       */

      if (!try_first_pass && pam_set_item (pamh, PAM_AUTHTOK, password) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't store password as PAM_AUTHTOK.");
          return SSH_AUTH_ERROR;
        }

      return ssh_result;
    }
  else
    {
      send_pam_msg (pamh, PAM_TEXT_INFO, ssh_get_error (session));
      return ssh_result;
    }
}

/*
 * do_sshauth ()
 *
 * Authenticate by attempting an ssh connection
 */

static int
do_sshauth (pam_handle_t * pamh, const char *username)
{
  ssh_session session;
  int method;
  int ssh_result;
  int pam_result;
  char *response = NULL;
  unsigned char *hash;
  int count;
  const char *host;
  const char *port;

  if (pam_get_data (pamh, HOST, (const void **) &host) != PAM_SUCCESS)
    {
      pam_syslog (pamh, LOG_ERR, "Couldn't retrieve hostname from pam handle.");
      return PAM_SYSTEM_ERR;
    }

  if (pam_get_data (pamh, PORT, (const void **) &port) != PAM_SUCCESS)
    {
      /* Couldn't retrieve port.  Fallback to NULL */
      port = NULL;
    }

  /*
   * Begin the authentication loop.  Loop until we're successfully
   * authenticated, or AUTHTRIES times, whichever comes first.
   */

  count = authtries;

  do
    {
      session = ssh_new ();

      if (!session)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't allocate ssh session structure for host %s", host);
          return PAM_SYSTEM_ERR;
        }

      ssh_options_set (session, SSH_OPTIONS_USER, username);
      ssh_options_set (session, SSH_OPTIONS_HOST, host);
#ifdef SSH_OPTIONS_STRICTHOSTKEYCHECK
      if (nostrict)
        {
          ssh_options_set (session, SSH_OPTIONS_STRICTHOSTKEYCHECK, 0);
        }
      else
        {
          ssh_options_set (session, SSH_OPTIONS_STRICTHOSTKEYCHECK, 1);
        }
#endif

      if (port)
        {
          ssh_options_set (session, SSH_OPTIONS_PORT_STR, port);
        }

      if (ssh_connect (session) != SSH_OK)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't contact host %s", host);
          return PAM_SYSTEM_ERR;
        }

      /*
       * Is this server known to us? Try our /etc/ssh/ssh_known_hosts file
       * first.
       */

      ssh_options_set (session, SSH_OPTIONS_KNOWNHOSTS, SYSTEM_KNOWNHOSTS);
      ssh_result = ssh_is_server_known (session);

      if (ssh_result != SSH_SERVER_KNOWN_OK)
        {
          /*
           * Re-try with the user's ~/.ssh/known_hosts
           */

          ssh_options_set (session, SSH_OPTIONS_KNOWNHOSTS, NULL);
          ssh_result = ssh_is_server_known (session);

          if ((ssh_result == SSH_SERVER_NOT_KNOWN) && nostrict)
            {
              /*
               * Unknown server.  Ask the user, via the pam prompts, if they'd
               * like to connect.
               */
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Server unknown. Trust?");
              ssh_get_pubkey_hash (session, &hash);
              pam_prompt (pamh, PAM_PROMPT_ECHO_ON, &response,
                          "Type 'yes' to continue: ");
              if (strncasecmp (response, "yes", 3) != 0)
                {
                  ssh_disconnect (session);
                  ssh_free (session);
                  return PAM_SYSTEM_ERR;
                }

              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Save in known_hosts?");
              pam_prompt (pamh, PAM_PROMPT_ECHO_ON, &response,
                          "Type 'yes' to continue: ");
              if (strncasecmp (response, "yes", 3) == 0)
                {
                  if (ssh_write_knownhost (session))
                    {
                      ssh_disconnect (session);
                      ssh_free (session);
                      return PAM_SYSTEM_ERR;
                    }
                }
            }
          else if ((ssh_result == SSH_SERVER_NOT_KNOWN) && !nostrict)
            {
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Server is unknown, and strict checking enabled.  Connection denied.");
              ssh_disconnect (session);
              ssh_free (session);
              return PAM_SYSTEM_ERR;
            }
          else if (ssh_result == SSH_SERVER_ERROR)
            {
              pam_syslog (pamh, LOG_ERR,
                          "ssh_is_server_known() returned an error for %s.",
                          host);
              send_pam_msg (pamh, PAM_TEXT_INFO, "Error connecting to server.");
              ssh_disconnect (session);
              ssh_free (session);
              return PAM_SYSTEM_ERR;
            }
          else if (ssh_result == SSH_SERVER_KNOWN_CHANGED)
            {
              pam_syslog (pamh, LOG_ERR, "Host key for %s changed.", host);
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Server's host key has changed.");
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Possible man-in-the-middle attack. Connection terminated.");
              ssh_disconnect (session);
              ssh_free (session);
              return PAM_SYSTEM_ERR;
            }
          else if (ssh_result == SSH_SERVER_FOUND_OTHER)
            {
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Server's host key was different from expected.");
              send_pam_msg (pamh, PAM_TEXT_INFO,
                            "Possible attack. Connection terminated.");
              ssh_disconnect (session);
              ssh_free (session);
              return PAM_SYSTEM_ERR;
            }
        }

      /*
       * Try ssh_userauth none first. This will initiate the authentication
       * system, so we can determine a list of methods the server supports.
       */

      ssh_result = ssh_userauth_none (session, NULL);
      if (ssh_result == SSH_AUTH_ERROR)
        {
          pam_syslog (pamh, LOG_ERR, "ssh_userauth_none() returned %s",
                      ssh_get_error (session));
          ssh_disconnect (session);
          ssh_free (session);
          return PAM_AUTH_ERR;
        }

      /*
       * Find out what methods the ssh server supports for authentication.
       */

      method = ssh_auth_list (session);

      /*
       * List auth methods that have been returned.
       */

      if (method == SSH_AUTH_METHOD_UNKNOWN && debug)
        {
          pam_syslog (pamh, LOG_INFO, "Auth method UNKNOWN.");
        }
      if (method & SSH_AUTH_METHOD_NONE && debug)
        {
          pam_syslog (pamh, LOG_INFO, "Server supports auth method NONE.");
        }
      if (method & SSH_AUTH_METHOD_PASSWORD && debug)
        {
          pam_syslog (pamh, LOG_INFO,
                      "Server supports auth method PASSWORD.");
        }
      if (method & SSH_AUTH_METHOD_HOSTBASED && debug)
        {
          pam_syslog (pamh, LOG_INFO,
                      "Server supports auth method HOSTBASED.");
        }
      if (method & SSH_AUTH_METHOD_INTERACTIVE && debug)
        {
          pam_syslog (pamh, LOG_INFO,
                      "Server supports auth method INTERACTIVE.");
        }
      if (method & SSH_AUTH_METHOD_PUBLICKEY && debug)
        {
          pam_syslog (pamh, LOG_INFO,
                      "Server supports auth method PUBLICKEY.");
        }


      /*
       * Authenticate depending on the method available.
       * Try public key first.
       */

      if (method & SSH_AUTH_METHOD_PUBLICKEY)
        {
          if (debug)
            {
              pam_syslog (pamh, LOG_INFO,
                          "Trying public key authentication.");
            }
          ssh_result = ssh_userauth_autopubkey (session, NULL);
          if (ssh_result == SSH_AUTH_SUCCESS)
            {
              ssh_disconnect (session);
              ssh_free (session);
              break;
            }
        }

      /*
       * Try keyboard interactive next, if supported.
       */

      if (method & SSH_AUTH_METHOD_INTERACTIVE)
        {
          /*
           * SSH_AUTH_METHOD_INTERACTIVE requires
           * ChallengeResponseAuthentication to be set to "yes"
           * in /etc/ssh/sshd_config.
           */
          if (debug)
            {
              pam_syslog (pamh, LOG_INFO,
                          "Trying keyboard interactive authentication.");
            }
          ssh_result = auth_kbdint (pamh, session);
          if (ssh_result == SSH_AUTH_SUCCESS)
            {
              ssh_disconnect (session);
              ssh_free (session);
              break;
            }
        }

      /*
       * Finally, plain password authentication.
       */

      if (method & SSH_AUTH_METHOD_PASSWORD)
        {
          /*
           * SSH_AUTH_METHOD_PASSWORD is simpler, and
           * won't handle password expiry.
           */
          if (debug)
            {
              pam_syslog (pamh, LOG_INFO,
                          "Trying simple password authentication.");
            }
          ssh_result = auth_pw (pamh, session);
          if (ssh_result == SSH_AUTH_SUCCESS)
            {
              ssh_disconnect (session);
              ssh_free (session);
              break;
            }
        }

      count--;

      if (ssh_result == SSH_AUTH_ERROR)
        {
          pam_syslog (pamh, LOG_ERR, "auth_pw returned %s.",
                      ssh_get_error (session));
        }

      ssh_disconnect (session);
      ssh_free (session);
    }
  while (count);

  if (ssh_result == SSH_AUTH_SUCCESS)
    {
      if (debug)
        {
          pam_syslog (pamh, LOG_INFO, "Authenticated successfully.");
        }

      return PAM_SUCCESS;
    }
  else
    {
      if (debug)
        {
          pam_syslog (pamh, LOG_INFO, "Authentication failed.");
        }
      return PAM_AUTH_ERR;
    }
}

/*
 * Authentication function
 */

PAM_EXTERN int
pam_sm_authenticate (pam_handle_t * pamh, int flags, int argc,
                     const char **argv)
{
  const char *username;
  int pam_result;
  char *host;
  char *port;

  authtries = AUTHTRIES;  /* default number of authtries unless passed */
  host = getenv (HOST);
  port = getenv (PORT);

  for (; argc-- > 0; ++argv)
    {
      if (!strncmp (*argv, "debug", 5))
        {
          debug++;
        }

      if (!strncmp (*argv, "authtries", 9))
        {
          authtries = atoi ((char *)(*argv + 10)); /* authtries=x */
        }

      if (!strncmp (*argv, "host", 4))
        {
          host = (char *)(*argv + 5);   /* host=foo.bar.com */
        }

      if (!strncmp (*argv, "port", 4))
        {
          port = (char *)(*argv + 5);   /* port=x */
        }

      if (!strncmp (*argv, "nostrict", 8))
        {
          nostrict++;
        }

      if (!strncmp (*argv, "try_first_pass", 14))
        {
          try_first_pass++;
        }
    }

  /*
   * Process the hostname
   */

  if (host && *host != '\0')
    {
      char host_env[BUFSIZ];

      if (pam_set_data(pamh, HOST, host, NULL) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't store hostname in pam handle.");
          return PAM_SYSTEM_ERR;
        }

      /*
       * Export host as pam environment variable for scripts to use.
       */

      if (snprintf(host_env, sizeof host_env, HOST "=%s", host) >= sizeof host_env)
        {
          pam_syslog (pamh, LOG_ERR, "Hostname truncated due to size.");
          return PAM_SYSTEM_ERR;
        }

      if (pam_putenv(pamh, host_env) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Could not set " HOST " in pam environment.");
          return PAM_SYSTEM_ERR;
        }
    }
  else
    {
      pam_syslog (pamh, LOG_ERR, "No valid hostname specified.");
      return PAM_SYSTEM_ERR;
    }

  /*
   * Process the hostname
   */

  if (port && *port != '\0')
    {
      char port_env[BUFSIZ];

      if (pam_set_data (pamh, PORT, port, NULL) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Couldn't store port in pam handle.");
          return PAM_SYSTEM_ERR;
        }

      /*
       * Export port as pam environment variable for scripts to use.
       */

      if (snprintf (port_env, sizeof port_env, PORT "=%s", port) >= sizeof port_env)
        {
          pam_syslog (pamh, LOG_ERR, "port truncated due to size.");
          return PAM_SYSTEM_ERR;
        }

      if (pam_putenv (pamh, port_env) != PAM_SUCCESS)
        {
          pam_syslog (pamh, LOG_ERR, "Could not set " PORT " in pam environment.");
          return PAM_SYSTEM_ERR;
        }
    }

  if (debug)
    {
      pam_syslog (pamh, LOG_INFO, "Beginning authentication.");
    }

  if (debug)
    {
      if (host)
        {
          pam_syslog (pamh, LOG_INFO, "Host provided on command line: %s", host);
        }

      if (port)
        {
          pam_syslog (pamh, LOG_INFO, "Port provided on command line: %s", port);
        }
    }

  /*
   * Get the username.
   */

  if (pam_get_user (pamh, &username, NULL) != PAM_SUCCESS)
    {
      pam_syslog (pamh, LOG_ERR, "Couldn't determine username.");
      return PAM_AUTHINFO_UNAVAIL;
    }

  /*
   * Perform the authentication.
   */

  pam_result = do_sshauth (pamh, username);

  if (debug)
    {
      pam_syslog (pamh, LOG_INFO, "Authentication finished.");
    }

  return pam_result;
}

PAM_EXTERN int
pam_sm_setcred (pam_handle_t * pamh, int flags, int argc, const char **argv)
{
  return PAM_SUCCESS;
}
