//===----------------------------------------------------------------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is dual licensed under the MIT and the University of Illinois Open
// Source Licenses. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// Copyright 2019, Intel Corporation
//
// Modified to test pmem::obj containers
//

#include "unittest.hpp"

#include <array>
#include <libpmemobj++/experimental/string.hpp>
#include <libpmemobj++/make_persistent.hpp>
#include <libpmemobj++/persistent_ptr.hpp>
#include <libpmemobj++/pool.hpp>
#include <libpmemobj++/transaction.hpp>

namespace pmem_exp = pmem::obj::experimental;
namespace nvobj = pmem::obj;

using S = pmem_exp::string;

struct root {
	nvobj::persistent_ptr<S> s1, s2, s3, s4;
};

int
sign(int x)
{
	if (x == 0)
		return 0;
	if (x < 0)
		return -1;
	return 1;
}

template <class S, class U>
void
test(const S &s, typename S::size_type pos1, typename S::size_type n1,
     const U &str, typename U::size_type pos2, typename U::size_type n2, int x)
{
	if (pos1 <= s.size() && pos2 <= str.size())
		UT_ASSERT(sign(s.compare(pos1, n1, str, pos2, n2)) == sign(x));
	else {
		try {
			s.compare(pos1, n1, str, pos2, n2);
			UT_ASSERT(0);
		} catch (const std::out_of_range &) {
			UT_ASSERT(pos1 > s.size() || pos2 > str.size());
		}
	}
}

template <class S, class U>
void
test_npos(const S &s, typename S::size_type pos1, typename S::size_type n1,
	  const U &str, typename U::size_type pos2, int x)
{
	if (pos1 <= s.size() && pos2 <= str.size())
		UT_ASSERT(sign(s.compare(pos1, n1, str, pos2)) == sign(x));
	else {
		try {
			s.compare(pos1, n1, str, pos2);
			UT_ASSERT(0);
		} catch (const std::out_of_range &) {
			UT_ASSERT(pos1 > s.size() || pos2 > str.size());
		}
	}
}

template <class S, class U>
void
test0(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s1, 0, 0, *arr[0], 0, 0, 0);
	test(*r->s1, 0, 0, *arr[0], 0, 1, 0);
	test(*r->s1, 0, 0, *arr[0], 1, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 0, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 0, 1, -1);
	test(*r->s1, 0, 0, *arr[1], 0, 2, -2);
	test(*r->s1, 0, 0, *arr[1], 0, 4, -4);
	test(*r->s1, 0, 0, *arr[1], 0, 5, -5);
	test(*r->s1, 0, 0, *arr[1], 0, 6, -5);
	test(*r->s1, 0, 0, *arr[1], 1, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 1, 1, -1);
	test(*r->s1, 0, 0, *arr[1], 1, 2, -2);
	test(*r->s1, 0, 0, *arr[1], 1, 3, -3);
	test(*r->s1, 0, 0, *arr[1], 1, 4, -4);
	test(*r->s1, 0, 0, *arr[1], 1, 5, -4);
	test(*r->s1, 0, 0, *arr[1], 2, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 2, 1, -1);
	test(*r->s1, 0, 0, *arr[1], 2, 2, -2);
	test(*r->s1, 0, 0, *arr[1], 2, 3, -3);
	test(*r->s1, 0, 0, *arr[1], 2, 4, -3);
	test(*r->s1, 0, 0, *arr[1], 4, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 4, 1, -1);
	test(*r->s1, 0, 0, *arr[1], 4, 2, -1);
	test(*r->s1, 0, 0, *arr[1], 5, 0, 0);
	test(*r->s1, 0, 0, *arr[1], 5, 1, 0);
	test(*r->s1, 0, 0, *arr[1], 6, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 0, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 0, 1, -1);
	test(*r->s1, 0, 0, *arr[2], 0, 5, -5);
	test(*r->s1, 0, 0, *arr[2], 0, 9, -9);
	test(*r->s1, 0, 0, *arr[2], 0, 10, -10);
	test(*r->s1, 0, 0, *arr[2], 0, 11, -10);
	test(*r->s1, 0, 0, *arr[2], 1, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 1, 1, -1);
	test(*r->s1, 0, 0, *arr[2], 1, 4, -4);
	test(*r->s1, 0, 0, *arr[2], 1, 8, -8);
	test(*r->s1, 0, 0, *arr[2], 1, 9, -9);
	test(*r->s1, 0, 0, *arr[2], 1, 10, -9);
	test(*r->s1, 0, 0, *arr[2], 5, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 5, 1, -1);
	test(*r->s1, 0, 0, *arr[2], 5, 2, -2);
	test(*r->s1, 0, 0, *arr[2], 5, 4, -4);
	test(*r->s1, 0, 0, *arr[2], 5, 5, -5);
	test(*r->s1, 0, 0, *arr[2], 5, 6, -5);
	test(*r->s1, 0, 0, *arr[2], 9, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 9, 1, -1);
	test(*r->s1, 0, 0, *arr[2], 9, 2, -1);
	test(*r->s1, 0, 0, *arr[2], 10, 0, 0);
	test(*r->s1, 0, 0, *arr[2], 10, 1, 0);
	test(*r->s1, 0, 0, *arr[2], 11, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 0, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 0, 1, -1);
	test(*r->s1, 0, 0, *arr[3], 0, 10, -10);
	test(*r->s1, 0, 0, *arr[3], 0, 19, -19);
	test(*r->s1, 0, 0, *arr[3], 0, 20, -20);
	test(*r->s1, 0, 0, *arr[3], 0, 21, -20);
	test(*r->s1, 0, 0, *arr[3], 1, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 1, 1, -1);
	test(*r->s1, 0, 0, *arr[3], 1, 9, -9);
	test(*r->s1, 0, 0, *arr[3], 1, 18, -18);
	test(*r->s1, 0, 0, *arr[3], 1, 19, -19);
	test(*r->s1, 0, 0, *arr[3], 1, 20, -19);
	test(*r->s1, 0, 0, *arr[3], 10, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 10, 1, -1);
	test(*r->s1, 0, 0, *arr[3], 10, 5, -5);
	test(*r->s1, 0, 0, *arr[3], 10, 9, -9);
	test(*r->s1, 0, 0, *arr[3], 10, 10, -10);
	test(*r->s1, 0, 0, *arr[3], 10, 11, -10);
	test(*r->s1, 0, 0, *arr[3], 19, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 19, 1, -1);
	test(*r->s1, 0, 0, *arr[3], 19, 2, -1);
	test(*r->s1, 0, 0, *arr[3], 20, 0, 0);
	test(*r->s1, 0, 0, *arr[3], 20, 1, 0);
	test(*r->s1, 0, 0, *arr[3], 21, 0, 0);
	test(*r->s1, 0, 1, *arr[0], 0, 0, 0);
	test(*r->s1, 0, 1, *arr[0], 0, 1, 0);
	test(*r->s1, 0, 1, *arr[0], 1, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 0, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 0, 1, -1);
	test(*r->s1, 0, 1, *arr[1], 0, 2, -2);
	test(*r->s1, 0, 1, *arr[1], 0, 4, -4);
	test(*r->s1, 0, 1, *arr[1], 0, 5, -5);
	test(*r->s1, 0, 1, *arr[1], 0, 6, -5);
	test(*r->s1, 0, 1, *arr[1], 1, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 1, 1, -1);
	test(*r->s1, 0, 1, *arr[1], 1, 2, -2);
	test(*r->s1, 0, 1, *arr[1], 1, 3, -3);
	test(*r->s1, 0, 1, *arr[1], 1, 4, -4);
	test(*r->s1, 0, 1, *arr[1], 1, 5, -4);
	test(*r->s1, 0, 1, *arr[1], 2, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 2, 1, -1);
	test(*r->s1, 0, 1, *arr[1], 2, 2, -2);
	test(*r->s1, 0, 1, *arr[1], 2, 3, -3);
	test(*r->s1, 0, 1, *arr[1], 2, 4, -3);
	test(*r->s1, 0, 1, *arr[1], 4, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 4, 1, -1);
	test(*r->s1, 0, 1, *arr[1], 4, 2, -1);
	test(*r->s1, 0, 1, *arr[1], 5, 0, 0);
	test(*r->s1, 0, 1, *arr[1], 5, 1, 0);
	test(*r->s1, 0, 1, *arr[1], 6, 0, 0);
}

template <class S, class U>
void
test1(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s1, 0, 1, *arr[2], 0, 0, 0);
	test(*r->s1, 0, 1, *arr[2], 0, 1, -1);
	test(*r->s1, 0, 1, *arr[2], 0, 5, -5);
	test(*r->s1, 0, 1, *arr[2], 0, 9, -9);
	test(*r->s1, 0, 1, *arr[2], 0, 10, -10);
	test(*r->s1, 0, 1, *arr[2], 0, 11, -10);
	test(*r->s1, 0, 1, *arr[2], 1, 0, 0);
	test(*r->s1, 0, 1, *arr[2], 1, 1, -1);
	test(*r->s1, 0, 1, *arr[2], 1, 4, -4);
	test(*r->s1, 0, 1, *arr[2], 1, 8, -8);
	test(*r->s1, 0, 1, *arr[2], 1, 9, -9);
	test(*r->s1, 0, 1, *arr[2], 1, 10, -9);
	test(*r->s1, 0, 1, *arr[2], 5, 0, 0);
	test(*r->s1, 0, 1, *arr[2], 5, 1, -1);
	test(*r->s1, 0, 1, *arr[2], 5, 2, -2);
	test(*r->s1, 0, 1, *arr[2], 5, 4, -4);
	test(*r->s1, 0, 1, *arr[2], 5, 5, -5);
	test(*r->s1, 0, 1, *arr[2], 5, 6, -5);
	test(*r->s1, 0, 1, *arr[2], 9, 0, 0);
	test(*r->s1, 0, 1, *arr[2], 9, 1, -1);
	test(*r->s1, 0, 1, *arr[2], 9, 2, -1);
	test(*r->s1, 0, 1, *arr[2], 10, 0, 0);
	test(*r->s1, 0, 1, *arr[2], 10, 1, 0);
	test(*r->s1, 0, 1, *arr[2], 11, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 0, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 0, 1, -1);
	test(*r->s1, 0, 1, *arr[3], 0, 10, -10);
	test(*r->s1, 0, 1, *arr[3], 0, 19, -19);
	test(*r->s1, 0, 1, *arr[3], 0, 20, -20);
	test(*r->s1, 0, 1, *arr[3], 0, 21, -20);
	test(*r->s1, 0, 1, *arr[3], 1, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 1, 1, -1);
	test(*r->s1, 0, 1, *arr[3], 1, 9, -9);
	test(*r->s1, 0, 1, *arr[3], 1, 18, -18);
	test(*r->s1, 0, 1, *arr[3], 1, 19, -19);
	test(*r->s1, 0, 1, *arr[3], 1, 20, -19);
	test(*r->s1, 0, 1, *arr[3], 10, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 10, 1, -1);
	test(*r->s1, 0, 1, *arr[3], 10, 5, -5);
	test(*r->s1, 0, 1, *arr[3], 10, 9, -9);
	test(*r->s1, 0, 1, *arr[3], 10, 10, -10);
	test(*r->s1, 0, 1, *arr[3], 10, 11, -10);
	test(*r->s1, 0, 1, *arr[3], 19, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 19, 1, -1);
	test(*r->s1, 0, 1, *arr[3], 19, 2, -1);
	test(*r->s1, 0, 1, *arr[3], 20, 0, 0);
	test(*r->s1, 0, 1, *arr[3], 20, 1, 0);
	test(*r->s1, 0, 1, *arr[3], 21, 0, 0);
	test(*r->s1, 1, 0, *arr[0], 0, 0, 0);
	test(*r->s1, 1, 0, *arr[0], 0, 1, 0);
	test(*r->s1, 1, 0, *arr[0], 1, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 1, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 2, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 4, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 5, 0);
	test(*r->s1, 1, 0, *arr[1], 0, 6, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 1, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 2, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 3, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 4, 0);
	test(*r->s1, 1, 0, *arr[1], 1, 5, 0);
	test(*r->s1, 1, 0, *arr[1], 2, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 2, 1, 0);
	test(*r->s1, 1, 0, *arr[1], 2, 2, 0);
	test(*r->s1, 1, 0, *arr[1], 2, 3, 0);
	test(*r->s1, 1, 0, *arr[1], 2, 4, 0);
	test(*r->s1, 1, 0, *arr[1], 4, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 4, 1, 0);
	test(*r->s1, 1, 0, *arr[1], 4, 2, 0);
	test(*r->s1, 1, 0, *arr[1], 5, 0, 0);
	test(*r->s1, 1, 0, *arr[1], 5, 1, 0);
	test(*r->s1, 1, 0, *arr[1], 6, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 1, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 5, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 9, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 10, 0);
	test(*r->s1, 1, 0, *arr[2], 0, 11, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 1, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 4, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 8, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 9, 0);
	test(*r->s1, 1, 0, *arr[2], 1, 10, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 1, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 2, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 4, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 5, 0);
	test(*r->s1, 1, 0, *arr[2], 5, 6, 0);
	test(*r->s1, 1, 0, *arr[2], 9, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 9, 1, 0);
	test(*r->s1, 1, 0, *arr[2], 9, 2, 0);
	test(*r->s1, 1, 0, *arr[2], 10, 0, 0);
	test(*r->s1, 1, 0, *arr[2], 10, 1, 0);
	test(*r->s1, 1, 0, *arr[2], 11, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 0, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 0, 1, 0);
}

template <class S, class U>
void
test2(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s1, 1, 0, *arr[3], 0, 10, 0);
	test(*r->s1, 1, 0, *arr[3], 0, 19, 0);
	test(*r->s1, 1, 0, *arr[3], 0, 20, 0);
	test(*r->s1, 1, 0, *arr[3], 0, 21, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 1, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 9, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 18, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 19, 0);
	test(*r->s1, 1, 0, *arr[3], 1, 20, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 1, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 5, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 9, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 10, 0);
	test(*r->s1, 1, 0, *arr[3], 10, 11, 0);
	test(*r->s1, 1, 0, *arr[3], 19, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 19, 1, 0);
	test(*r->s1, 1, 0, *arr[3], 19, 2, 0);
	test(*r->s1, 1, 0, *arr[3], 20, 0, 0);
	test(*r->s1, 1, 0, *arr[3], 20, 1, 0);
	test(*r->s1, 1, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 0, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 0, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 0, 1, -1);
	test(*r->s2, 0, 0, *arr[1], 0, 2, -2);
	test(*r->s2, 0, 0, *arr[1], 0, 4, -4);
	test(*r->s2, 0, 0, *arr[1], 0, 5, -5);
	test(*r->s2, 0, 0, *arr[1], 0, 6, -5);
	test(*r->s2, 0, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 0, *arr[1], 1, 2, -2);
	test(*r->s2, 0, 0, *arr[1], 1, 3, -3);
	test(*r->s2, 0, 0, *arr[1], 1, 4, -4);
	test(*r->s2, 0, 0, *arr[1], 1, 5, -4);
	test(*r->s2, 0, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 2, 1, -1);
	test(*r->s2, 0, 0, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 0, *arr[1], 2, 3, -3);
	test(*r->s2, 0, 0, *arr[1], 2, 4, -3);
	test(*r->s2, 0, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 4, 1, -1);
	test(*r->s2, 0, 0, *arr[1], 4, 2, -1);
	test(*r->s2, 0, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 0, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 0, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 0, 1, -1);
	test(*r->s2, 0, 0, *arr[2], 0, 5, -5);
	test(*r->s2, 0, 0, *arr[2], 0, 9, -9);
	test(*r->s2, 0, 0, *arr[2], 0, 10, -10);
	test(*r->s2, 0, 0, *arr[2], 0, 11, -10);
	test(*r->s2, 0, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 0, *arr[2], 1, 4, -4);
	test(*r->s2, 0, 0, *arr[2], 1, 8, -8);
	test(*r->s2, 0, 0, *arr[2], 1, 9, -9);
	test(*r->s2, 0, 0, *arr[2], 1, 10, -9);
	test(*r->s2, 0, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 5, 1, -1);
	test(*r->s2, 0, 0, *arr[2], 5, 2, -2);
	test(*r->s2, 0, 0, *arr[2], 5, 4, -4);
	test(*r->s2, 0, 0, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 0, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 9, 1, -1);
	test(*r->s2, 0, 0, *arr[2], 9, 2, -1);
	test(*r->s2, 0, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 0, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 0, 0, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 0, 1, -1);
	test(*r->s2, 0, 0, *arr[3], 0, 10, -10);
	test(*r->s2, 0, 0, *arr[3], 0, 19, -19);
	test(*r->s2, 0, 0, *arr[3], 0, 20, -20);
	test(*r->s2, 0, 0, *arr[3], 0, 21, -20);
	test(*r->s2, 0, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 0, *arr[3], 1, 9, -9);
	test(*r->s2, 0, 0, *arr[3], 1, 18, -18);
	test(*r->s2, 0, 0, *arr[3], 1, 19, -19);
	test(*r->s2, 0, 0, *arr[3], 1, 20, -19);
	test(*r->s2, 0, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 10, 1, -1);
	test(*r->s2, 0, 0, *arr[3], 10, 5, -5);
	test(*r->s2, 0, 0, *arr[3], 10, 9, -9);
	test(*r->s2, 0, 0, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 0, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 19, 1, -1);
	test(*r->s2, 0, 0, *arr[3], 19, 2, -1);
	test(*r->s2, 0, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 0, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 0, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 1, *arr[0], 0, 0, 1);
	test(*r->s2, 0, 1, *arr[0], 0, 1, 1);
	test(*r->s2, 0, 1, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 1, *arr[1], 0, 0, 1);
}

template <class S, class U>
void
test3(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 0, 1, *arr[1], 0, 1, 0);
	test(*r->s2, 0, 1, *arr[1], 0, 2, -1);
	test(*r->s2, 0, 1, *arr[1], 0, 4, -3);
	test(*r->s2, 0, 1, *arr[1], 0, 5, -4);
	test(*r->s2, 0, 1, *arr[1], 0, 6, -4);
	test(*r->s2, 0, 1, *arr[1], 1, 0, 1);
	test(*r->s2, 0, 1, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 1, *arr[1], 1, 2, -1);
	test(*r->s2, 0, 1, *arr[1], 1, 3, -1);
	test(*r->s2, 0, 1, *arr[1], 1, 4, -1);
	test(*r->s2, 0, 1, *arr[1], 1, 5, -1);
	test(*r->s2, 0, 1, *arr[1], 2, 0, 1);
	test(*r->s2, 0, 1, *arr[1], 2, 1, -2);
	test(*r->s2, 0, 1, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 1, *arr[1], 2, 3, -2);
	test(*r->s2, 0, 1, *arr[1], 2, 4, -2);
	test(*r->s2, 0, 1, *arr[1], 4, 0, 1);
	test(*r->s2, 0, 1, *arr[1], 4, 1, -4);
	test(*r->s2, 0, 1, *arr[1], 4, 2, -4);
	test(*r->s2, 0, 1, *arr[1], 5, 0, 1);
	test(*r->s2, 0, 1, *arr[1], 5, 1, 1);
	test(*r->s2, 0, 1, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 1, *arr[2], 0, 0, 1);
	test(*r->s2, 0, 1, *arr[2], 0, 1, 0);
	test(*r->s2, 0, 1, *arr[2], 0, 5, -4);
	test(*r->s2, 0, 1, *arr[2], 0, 9, -8);
	test(*r->s2, 0, 1, *arr[2], 0, 10, -9);
	test(*r->s2, 0, 1, *arr[2], 0, 11, -9);
	test(*r->s2, 0, 1, *arr[2], 1, 0, 1);
	test(*r->s2, 0, 1, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 1, *arr[2], 1, 4, -1);
	test(*r->s2, 0, 1, *arr[2], 1, 8, -1);
	test(*r->s2, 0, 1, *arr[2], 1, 9, -1);
	test(*r->s2, 0, 1, *arr[2], 1, 10, -1);
	test(*r->s2, 0, 1, *arr[2], 5, 0, 1);
	test(*r->s2, 0, 1, *arr[2], 5, 1, -5);
	test(*r->s2, 0, 1, *arr[2], 5, 2, -5);
	test(*r->s2, 0, 1, *arr[2], 5, 4, -5);
	test(*r->s2, 0, 1, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 1, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 1, *arr[2], 9, 0, 1);
	test(*r->s2, 0, 1, *arr[2], 9, 1, -9);
	test(*r->s2, 0, 1, *arr[2], 9, 2, -9);
	test(*r->s2, 0, 1, *arr[2], 10, 0, 1);
	test(*r->s2, 0, 1, *arr[2], 10, 1, 1);
	test(*r->s2, 0, 1, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 1, *arr[3], 0, 0, 1);
	test(*r->s2, 0, 1, *arr[3], 0, 1, 0);
	test(*r->s2, 0, 1, *arr[3], 0, 10, -9);
	test(*r->s2, 0, 1, *arr[3], 0, 19, -18);
	test(*r->s2, 0, 1, *arr[3], 0, 20, -19);
	test(*r->s2, 0, 1, *arr[3], 0, 21, -19);
	test(*r->s2, 0, 1, *arr[3], 1, 0, 1);
	test(*r->s2, 0, 1, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 1, *arr[3], 1, 9, -1);
	test(*r->s2, 0, 1, *arr[3], 1, 18, -1);
	test(*r->s2, 0, 1, *arr[3], 1, 19, -1);
	test(*r->s2, 0, 1, *arr[3], 1, 20, -1);
	test(*r->s2, 0, 1, *arr[3], 10, 0, 1);
	test(*r->s2, 0, 1, *arr[3], 10, 1, -10);
	test(*r->s2, 0, 1, *arr[3], 10, 5, -10);
	test(*r->s2, 0, 1, *arr[3], 10, 9, -10);
	test(*r->s2, 0, 1, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 1, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 1, *arr[3], 19, 0, 1);
	test(*r->s2, 0, 1, *arr[3], 19, 1, -19);
	test(*r->s2, 0, 1, *arr[3], 19, 2, -19);
	test(*r->s2, 0, 1, *arr[3], 20, 0, 1);
	test(*r->s2, 0, 1, *arr[3], 20, 1, 1);
	test(*r->s2, 0, 1, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 2, *arr[0], 0, 0, 2);
	test(*r->s2, 0, 2, *arr[0], 0, 1, 2);
	test(*r->s2, 0, 2, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 2, *arr[1], 0, 0, 2);
	test(*r->s2, 0, 2, *arr[1], 0, 1, 1);
	test(*r->s2, 0, 2, *arr[1], 0, 2, 0);
	test(*r->s2, 0, 2, *arr[1], 0, 4, -2);
	test(*r->s2, 0, 2, *arr[1], 0, 5, -3);
	test(*r->s2, 0, 2, *arr[1], 0, 6, -3);
	test(*r->s2, 0, 2, *arr[1], 1, 0, 2);
	test(*r->s2, 0, 2, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 2, *arr[1], 1, 2, -1);
	test(*r->s2, 0, 2, *arr[1], 1, 3, -1);
	test(*r->s2, 0, 2, *arr[1], 1, 4, -1);
	test(*r->s2, 0, 2, *arr[1], 1, 5, -1);
	test(*r->s2, 0, 2, *arr[1], 2, 0, 2);
	test(*r->s2, 0, 2, *arr[1], 2, 1, -2);
	test(*r->s2, 0, 2, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 2, *arr[1], 2, 3, -2);
	test(*r->s2, 0, 2, *arr[1], 2, 4, -2);
	test(*r->s2, 0, 2, *arr[1], 4, 0, 2);
	test(*r->s2, 0, 2, *arr[1], 4, 1, -4);
	test(*r->s2, 0, 2, *arr[1], 4, 2, -4);
	test(*r->s2, 0, 2, *arr[1], 5, 0, 2);
	test(*r->s2, 0, 2, *arr[1], 5, 1, 2);
	test(*r->s2, 0, 2, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 2, *arr[2], 0, 0, 2);
	test(*r->s2, 0, 2, *arr[2], 0, 1, 1);
	test(*r->s2, 0, 2, *arr[2], 0, 5, -3);
	test(*r->s2, 0, 2, *arr[2], 0, 9, -7);
}

template <class S, class U>
void
test4(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 0, 2, *arr[2], 0, 10, -8);
	test(*r->s2, 0, 2, *arr[2], 0, 11, -8);
	test(*r->s2, 0, 2, *arr[2], 1, 0, 2);
	test(*r->s2, 0, 2, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 2, *arr[2], 1, 4, -1);
	test(*r->s2, 0, 2, *arr[2], 1, 8, -1);
	test(*r->s2, 0, 2, *arr[2], 1, 9, -1);
	test(*r->s2, 0, 2, *arr[2], 1, 10, -1);
	test(*r->s2, 0, 2, *arr[2], 5, 0, 2);
	test(*r->s2, 0, 2, *arr[2], 5, 1, -5);
	test(*r->s2, 0, 2, *arr[2], 5, 2, -5);
	test(*r->s2, 0, 2, *arr[2], 5, 4, -5);
	test(*r->s2, 0, 2, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 2, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 2, *arr[2], 9, 0, 2);
	test(*r->s2, 0, 2, *arr[2], 9, 1, -9);
	test(*r->s2, 0, 2, *arr[2], 9, 2, -9);
	test(*r->s2, 0, 2, *arr[2], 10, 0, 2);
	test(*r->s2, 0, 2, *arr[2], 10, 1, 2);
	test(*r->s2, 0, 2, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 2, *arr[3], 0, 0, 2);
	test(*r->s2, 0, 2, *arr[3], 0, 1, 1);
	test(*r->s2, 0, 2, *arr[3], 0, 10, -8);
	test(*r->s2, 0, 2, *arr[3], 0, 19, -17);
	test(*r->s2, 0, 2, *arr[3], 0, 20, -18);
	test(*r->s2, 0, 2, *arr[3], 0, 21, -18);
	test(*r->s2, 0, 2, *arr[3], 1, 0, 2);
	test(*r->s2, 0, 2, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 2, *arr[3], 1, 9, -1);
	test(*r->s2, 0, 2, *arr[3], 1, 18, -1);
	test(*r->s2, 0, 2, *arr[3], 1, 19, -1);
	test(*r->s2, 0, 2, *arr[3], 1, 20, -1);
	test(*r->s2, 0, 2, *arr[3], 10, 0, 2);
	test(*r->s2, 0, 2, *arr[3], 10, 1, -10);
	test(*r->s2, 0, 2, *arr[3], 10, 5, -10);
	test(*r->s2, 0, 2, *arr[3], 10, 9, -10);
	test(*r->s2, 0, 2, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 2, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 2, *arr[3], 19, 0, 2);
	test(*r->s2, 0, 2, *arr[3], 19, 1, -19);
	test(*r->s2, 0, 2, *arr[3], 19, 2, -19);
	test(*r->s2, 0, 2, *arr[3], 20, 0, 2);
	test(*r->s2, 0, 2, *arr[3], 20, 1, 2);
	test(*r->s2, 0, 2, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 4, *arr[0], 0, 0, 4);
	test(*r->s2, 0, 4, *arr[0], 0, 1, 4);
	test(*r->s2, 0, 4, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 4, *arr[1], 0, 0, 4);
	test(*r->s2, 0, 4, *arr[1], 0, 1, 3);
	test(*r->s2, 0, 4, *arr[1], 0, 2, 2);
	test(*r->s2, 0, 4, *arr[1], 0, 4, 0);
	test(*r->s2, 0, 4, *arr[1], 0, 5, -1);
	test(*r->s2, 0, 4, *arr[1], 0, 6, -1);
	test(*r->s2, 0, 4, *arr[1], 1, 0, 4);
	test(*r->s2, 0, 4, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 4, *arr[1], 1, 2, -1);
	test(*r->s2, 0, 4, *arr[1], 1, 3, -1);
	test(*r->s2, 0, 4, *arr[1], 1, 4, -1);
	test(*r->s2, 0, 4, *arr[1], 1, 5, -1);
	test(*r->s2, 0, 4, *arr[1], 2, 0, 4);
	test(*r->s2, 0, 4, *arr[1], 2, 1, -2);
	test(*r->s2, 0, 4, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 4, *arr[1], 2, 3, -2);
	test(*r->s2, 0, 4, *arr[1], 2, 4, -2);
	test(*r->s2, 0, 4, *arr[1], 4, 0, 4);
	test(*r->s2, 0, 4, *arr[1], 4, 1, -4);
	test(*r->s2, 0, 4, *arr[1], 4, 2, -4);
	test(*r->s2, 0, 4, *arr[1], 5, 0, 4);
	test(*r->s2, 0, 4, *arr[1], 5, 1, 4);
	test(*r->s2, 0, 4, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 4, *arr[2], 0, 0, 4);
	test(*r->s2, 0, 4, *arr[2], 0, 1, 3);
	test(*r->s2, 0, 4, *arr[2], 0, 5, -1);
	test(*r->s2, 0, 4, *arr[2], 0, 9, -5);
	test(*r->s2, 0, 4, *arr[2], 0, 10, -6);
	test(*r->s2, 0, 4, *arr[2], 0, 11, -6);
	test(*r->s2, 0, 4, *arr[2], 1, 0, 4);
	test(*r->s2, 0, 4, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 4, *arr[2], 1, 4, -1);
	test(*r->s2, 0, 4, *arr[2], 1, 8, -1);
	test(*r->s2, 0, 4, *arr[2], 1, 9, -1);
	test(*r->s2, 0, 4, *arr[2], 1, 10, -1);
	test(*r->s2, 0, 4, *arr[2], 5, 0, 4);
	test(*r->s2, 0, 4, *arr[2], 5, 1, -5);
	test(*r->s2, 0, 4, *arr[2], 5, 2, -5);
	test(*r->s2, 0, 4, *arr[2], 5, 4, -5);
	test(*r->s2, 0, 4, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 4, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 4, *arr[2], 9, 0, 4);
	test(*r->s2, 0, 4, *arr[2], 9, 1, -9);
	test(*r->s2, 0, 4, *arr[2], 9, 2, -9);
	test(*r->s2, 0, 4, *arr[2], 10, 0, 4);
	test(*r->s2, 0, 4, *arr[2], 10, 1, 4);
	test(*r->s2, 0, 4, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 4, *arr[3], 0, 0, 4);
	test(*r->s2, 0, 4, *arr[3], 0, 1, 3);
	test(*r->s2, 0, 4, *arr[3], 0, 10, -6);
	test(*r->s2, 0, 4, *arr[3], 0, 19, -15);
	test(*r->s2, 0, 4, *arr[3], 0, 20, -16);
	test(*r->s2, 0, 4, *arr[3], 0, 21, -16);
}

template <class S, class U>
void
test5(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 0, 4, *arr[3], 1, 0, 4);
	test(*r->s2, 0, 4, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 4, *arr[3], 1, 9, -1);
	test(*r->s2, 0, 4, *arr[3], 1, 18, -1);
	test(*r->s2, 0, 4, *arr[3], 1, 19, -1);
	test(*r->s2, 0, 4, *arr[3], 1, 20, -1);
	test(*r->s2, 0, 4, *arr[3], 10, 0, 4);
	test(*r->s2, 0, 4, *arr[3], 10, 1, -10);
	test(*r->s2, 0, 4, *arr[3], 10, 5, -10);
	test(*r->s2, 0, 4, *arr[3], 10, 9, -10);
	test(*r->s2, 0, 4, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 4, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 4, *arr[3], 19, 0, 4);
	test(*r->s2, 0, 4, *arr[3], 19, 1, -19);
	test(*r->s2, 0, 4, *arr[3], 19, 2, -19);
	test(*r->s2, 0, 4, *arr[3], 20, 0, 4);
	test(*r->s2, 0, 4, *arr[3], 20, 1, 4);
	test(*r->s2, 0, 4, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 5, *arr[0], 0, 0, 5);
	test(*r->s2, 0, 5, *arr[0], 0, 1, 5);
	test(*r->s2, 0, 5, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 5, *arr[1], 0, 0, 5);
	test(*r->s2, 0, 5, *arr[1], 0, 1, 4);
	test(*r->s2, 0, 5, *arr[1], 0, 2, 3);
	test(*r->s2, 0, 5, *arr[1], 0, 4, 1);
	test(*r->s2, 0, 5, *arr[1], 0, 5, 0);
	test(*r->s2, 0, 5, *arr[1], 0, 6, 0);
	test(*r->s2, 0, 5, *arr[1], 1, 0, 5);
	test(*r->s2, 0, 5, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 5, *arr[1], 1, 2, -1);
	test(*r->s2, 0, 5, *arr[1], 1, 3, -1);
	test(*r->s2, 0, 5, *arr[1], 1, 4, -1);
	test(*r->s2, 0, 5, *arr[1], 1, 5, -1);
	test(*r->s2, 0, 5, *arr[1], 2, 0, 5);
	test(*r->s2, 0, 5, *arr[1], 2, 1, -2);
	test(*r->s2, 0, 5, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 5, *arr[1], 2, 3, -2);
	test(*r->s2, 0, 5, *arr[1], 2, 4, -2);
	test(*r->s2, 0, 5, *arr[1], 4, 0, 5);
	test(*r->s2, 0, 5, *arr[1], 4, 1, -4);
	test(*r->s2, 0, 5, *arr[1], 4, 2, -4);
	test(*r->s2, 0, 5, *arr[1], 5, 0, 5);
	test(*r->s2, 0, 5, *arr[1], 5, 1, 5);
	test(*r->s2, 0, 5, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 5, *arr[2], 0, 0, 5);
	test(*r->s2, 0, 5, *arr[2], 0, 1, 4);
	test(*r->s2, 0, 5, *arr[2], 0, 5, 0);
	test(*r->s2, 0, 5, *arr[2], 0, 9, -4);
	test(*r->s2, 0, 5, *arr[2], 0, 10, -5);
	test(*r->s2, 0, 5, *arr[2], 0, 11, -5);
	test(*r->s2, 0, 5, *arr[2], 1, 0, 5);
	test(*r->s2, 0, 5, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 5, *arr[2], 1, 4, -1);
	test(*r->s2, 0, 5, *arr[2], 1, 8, -1);
	test(*r->s2, 0, 5, *arr[2], 1, 9, -1);
	test(*r->s2, 0, 5, *arr[2], 1, 10, -1);
	test(*r->s2, 0, 5, *arr[2], 5, 0, 5);
	test(*r->s2, 0, 5, *arr[2], 5, 1, -5);
	test(*r->s2, 0, 5, *arr[2], 5, 2, -5);
	test(*r->s2, 0, 5, *arr[2], 5, 4, -5);
	test(*r->s2, 0, 5, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 5, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 5, *arr[2], 9, 0, 5);
	test(*r->s2, 0, 5, *arr[2], 9, 1, -9);
	test(*r->s2, 0, 5, *arr[2], 9, 2, -9);
	test(*r->s2, 0, 5, *arr[2], 10, 0, 5);
	test(*r->s2, 0, 5, *arr[2], 10, 1, 5);
	test(*r->s2, 0, 5, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 5, *arr[3], 0, 0, 5);
	test(*r->s2, 0, 5, *arr[3], 0, 1, 4);
	test(*r->s2, 0, 5, *arr[3], 0, 10, -5);
	test(*r->s2, 0, 5, *arr[3], 0, 19, -14);
	test(*r->s2, 0, 5, *arr[3], 0, 20, -15);
	test(*r->s2, 0, 5, *arr[3], 0, 21, -15);
	test(*r->s2, 0, 5, *arr[3], 1, 0, 5);
	test(*r->s2, 0, 5, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 5, *arr[3], 1, 9, -1);
	test(*r->s2, 0, 5, *arr[3], 1, 18, -1);
	test(*r->s2, 0, 5, *arr[3], 1, 19, -1);
	test(*r->s2, 0, 5, *arr[3], 1, 20, -1);
	test(*r->s2, 0, 5, *arr[3], 10, 0, 5);
	test(*r->s2, 0, 5, *arr[3], 10, 1, -10);
	test(*r->s2, 0, 5, *arr[3], 10, 5, -10);
	test(*r->s2, 0, 5, *arr[3], 10, 9, -10);
	test(*r->s2, 0, 5, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 5, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 5, *arr[3], 19, 0, 5);
	test(*r->s2, 0, 5, *arr[3], 19, 1, -19);
	test(*r->s2, 0, 5, *arr[3], 19, 2, -19);
	test(*r->s2, 0, 5, *arr[3], 20, 0, 5);
	test(*r->s2, 0, 5, *arr[3], 20, 1, 5);
	test(*r->s2, 0, 5, *arr[3], 21, 0, 0);
	test(*r->s2, 0, 6, *arr[0], 0, 0, 5);
	test(*r->s2, 0, 6, *arr[0], 0, 1, 5);
	test(*r->s2, 0, 6, *arr[0], 1, 0, 0);
	test(*r->s2, 0, 6, *arr[1], 0, 0, 5);
	test(*r->s2, 0, 6, *arr[1], 0, 1, 4);
	test(*r->s2, 0, 6, *arr[1], 0, 2, 3);
	test(*r->s2, 0, 6, *arr[1], 0, 4, 1);
	test(*r->s2, 0, 6, *arr[1], 0, 5, 0);
}

template <class S, class U>
void
test6(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 0, 6, *arr[1], 0, 6, 0);
	test(*r->s2, 0, 6, *arr[1], 1, 0, 5);
	test(*r->s2, 0, 6, *arr[1], 1, 1, -1);
	test(*r->s2, 0, 6, *arr[1], 1, 2, -1);
	test(*r->s2, 0, 6, *arr[1], 1, 3, -1);
	test(*r->s2, 0, 6, *arr[1], 1, 4, -1);
	test(*r->s2, 0, 6, *arr[1], 1, 5, -1);
	test(*r->s2, 0, 6, *arr[1], 2, 0, 5);
	test(*r->s2, 0, 6, *arr[1], 2, 1, -2);
	test(*r->s2, 0, 6, *arr[1], 2, 2, -2);
	test(*r->s2, 0, 6, *arr[1], 2, 3, -2);
	test(*r->s2, 0, 6, *arr[1], 2, 4, -2);
	test(*r->s2, 0, 6, *arr[1], 4, 0, 5);
	test(*r->s2, 0, 6, *arr[1], 4, 1, -4);
	test(*r->s2, 0, 6, *arr[1], 4, 2, -4);
	test(*r->s2, 0, 6, *arr[1], 5, 0, 5);
	test(*r->s2, 0, 6, *arr[1], 5, 1, 5);
	test(*r->s2, 0, 6, *arr[1], 6, 0, 0);
	test(*r->s2, 0, 6, *arr[2], 0, 0, 5);
	test(*r->s2, 0, 6, *arr[2], 0, 1, 4);
	test(*r->s2, 0, 6, *arr[2], 0, 5, 0);
	test(*r->s2, 0, 6, *arr[2], 0, 9, -4);
	test(*r->s2, 0, 6, *arr[2], 0, 10, -5);
	test(*r->s2, 0, 6, *arr[2], 0, 11, -5);
	test(*r->s2, 0, 6, *arr[2], 1, 0, 5);
	test(*r->s2, 0, 6, *arr[2], 1, 1, -1);
	test(*r->s2, 0, 6, *arr[2], 1, 4, -1);
	test(*r->s2, 0, 6, *arr[2], 1, 8, -1);
	test(*r->s2, 0, 6, *arr[2], 1, 9, -1);
	test(*r->s2, 0, 6, *arr[2], 1, 10, -1);
	test(*r->s2, 0, 6, *arr[2], 5, 0, 5);
	test(*r->s2, 0, 6, *arr[2], 5, 1, -5);
	test(*r->s2, 0, 6, *arr[2], 5, 2, -5);
	test(*r->s2, 0, 6, *arr[2], 5, 4, -5);
	test(*r->s2, 0, 6, *arr[2], 5, 5, -5);
	test(*r->s2, 0, 6, *arr[2], 5, 6, -5);
	test(*r->s2, 0, 6, *arr[2], 9, 0, 5);
	test(*r->s2, 0, 6, *arr[2], 9, 1, -9);
	test(*r->s2, 0, 6, *arr[2], 9, 2, -9);
	test(*r->s2, 0, 6, *arr[2], 10, 0, 5);
	test(*r->s2, 0, 6, *arr[2], 10, 1, 5);
	test(*r->s2, 0, 6, *arr[2], 11, 0, 0);
	test(*r->s2, 0, 6, *arr[3], 0, 0, 5);
	test(*r->s2, 0, 6, *arr[3], 0, 1, 4);
	test(*r->s2, 0, 6, *arr[3], 0, 10, -5);
	test(*r->s2, 0, 6, *arr[3], 0, 19, -14);
	test(*r->s2, 0, 6, *arr[3], 0, 20, -15);
	test(*r->s2, 0, 6, *arr[3], 0, 21, -15);
	test(*r->s2, 0, 6, *arr[3], 1, 0, 5);
	test(*r->s2, 0, 6, *arr[3], 1, 1, -1);
	test(*r->s2, 0, 6, *arr[3], 1, 9, -1);
	test(*r->s2, 0, 6, *arr[3], 1, 18, -1);
	test(*r->s2, 0, 6, *arr[3], 1, 19, -1);
	test(*r->s2, 0, 6, *arr[3], 1, 20, -1);
	test(*r->s2, 0, 6, *arr[3], 10, 0, 5);
	test(*r->s2, 0, 6, *arr[3], 10, 1, -10);
	test(*r->s2, 0, 6, *arr[3], 10, 5, -10);
	test(*r->s2, 0, 6, *arr[3], 10, 9, -10);
	test(*r->s2, 0, 6, *arr[3], 10, 10, -10);
	test(*r->s2, 0, 6, *arr[3], 10, 11, -10);
	test(*r->s2, 0, 6, *arr[3], 19, 0, 5);
	test(*r->s2, 0, 6, *arr[3], 19, 1, -19);
	test(*r->s2, 0, 6, *arr[3], 19, 2, -19);
	test(*r->s2, 0, 6, *arr[3], 20, 0, 5);
	test(*r->s2, 0, 6, *arr[3], 20, 1, 5);
	test(*r->s2, 0, 6, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 1, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 1, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 0, 1, -1);
	test(*r->s2, 1, 0, *arr[1], 0, 2, -2);
	test(*r->s2, 1, 0, *arr[1], 0, 4, -4);
	test(*r->s2, 1, 0, *arr[1], 0, 5, -5);
	test(*r->s2, 1, 0, *arr[1], 0, 6, -5);
	test(*r->s2, 1, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 1, 1, -1);
	test(*r->s2, 1, 0, *arr[1], 1, 2, -2);
	test(*r->s2, 1, 0, *arr[1], 1, 3, -3);
	test(*r->s2, 1, 0, *arr[1], 1, 4, -4);
	test(*r->s2, 1, 0, *arr[1], 1, 5, -4);
	test(*r->s2, 1, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 0, *arr[1], 2, 2, -2);
	test(*r->s2, 1, 0, *arr[1], 2, 3, -3);
	test(*r->s2, 1, 0, *arr[1], 2, 4, -3);
	test(*r->s2, 1, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 4, 1, -1);
	test(*r->s2, 1, 0, *arr[1], 4, 2, -1);
	test(*r->s2, 1, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 1, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 1, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 0, 1, -1);
	test(*r->s2, 1, 0, *arr[2], 0, 5, -5);
	test(*r->s2, 1, 0, *arr[2], 0, 9, -9);
	test(*r->s2, 1, 0, *arr[2], 0, 10, -10);
	test(*r->s2, 1, 0, *arr[2], 0, 11, -10);
	test(*r->s2, 1, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 1, 1, -1);
}

template <class S, class U>
void
test7(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 1, 0, *arr[2], 1, 4, -4);
	test(*r->s2, 1, 0, *arr[2], 1, 8, -8);
	test(*r->s2, 1, 0, *arr[2], 1, 9, -9);
	test(*r->s2, 1, 0, *arr[2], 1, 10, -9);
	test(*r->s2, 1, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 5, 1, -1);
	test(*r->s2, 1, 0, *arr[2], 5, 2, -2);
	test(*r->s2, 1, 0, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 0, *arr[2], 5, 5, -5);
	test(*r->s2, 1, 0, *arr[2], 5, 6, -5);
	test(*r->s2, 1, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 9, 1, -1);
	test(*r->s2, 1, 0, *arr[2], 9, 2, -1);
	test(*r->s2, 1, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 1, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 1, 0, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 0, 1, -1);
	test(*r->s2, 1, 0, *arr[3], 0, 10, -10);
	test(*r->s2, 1, 0, *arr[3], 0, 19, -19);
	test(*r->s2, 1, 0, *arr[3], 0, 20, -20);
	test(*r->s2, 1, 0, *arr[3], 0, 21, -20);
	test(*r->s2, 1, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 1, 1, -1);
	test(*r->s2, 1, 0, *arr[3], 1, 9, -9);
	test(*r->s2, 1, 0, *arr[3], 1, 18, -18);
	test(*r->s2, 1, 0, *arr[3], 1, 19, -19);
	test(*r->s2, 1, 0, *arr[3], 1, 20, -19);
	test(*r->s2, 1, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 10, 1, -1);
	test(*r->s2, 1, 0, *arr[3], 10, 5, -5);
	test(*r->s2, 1, 0, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 0, *arr[3], 10, 10, -10);
	test(*r->s2, 1, 0, *arr[3], 10, 11, -10);
	test(*r->s2, 1, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 19, 1, -1);
	test(*r->s2, 1, 0, *arr[3], 19, 2, -1);
	test(*r->s2, 1, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 1, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 1, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 1, *arr[0], 0, 0, 1);
	test(*r->s2, 1, 1, *arr[0], 0, 1, 1);
	test(*r->s2, 1, 1, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 1, *arr[1], 0, 0, 1);
	test(*r->s2, 1, 1, *arr[1], 0, 1, 1);
	test(*r->s2, 1, 1, *arr[1], 0, 2, 1);
	test(*r->s2, 1, 1, *arr[1], 0, 4, 1);
	test(*r->s2, 1, 1, *arr[1], 0, 5, 1);
	test(*r->s2, 1, 1, *arr[1], 0, 6, 1);
	test(*r->s2, 1, 1, *arr[1], 1, 0, 1);
	test(*r->s2, 1, 1, *arr[1], 1, 1, 0);
	test(*r->s2, 1, 1, *arr[1], 1, 2, -1);
	test(*r->s2, 1, 1, *arr[1], 1, 3, -2);
	test(*r->s2, 1, 1, *arr[1], 1, 4, -3);
	test(*r->s2, 1, 1, *arr[1], 1, 5, -3);
	test(*r->s2, 1, 1, *arr[1], 2, 0, 1);
	test(*r->s2, 1, 1, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 1, *arr[1], 2, 2, -1);
	test(*r->s2, 1, 1, *arr[1], 2, 3, -1);
	test(*r->s2, 1, 1, *arr[1], 2, 4, -1);
	test(*r->s2, 1, 1, *arr[1], 4, 0, 1);
	test(*r->s2, 1, 1, *arr[1], 4, 1, -3);
	test(*r->s2, 1, 1, *arr[1], 4, 2, -3);
	test(*r->s2, 1, 1, *arr[1], 5, 0, 1);
	test(*r->s2, 1, 1, *arr[1], 5, 1, 1);
	test(*r->s2, 1, 1, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 1, *arr[2], 0, 0, 1);
	test(*r->s2, 1, 1, *arr[2], 0, 1, 1);
	test(*r->s2, 1, 1, *arr[2], 0, 5, 1);
	test(*r->s2, 1, 1, *arr[2], 0, 9, 1);
	test(*r->s2, 1, 1, *arr[2], 0, 10, 1);
	test(*r->s2, 1, 1, *arr[2], 0, 11, 1);
	test(*r->s2, 1, 1, *arr[2], 1, 0, 1);
	test(*r->s2, 1, 1, *arr[2], 1, 1, 0);
	test(*r->s2, 1, 1, *arr[2], 1, 4, -3);
	test(*r->s2, 1, 1, *arr[2], 1, 8, -7);
	test(*r->s2, 1, 1, *arr[2], 1, 9, -8);
	test(*r->s2, 1, 1, *arr[2], 1, 10, -8);
	test(*r->s2, 1, 1, *arr[2], 5, 0, 1);
	test(*r->s2, 1, 1, *arr[2], 5, 1, -4);
	test(*r->s2, 1, 1, *arr[2], 5, 2, -4);
	test(*r->s2, 1, 1, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 1, *arr[2], 5, 5, -4);
	test(*r->s2, 1, 1, *arr[2], 5, 6, -4);
	test(*r->s2, 1, 1, *arr[2], 9, 0, 1);
	test(*r->s2, 1, 1, *arr[2], 9, 1, -8);
	test(*r->s2, 1, 1, *arr[2], 9, 2, -8);
	test(*r->s2, 1, 1, *arr[2], 10, 0, 1);
	test(*r->s2, 1, 1, *arr[2], 10, 1, 1);
	test(*r->s2, 1, 1, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 1, *arr[3], 0, 0, 1);
	test(*r->s2, 1, 1, *arr[3], 0, 1, 1);
	test(*r->s2, 1, 1, *arr[3], 0, 10, 1);
	test(*r->s2, 1, 1, *arr[3], 0, 19, 1);
	test(*r->s2, 1, 1, *arr[3], 0, 20, 1);
	test(*r->s2, 1, 1, *arr[3], 0, 21, 1);
	test(*r->s2, 1, 1, *arr[3], 1, 0, 1);
	test(*r->s2, 1, 1, *arr[3], 1, 1, 0);
	test(*r->s2, 1, 1, *arr[3], 1, 9, -8);
	test(*r->s2, 1, 1, *arr[3], 1, 18, -17);
}

template <class S, class U>
void
test8(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 1, 1, *arr[3], 1, 19, -18);
	test(*r->s2, 1, 1, *arr[3], 1, 20, -18);
	test(*r->s2, 1, 1, *arr[3], 10, 0, 1);
	test(*r->s2, 1, 1, *arr[3], 10, 1, -9);
	test(*r->s2, 1, 1, *arr[3], 10, 5, -9);
	test(*r->s2, 1, 1, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 1, *arr[3], 10, 10, -9);
	test(*r->s2, 1, 1, *arr[3], 10, 11, -9);
	test(*r->s2, 1, 1, *arr[3], 19, 0, 1);
	test(*r->s2, 1, 1, *arr[3], 19, 1, -18);
	test(*r->s2, 1, 1, *arr[3], 19, 2, -18);
	test(*r->s2, 1, 1, *arr[3], 20, 0, 1);
	test(*r->s2, 1, 1, *arr[3], 20, 1, 1);
	test(*r->s2, 1, 1, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 2, *arr[0], 0, 0, 2);
	test(*r->s2, 1, 2, *arr[0], 0, 1, 2);
	test(*r->s2, 1, 2, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 2, *arr[1], 0, 0, 2);
	test(*r->s2, 1, 2, *arr[1], 0, 1, 1);
	test(*r->s2, 1, 2, *arr[1], 0, 2, 1);
	test(*r->s2, 1, 2, *arr[1], 0, 4, 1);
	test(*r->s2, 1, 2, *arr[1], 0, 5, 1);
	test(*r->s2, 1, 2, *arr[1], 0, 6, 1);
	test(*r->s2, 1, 2, *arr[1], 1, 0, 2);
	test(*r->s2, 1, 2, *arr[1], 1, 1, 1);
	test(*r->s2, 1, 2, *arr[1], 1, 2, 0);
	test(*r->s2, 1, 2, *arr[1], 1, 3, -1);
	test(*r->s2, 1, 2, *arr[1], 1, 4, -2);
	test(*r->s2, 1, 2, *arr[1], 1, 5, -2);
	test(*r->s2, 1, 2, *arr[1], 2, 0, 2);
	test(*r->s2, 1, 2, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 2, *arr[1], 2, 2, -1);
	test(*r->s2, 1, 2, *arr[1], 2, 3, -1);
	test(*r->s2, 1, 2, *arr[1], 2, 4, -1);
	test(*r->s2, 1, 2, *arr[1], 4, 0, 2);
	test(*r->s2, 1, 2, *arr[1], 4, 1, -3);
	test(*r->s2, 1, 2, *arr[1], 4, 2, -3);
	test(*r->s2, 1, 2, *arr[1], 5, 0, 2);
	test(*r->s2, 1, 2, *arr[1], 5, 1, 2);
	test(*r->s2, 1, 2, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 2, *arr[2], 0, 0, 2);
	test(*r->s2, 1, 2, *arr[2], 0, 1, 1);
	test(*r->s2, 1, 2, *arr[2], 0, 5, 1);
	test(*r->s2, 1, 2, *arr[2], 0, 9, 1);
	test(*r->s2, 1, 2, *arr[2], 0, 10, 1);
	test(*r->s2, 1, 2, *arr[2], 0, 11, 1);
	test(*r->s2, 1, 2, *arr[2], 1, 0, 2);
	test(*r->s2, 1, 2, *arr[2], 1, 1, 1);
	test(*r->s2, 1, 2, *arr[2], 1, 4, -2);
	test(*r->s2, 1, 2, *arr[2], 1, 8, -6);
	test(*r->s2, 1, 2, *arr[2], 1, 9, -7);
	test(*r->s2, 1, 2, *arr[2], 1, 10, -7);
	test(*r->s2, 1, 2, *arr[2], 5, 0, 2);
	test(*r->s2, 1, 2, *arr[2], 5, 1, -4);
	test(*r->s2, 1, 2, *arr[2], 5, 2, -4);
	test(*r->s2, 1, 2, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 2, *arr[2], 5, 5, -4);
	test(*r->s2, 1, 2, *arr[2], 5, 6, -4);
	test(*r->s2, 1, 2, *arr[2], 9, 0, 2);
	test(*r->s2, 1, 2, *arr[2], 9, 1, -8);
	test(*r->s2, 1, 2, *arr[2], 9, 2, -8);
	test(*r->s2, 1, 2, *arr[2], 10, 0, 2);
	test(*r->s2, 1, 2, *arr[2], 10, 1, 2);
	test(*r->s2, 1, 2, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 2, *arr[3], 0, 0, 2);
	test(*r->s2, 1, 2, *arr[3], 0, 1, 1);
	test(*r->s2, 1, 2, *arr[3], 0, 10, 1);
	test(*r->s2, 1, 2, *arr[3], 0, 19, 1);
	test(*r->s2, 1, 2, *arr[3], 0, 20, 1);
	test(*r->s2, 1, 2, *arr[3], 0, 21, 1);
	test(*r->s2, 1, 2, *arr[3], 1, 0, 2);
	test(*r->s2, 1, 2, *arr[3], 1, 1, 1);
	test(*r->s2, 1, 2, *arr[3], 1, 9, -7);
	test(*r->s2, 1, 2, *arr[3], 1, 18, -16);
	test(*r->s2, 1, 2, *arr[3], 1, 19, -17);
	test(*r->s2, 1, 2, *arr[3], 1, 20, -17);
	test(*r->s2, 1, 2, *arr[3], 10, 0, 2);
	test(*r->s2, 1, 2, *arr[3], 10, 1, -9);
	test(*r->s2, 1, 2, *arr[3], 10, 5, -9);
	test(*r->s2, 1, 2, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 2, *arr[3], 10, 10, -9);
	test(*r->s2, 1, 2, *arr[3], 10, 11, -9);
	test(*r->s2, 1, 2, *arr[3], 19, 0, 2);
	test(*r->s2, 1, 2, *arr[3], 19, 1, -18);
	test(*r->s2, 1, 2, *arr[3], 19, 2, -18);
	test(*r->s2, 1, 2, *arr[3], 20, 0, 2);
	test(*r->s2, 1, 2, *arr[3], 20, 1, 2);
	test(*r->s2, 1, 2, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 3, *arr[0], 0, 0, 3);
	test(*r->s2, 1, 3, *arr[0], 0, 1, 3);
	test(*r->s2, 1, 3, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 3, *arr[1], 0, 0, 3);
	test(*r->s2, 1, 3, *arr[1], 0, 1, 1);
	test(*r->s2, 1, 3, *arr[1], 0, 2, 1);
	test(*r->s2, 1, 3, *arr[1], 0, 4, 1);
	test(*r->s2, 1, 3, *arr[1], 0, 5, 1);
	test(*r->s2, 1, 3, *arr[1], 0, 6, 1);
	test(*r->s2, 1, 3, *arr[1], 1, 0, 3);
	test(*r->s2, 1, 3, *arr[1], 1, 1, 2);
	test(*r->s2, 1, 3, *arr[1], 1, 2, 1);
}

template <class S, class U>
void
test9(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 1, 3, *arr[1], 1, 3, 0);
	test(*r->s2, 1, 3, *arr[1], 1, 4, -1);
	test(*r->s2, 1, 3, *arr[1], 1, 5, -1);
	test(*r->s2, 1, 3, *arr[1], 2, 0, 3);
	test(*r->s2, 1, 3, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 3, *arr[1], 2, 2, -1);
	test(*r->s2, 1, 3, *arr[1], 2, 3, -1);
	test(*r->s2, 1, 3, *arr[1], 2, 4, -1);
	test(*r->s2, 1, 3, *arr[1], 4, 0, 3);
	test(*r->s2, 1, 3, *arr[1], 4, 1, -3);
	test(*r->s2, 1, 3, *arr[1], 4, 2, -3);
	test(*r->s2, 1, 3, *arr[1], 5, 0, 3);
	test(*r->s2, 1, 3, *arr[1], 5, 1, 3);
	test(*r->s2, 1, 3, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 3, *arr[2], 0, 0, 3);
	test(*r->s2, 1, 3, *arr[2], 0, 1, 1);
	test(*r->s2, 1, 3, *arr[2], 0, 5, 1);
	test(*r->s2, 1, 3, *arr[2], 0, 9, 1);
	test(*r->s2, 1, 3, *arr[2], 0, 10, 1);
	test(*r->s2, 1, 3, *arr[2], 0, 11, 1);
	test(*r->s2, 1, 3, *arr[2], 1, 0, 3);
	test(*r->s2, 1, 3, *arr[2], 1, 1, 2);
	test(*r->s2, 1, 3, *arr[2], 1, 4, -1);
	test(*r->s2, 1, 3, *arr[2], 1, 8, -5);
	test(*r->s2, 1, 3, *arr[2], 1, 9, -6);
	test(*r->s2, 1, 3, *arr[2], 1, 10, -6);
	test(*r->s2, 1, 3, *arr[2], 5, 0, 3);
	test(*r->s2, 1, 3, *arr[2], 5, 1, -4);
	test(*r->s2, 1, 3, *arr[2], 5, 2, -4);
	test(*r->s2, 1, 3, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 3, *arr[2], 5, 5, -4);
	test(*r->s2, 1, 3, *arr[2], 5, 6, -4);
	test(*r->s2, 1, 3, *arr[2], 9, 0, 3);
	test(*r->s2, 1, 3, *arr[2], 9, 1, -8);
	test(*r->s2, 1, 3, *arr[2], 9, 2, -8);
	test(*r->s2, 1, 3, *arr[2], 10, 0, 3);
	test(*r->s2, 1, 3, *arr[2], 10, 1, 3);
	test(*r->s2, 1, 3, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 3, *arr[3], 0, 0, 3);
	test(*r->s2, 1, 3, *arr[3], 0, 1, 1);
	test(*r->s2, 1, 3, *arr[3], 0, 10, 1);
	test(*r->s2, 1, 3, *arr[3], 0, 19, 1);
	test(*r->s2, 1, 3, *arr[3], 0, 20, 1);
	test(*r->s2, 1, 3, *arr[3], 0, 21, 1);
	test(*r->s2, 1, 3, *arr[3], 1, 0, 3);
	test(*r->s2, 1, 3, *arr[3], 1, 1, 2);
	test(*r->s2, 1, 3, *arr[3], 1, 9, -6);
	test(*r->s2, 1, 3, *arr[3], 1, 18, -15);
	test(*r->s2, 1, 3, *arr[3], 1, 19, -16);
	test(*r->s2, 1, 3, *arr[3], 1, 20, -16);
	test(*r->s2, 1, 3, *arr[3], 10, 0, 3);
	test(*r->s2, 1, 3, *arr[3], 10, 1, -9);
	test(*r->s2, 1, 3, *arr[3], 10, 5, -9);
	test(*r->s2, 1, 3, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 3, *arr[3], 10, 10, -9);
	test(*r->s2, 1, 3, *arr[3], 10, 11, -9);
	test(*r->s2, 1, 3, *arr[3], 19, 0, 3);
	test(*r->s2, 1, 3, *arr[3], 19, 1, -18);
	test(*r->s2, 1, 3, *arr[3], 19, 2, -18);
	test(*r->s2, 1, 3, *arr[3], 20, 0, 3);
	test(*r->s2, 1, 3, *arr[3], 20, 1, 3);
	test(*r->s2, 1, 3, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 4, *arr[0], 0, 0, 4);
	test(*r->s2, 1, 4, *arr[0], 0, 1, 4);
	test(*r->s2, 1, 4, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 4, *arr[1], 0, 0, 4);
	test(*r->s2, 1, 4, *arr[1], 0, 1, 1);
	test(*r->s2, 1, 4, *arr[1], 0, 2, 1);
	test(*r->s2, 1, 4, *arr[1], 0, 4, 1);
	test(*r->s2, 1, 4, *arr[1], 0, 5, 1);
	test(*r->s2, 1, 4, *arr[1], 0, 6, 1);
	test(*r->s2, 1, 4, *arr[1], 1, 0, 4);
	test(*r->s2, 1, 4, *arr[1], 1, 1, 3);
	test(*r->s2, 1, 4, *arr[1], 1, 2, 2);
	test(*r->s2, 1, 4, *arr[1], 1, 3, 1);
	test(*r->s2, 1, 4, *arr[1], 1, 4, 0);
	test(*r->s2, 1, 4, *arr[1], 1, 5, 0);
	test(*r->s2, 1, 4, *arr[1], 2, 0, 4);
	test(*r->s2, 1, 4, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 4, *arr[1], 2, 2, -1);
	test(*r->s2, 1, 4, *arr[1], 2, 3, -1);
	test(*r->s2, 1, 4, *arr[1], 2, 4, -1);
	test(*r->s2, 1, 4, *arr[1], 4, 0, 4);
	test(*r->s2, 1, 4, *arr[1], 4, 1, -3);
	test(*r->s2, 1, 4, *arr[1], 4, 2, -3);
	test(*r->s2, 1, 4, *arr[1], 5, 0, 4);
	test(*r->s2, 1, 4, *arr[1], 5, 1, 4);
	test(*r->s2, 1, 4, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 4, *arr[2], 0, 0, 4);
	test(*r->s2, 1, 4, *arr[2], 0, 1, 1);
	test(*r->s2, 1, 4, *arr[2], 0, 5, 1);
	test(*r->s2, 1, 4, *arr[2], 0, 9, 1);
	test(*r->s2, 1, 4, *arr[2], 0, 10, 1);
	test(*r->s2, 1, 4, *arr[2], 0, 11, 1);
	test(*r->s2, 1, 4, *arr[2], 1, 0, 4);
	test(*r->s2, 1, 4, *arr[2], 1, 1, 3);
	test(*r->s2, 1, 4, *arr[2], 1, 4, 0);
	test(*r->s2, 1, 4, *arr[2], 1, 8, -4);
	test(*r->s2, 1, 4, *arr[2], 1, 9, -5);
	test(*r->s2, 1, 4, *arr[2], 1, 10, -5);
}

template <class S, class U>
void
test10(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 1, 4, *arr[2], 5, 0, 4);
	test(*r->s2, 1, 4, *arr[2], 5, 1, -4);
	test(*r->s2, 1, 4, *arr[2], 5, 2, -4);
	test(*r->s2, 1, 4, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 4, *arr[2], 5, 5, -4);
	test(*r->s2, 1, 4, *arr[2], 5, 6, -4);
	test(*r->s2, 1, 4, *arr[2], 9, 0, 4);
	test(*r->s2, 1, 4, *arr[2], 9, 1, -8);
	test(*r->s2, 1, 4, *arr[2], 9, 2, -8);
	test(*r->s2, 1, 4, *arr[2], 10, 0, 4);
	test(*r->s2, 1, 4, *arr[2], 10, 1, 4);
	test(*r->s2, 1, 4, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 4, *arr[3], 0, 0, 4);
	test(*r->s2, 1, 4, *arr[3], 0, 1, 1);
	test(*r->s2, 1, 4, *arr[3], 0, 10, 1);
	test(*r->s2, 1, 4, *arr[3], 0, 19, 1);
	test(*r->s2, 1, 4, *arr[3], 0, 20, 1);
	test(*r->s2, 1, 4, *arr[3], 0, 21, 1);
	test(*r->s2, 1, 4, *arr[3], 1, 0, 4);
	test(*r->s2, 1, 4, *arr[3], 1, 1, 3);
	test(*r->s2, 1, 4, *arr[3], 1, 9, -5);
	test(*r->s2, 1, 4, *arr[3], 1, 18, -14);
	test(*r->s2, 1, 4, *arr[3], 1, 19, -15);
	test(*r->s2, 1, 4, *arr[3], 1, 20, -15);
	test(*r->s2, 1, 4, *arr[3], 10, 0, 4);
	test(*r->s2, 1, 4, *arr[3], 10, 1, -9);
	test(*r->s2, 1, 4, *arr[3], 10, 5, -9);
	test(*r->s2, 1, 4, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 4, *arr[3], 10, 10, -9);
	test(*r->s2, 1, 4, *arr[3], 10, 11, -9);
	test(*r->s2, 1, 4, *arr[3], 19, 0, 4);
	test(*r->s2, 1, 4, *arr[3], 19, 1, -18);
	test(*r->s2, 1, 4, *arr[3], 19, 2, -18);
	test(*r->s2, 1, 4, *arr[3], 20, 0, 4);
	test(*r->s2, 1, 4, *arr[3], 20, 1, 4);
	test(*r->s2, 1, 4, *arr[3], 21, 0, 0);
	test(*r->s2, 1, 5, *arr[0], 0, 0, 4);
	test(*r->s2, 1, 5, *arr[0], 0, 1, 4);
	test(*r->s2, 1, 5, *arr[0], 1, 0, 0);
	test(*r->s2, 1, 5, *arr[1], 0, 0, 4);
	test(*r->s2, 1, 5, *arr[1], 0, 1, 1);
	test(*r->s2, 1, 5, *arr[1], 0, 2, 1);
	test(*r->s2, 1, 5, *arr[1], 0, 4, 1);
	test(*r->s2, 1, 5, *arr[1], 0, 5, 1);
	test(*r->s2, 1, 5, *arr[1], 0, 6, 1);
	test(*r->s2, 1, 5, *arr[1], 1, 0, 4);
	test(*r->s2, 1, 5, *arr[1], 1, 1, 3);
	test(*r->s2, 1, 5, *arr[1], 1, 2, 2);
	test(*r->s2, 1, 5, *arr[1], 1, 3, 1);
	test(*r->s2, 1, 5, *arr[1], 1, 4, 0);
	test(*r->s2, 1, 5, *arr[1], 1, 5, 0);
	test(*r->s2, 1, 5, *arr[1], 2, 0, 4);
	test(*r->s2, 1, 5, *arr[1], 2, 1, -1);
	test(*r->s2, 1, 5, *arr[1], 2, 2, -1);
	test(*r->s2, 1, 5, *arr[1], 2, 3, -1);
	test(*r->s2, 1, 5, *arr[1], 2, 4, -1);
	test(*r->s2, 1, 5, *arr[1], 4, 0, 4);
	test(*r->s2, 1, 5, *arr[1], 4, 1, -3);
	test(*r->s2, 1, 5, *arr[1], 4, 2, -3);
	test(*r->s2, 1, 5, *arr[1], 5, 0, 4);
	test(*r->s2, 1, 5, *arr[1], 5, 1, 4);
	test(*r->s2, 1, 5, *arr[1], 6, 0, 0);
	test(*r->s2, 1, 5, *arr[2], 0, 0, 4);
	test(*r->s2, 1, 5, *arr[2], 0, 1, 1);
	test(*r->s2, 1, 5, *arr[2], 0, 5, 1);
	test(*r->s2, 1, 5, *arr[2], 0, 9, 1);
	test(*r->s2, 1, 5, *arr[2], 0, 10, 1);
	test(*r->s2, 1, 5, *arr[2], 0, 11, 1);
	test(*r->s2, 1, 5, *arr[2], 1, 0, 4);
	test(*r->s2, 1, 5, *arr[2], 1, 1, 3);
	test(*r->s2, 1, 5, *arr[2], 1, 4, 0);
	test(*r->s2, 1, 5, *arr[2], 1, 8, -4);
	test(*r->s2, 1, 5, *arr[2], 1, 9, -5);
	test(*r->s2, 1, 5, *arr[2], 1, 10, -5);
	test(*r->s2, 1, 5, *arr[2], 5, 0, 4);
	test(*r->s2, 1, 5, *arr[2], 5, 1, -4);
	test(*r->s2, 1, 5, *arr[2], 5, 2, -4);
	test(*r->s2, 1, 5, *arr[2], 5, 4, -4);
	test(*r->s2, 1, 5, *arr[2], 5, 5, -4);
	test(*r->s2, 1, 5, *arr[2], 5, 6, -4);
	test(*r->s2, 1, 5, *arr[2], 9, 0, 4);
	test(*r->s2, 1, 5, *arr[2], 9, 1, -8);
	test(*r->s2, 1, 5, *arr[2], 9, 2, -8);
	test(*r->s2, 1, 5, *arr[2], 10, 0, 4);
	test(*r->s2, 1, 5, *arr[2], 10, 1, 4);
	test(*r->s2, 1, 5, *arr[2], 11, 0, 0);
	test(*r->s2, 1, 5, *arr[3], 0, 0, 4);
	test(*r->s2, 1, 5, *arr[3], 0, 1, 1);
	test(*r->s2, 1, 5, *arr[3], 0, 10, 1);
	test(*r->s2, 1, 5, *arr[3], 0, 19, 1);
	test(*r->s2, 1, 5, *arr[3], 0, 20, 1);
	test(*r->s2, 1, 5, *arr[3], 0, 21, 1);
	test(*r->s2, 1, 5, *arr[3], 1, 0, 4);
	test(*r->s2, 1, 5, *arr[3], 1, 1, 3);
	test(*r->s2, 1, 5, *arr[3], 1, 9, -5);
	test(*r->s2, 1, 5, *arr[3], 1, 18, -14);
	test(*r->s2, 1, 5, *arr[3], 1, 19, -15);
	test(*r->s2, 1, 5, *arr[3], 1, 20, -15);
	test(*r->s2, 1, 5, *arr[3], 10, 0, 4);
	test(*r->s2, 1, 5, *arr[3], 10, 1, -9);
}

template <class S, class U>
void
test11(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 1, 5, *arr[3], 10, 5, -9);
	test(*r->s2, 1, 5, *arr[3], 10, 9, -9);
	test(*r->s2, 1, 5, *arr[3], 10, 10, -9);
	test(*r->s2, 1, 5, *arr[3], 10, 11, -9);
	test(*r->s2, 1, 5, *arr[3], 19, 0, 4);
	test(*r->s2, 1, 5, *arr[3], 19, 1, -18);
	test(*r->s2, 1, 5, *arr[3], 19, 2, -18);
	test(*r->s2, 1, 5, *arr[3], 20, 0, 4);
	test(*r->s2, 1, 5, *arr[3], 20, 1, 4);
	test(*r->s2, 1, 5, *arr[3], 21, 0, 0);
	test(*r->s2, 2, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 2, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 2, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 0, 1, -1);
	test(*r->s2, 2, 0, *arr[1], 0, 2, -2);
	test(*r->s2, 2, 0, *arr[1], 0, 4, -4);
	test(*r->s2, 2, 0, *arr[1], 0, 5, -5);
	test(*r->s2, 2, 0, *arr[1], 0, 6, -5);
	test(*r->s2, 2, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 1, 1, -1);
	test(*r->s2, 2, 0, *arr[1], 1, 2, -2);
	test(*r->s2, 2, 0, *arr[1], 1, 3, -3);
	test(*r->s2, 2, 0, *arr[1], 1, 4, -4);
	test(*r->s2, 2, 0, *arr[1], 1, 5, -4);
	test(*r->s2, 2, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 2, 1, -1);
	test(*r->s2, 2, 0, *arr[1], 2, 2, -2);
	test(*r->s2, 2, 0, *arr[1], 2, 3, -3);
	test(*r->s2, 2, 0, *arr[1], 2, 4, -3);
	test(*r->s2, 2, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 4, 1, -1);
	test(*r->s2, 2, 0, *arr[1], 4, 2, -1);
	test(*r->s2, 2, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 2, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 2, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 0, 1, -1);
	test(*r->s2, 2, 0, *arr[2], 0, 5, -5);
	test(*r->s2, 2, 0, *arr[2], 0, 9, -9);
	test(*r->s2, 2, 0, *arr[2], 0, 10, -10);
	test(*r->s2, 2, 0, *arr[2], 0, 11, -10);
	test(*r->s2, 2, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 1, 1, -1);
	test(*r->s2, 2, 0, *arr[2], 1, 4, -4);
	test(*r->s2, 2, 0, *arr[2], 1, 8, -8);
	test(*r->s2, 2, 0, *arr[2], 1, 9, -9);
	test(*r->s2, 2, 0, *arr[2], 1, 10, -9);
	test(*r->s2, 2, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 5, 1, -1);
	test(*r->s2, 2, 0, *arr[2], 5, 2, -2);
	test(*r->s2, 2, 0, *arr[2], 5, 4, -4);
	test(*r->s2, 2, 0, *arr[2], 5, 5, -5);
	test(*r->s2, 2, 0, *arr[2], 5, 6, -5);
	test(*r->s2, 2, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 9, 1, -1);
	test(*r->s2, 2, 0, *arr[2], 9, 2, -1);
	test(*r->s2, 2, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 2, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 2, 0, *arr[2], 11, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 0, 1, -1);
	test(*r->s2, 2, 0, *arr[3], 0, 10, -10);
	test(*r->s2, 2, 0, *arr[3], 0, 19, -19);
	test(*r->s2, 2, 0, *arr[3], 0, 20, -20);
	test(*r->s2, 2, 0, *arr[3], 0, 21, -20);
	test(*r->s2, 2, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 1, 1, -1);
	test(*r->s2, 2, 0, *arr[3], 1, 9, -9);
	test(*r->s2, 2, 0, *arr[3], 1, 18, -18);
	test(*r->s2, 2, 0, *arr[3], 1, 19, -19);
	test(*r->s2, 2, 0, *arr[3], 1, 20, -19);
	test(*r->s2, 2, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 10, 1, -1);
	test(*r->s2, 2, 0, *arr[3], 10, 5, -5);
	test(*r->s2, 2, 0, *arr[3], 10, 9, -9);
	test(*r->s2, 2, 0, *arr[3], 10, 10, -10);
	test(*r->s2, 2, 0, *arr[3], 10, 11, -10);
	test(*r->s2, 2, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 19, 1, -1);
	test(*r->s2, 2, 0, *arr[3], 19, 2, -1);
	test(*r->s2, 2, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 2, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 2, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 2, 1, *arr[0], 0, 0, 1);
	test(*r->s2, 2, 1, *arr[0], 0, 1, 1);
	test(*r->s2, 2, 1, *arr[0], 1, 0, 0);
	test(*r->s2, 2, 1, *arr[1], 0, 0, 1);
	test(*r->s2, 2, 1, *arr[1], 0, 1, 2);
	test(*r->s2, 2, 1, *arr[1], 0, 2, 2);
	test(*r->s2, 2, 1, *arr[1], 0, 4, 2);
	test(*r->s2, 2, 1, *arr[1], 0, 5, 2);
	test(*r->s2, 2, 1, *arr[1], 0, 6, 2);
	test(*r->s2, 2, 1, *arr[1], 1, 0, 1);
	test(*r->s2, 2, 1, *arr[1], 1, 1, 1);
	test(*r->s2, 2, 1, *arr[1], 1, 2, 1);
	test(*r->s2, 2, 1, *arr[1], 1, 3, 1);
	test(*r->s2, 2, 1, *arr[1], 1, 4, 1);
	test(*r->s2, 2, 1, *arr[1], 1, 5, 1);
	test(*r->s2, 2, 1, *arr[1], 2, 0, 1);
}

template <class S, class U>
void
test12(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 2, 1, *arr[1], 2, 1, 0);
	test(*r->s2, 2, 1, *arr[1], 2, 2, -1);
	test(*r->s2, 2, 1, *arr[1], 2, 3, -2);
	test(*r->s2, 2, 1, *arr[1], 2, 4, -2);
	test(*r->s2, 2, 1, *arr[1], 4, 0, 1);
	test(*r->s2, 2, 1, *arr[1], 4, 1, -2);
	test(*r->s2, 2, 1, *arr[1], 4, 2, -2);
	test(*r->s2, 2, 1, *arr[1], 5, 0, 1);
	test(*r->s2, 2, 1, *arr[1], 5, 1, 1);
	test(*r->s2, 2, 1, *arr[1], 6, 0, 0);
	test(*r->s2, 2, 1, *arr[2], 0, 0, 1);
	test(*r->s2, 2, 1, *arr[2], 0, 1, 2);
	test(*r->s2, 2, 1, *arr[2], 0, 5, 2);
	test(*r->s2, 2, 1, *arr[2], 0, 9, 2);
	test(*r->s2, 2, 1, *arr[2], 0, 10, 2);
	test(*r->s2, 2, 1, *arr[2], 0, 11, 2);
	test(*r->s2, 2, 1, *arr[2], 1, 0, 1);
	test(*r->s2, 2, 1, *arr[2], 1, 1, 1);
	test(*r->s2, 2, 1, *arr[2], 1, 4, 1);
	test(*r->s2, 2, 1, *arr[2], 1, 8, 1);
	test(*r->s2, 2, 1, *arr[2], 1, 9, 1);
	test(*r->s2, 2, 1, *arr[2], 1, 10, 1);
	test(*r->s2, 2, 1, *arr[2], 5, 0, 1);
	test(*r->s2, 2, 1, *arr[2], 5, 1, -3);
	test(*r->s2, 2, 1, *arr[2], 5, 2, -3);
	test(*r->s2, 2, 1, *arr[2], 5, 4, -3);
	test(*r->s2, 2, 1, *arr[2], 5, 5, -3);
	test(*r->s2, 2, 1, *arr[2], 5, 6, -3);
	test(*r->s2, 2, 1, *arr[2], 9, 0, 1);
	test(*r->s2, 2, 1, *arr[2], 9, 1, -7);
	test(*r->s2, 2, 1, *arr[2], 9, 2, -7);
	test(*r->s2, 2, 1, *arr[2], 10, 0, 1);
	test(*r->s2, 2, 1, *arr[2], 10, 1, 1);
	test(*r->s2, 2, 1, *arr[2], 11, 0, 0);
	test(*r->s2, 2, 1, *arr[3], 0, 0, 1);
	test(*r->s2, 2, 1, *arr[3], 0, 1, 2);
	test(*r->s2, 2, 1, *arr[3], 0, 10, 2);
	test(*r->s2, 2, 1, *arr[3], 0, 19, 2);
	test(*r->s2, 2, 1, *arr[3], 0, 20, 2);
	test(*r->s2, 2, 1, *arr[3], 0, 21, 2);
	test(*r->s2, 2, 1, *arr[3], 1, 0, 1);
	test(*r->s2, 2, 1, *arr[3], 1, 1, 1);
	test(*r->s2, 2, 1, *arr[3], 1, 9, 1);
	test(*r->s2, 2, 1, *arr[3], 1, 18, 1);
	test(*r->s2, 2, 1, *arr[3], 1, 19, 1);
	test(*r->s2, 2, 1, *arr[3], 1, 20, 1);
	test(*r->s2, 2, 1, *arr[3], 10, 0, 1);
	test(*r->s2, 2, 1, *arr[3], 10, 1, -8);
	test(*r->s2, 2, 1, *arr[3], 10, 5, -8);
	test(*r->s2, 2, 1, *arr[3], 10, 9, -8);
	test(*r->s2, 2, 1, *arr[3], 10, 10, -8);
	test(*r->s2, 2, 1, *arr[3], 10, 11, -8);
	test(*r->s2, 2, 1, *arr[3], 19, 0, 1);
	test(*r->s2, 2, 1, *arr[3], 19, 1, -17);
	test(*r->s2, 2, 1, *arr[3], 19, 2, -17);
	test(*r->s2, 2, 1, *arr[3], 20, 0, 1);
	test(*r->s2, 2, 1, *arr[3], 20, 1, 1);
	test(*r->s2, 2, 1, *arr[3], 21, 0, 0);
	test(*r->s2, 2, 2, *arr[0], 0, 0, 2);
	test(*r->s2, 2, 2, *arr[0], 0, 1, 2);
	test(*r->s2, 2, 2, *arr[0], 1, 0, 0);
	test(*r->s2, 2, 2, *arr[1], 0, 0, 2);
	test(*r->s2, 2, 2, *arr[1], 0, 1, 2);
	test(*r->s2, 2, 2, *arr[1], 0, 2, 2);
	test(*r->s2, 2, 2, *arr[1], 0, 4, 2);
	test(*r->s2, 2, 2, *arr[1], 0, 5, 2);
	test(*r->s2, 2, 2, *arr[1], 0, 6, 2);
	test(*r->s2, 2, 2, *arr[1], 1, 0, 2);
	test(*r->s2, 2, 2, *arr[1], 1, 1, 1);
	test(*r->s2, 2, 2, *arr[1], 1, 2, 1);
	test(*r->s2, 2, 2, *arr[1], 1, 3, 1);
	test(*r->s2, 2, 2, *arr[1], 1, 4, 1);
	test(*r->s2, 2, 2, *arr[1], 1, 5, 1);
	test(*r->s2, 2, 2, *arr[1], 2, 0, 2);
	test(*r->s2, 2, 2, *arr[1], 2, 1, 1);
	test(*r->s2, 2, 2, *arr[1], 2, 2, 0);
	test(*r->s2, 2, 2, *arr[1], 2, 3, -1);
	test(*r->s2, 2, 2, *arr[1], 2, 4, -1);
	test(*r->s2, 2, 2, *arr[1], 4, 0, 2);
	test(*r->s2, 2, 2, *arr[1], 4, 1, -2);
	test(*r->s2, 2, 2, *arr[1], 4, 2, -2);
	test(*r->s2, 2, 2, *arr[1], 5, 0, 2);
	test(*r->s2, 2, 2, *arr[1], 5, 1, 2);
	test(*r->s2, 2, 2, *arr[1], 6, 0, 0);
	test(*r->s2, 2, 2, *arr[2], 0, 0, 2);
	test(*r->s2, 2, 2, *arr[2], 0, 1, 2);
	test(*r->s2, 2, 2, *arr[2], 0, 5, 2);
	test(*r->s2, 2, 2, *arr[2], 0, 9, 2);
	test(*r->s2, 2, 2, *arr[2], 0, 10, 2);
	test(*r->s2, 2, 2, *arr[2], 0, 11, 2);
	test(*r->s2, 2, 2, *arr[2], 1, 0, 2);
	test(*r->s2, 2, 2, *arr[2], 1, 1, 1);
	test(*r->s2, 2, 2, *arr[2], 1, 4, 1);
	test(*r->s2, 2, 2, *arr[2], 1, 8, 1);
	test(*r->s2, 2, 2, *arr[2], 1, 9, 1);
	test(*r->s2, 2, 2, *arr[2], 1, 10, 1);
	test(*r->s2, 2, 2, *arr[2], 5, 0, 2);
	test(*r->s2, 2, 2, *arr[2], 5, 1, -3);
	test(*r->s2, 2, 2, *arr[2], 5, 2, -3);
	test(*r->s2, 2, 2, *arr[2], 5, 4, -3);
}

template <class S, class U>
void
test13(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 2, 2, *arr[2], 5, 5, -3);
	test(*r->s2, 2, 2, *arr[2], 5, 6, -3);
	test(*r->s2, 2, 2, *arr[2], 9, 0, 2);
	test(*r->s2, 2, 2, *arr[2], 9, 1, -7);
	test(*r->s2, 2, 2, *arr[2], 9, 2, -7);
	test(*r->s2, 2, 2, *arr[2], 10, 0, 2);
	test(*r->s2, 2, 2, *arr[2], 10, 1, 2);
	test(*r->s2, 2, 2, *arr[2], 11, 0, 0);
	test(*r->s2, 2, 2, *arr[3], 0, 0, 2);
	test(*r->s2, 2, 2, *arr[3], 0, 1, 2);
	test(*r->s2, 2, 2, *arr[3], 0, 10, 2);
	test(*r->s2, 2, 2, *arr[3], 0, 19, 2);
	test(*r->s2, 2, 2, *arr[3], 0, 20, 2);
	test(*r->s2, 2, 2, *arr[3], 0, 21, 2);
	test(*r->s2, 2, 2, *arr[3], 1, 0, 2);
	test(*r->s2, 2, 2, *arr[3], 1, 1, 1);
	test(*r->s2, 2, 2, *arr[3], 1, 9, 1);
	test(*r->s2, 2, 2, *arr[3], 1, 18, 1);
	test(*r->s2, 2, 2, *arr[3], 1, 19, 1);
	test(*r->s2, 2, 2, *arr[3], 1, 20, 1);
	test(*r->s2, 2, 2, *arr[3], 10, 0, 2);
	test(*r->s2, 2, 2, *arr[3], 10, 1, -8);
	test(*r->s2, 2, 2, *arr[3], 10, 5, -8);
	test(*r->s2, 2, 2, *arr[3], 10, 9, -8);
	test(*r->s2, 2, 2, *arr[3], 10, 10, -8);
	test(*r->s2, 2, 2, *arr[3], 10, 11, -8);
	test(*r->s2, 2, 2, *arr[3], 19, 0, 2);
	test(*r->s2, 2, 2, *arr[3], 19, 1, -17);
	test(*r->s2, 2, 2, *arr[3], 19, 2, -17);
	test(*r->s2, 2, 2, *arr[3], 20, 0, 2);
	test(*r->s2, 2, 2, *arr[3], 20, 1, 2);
	test(*r->s2, 2, 2, *arr[3], 21, 0, 0);
	test(*r->s2, 2, 3, *arr[0], 0, 0, 3);
	test(*r->s2, 2, 3, *arr[0], 0, 1, 3);
	test(*r->s2, 2, 3, *arr[0], 1, 0, 0);
	test(*r->s2, 2, 3, *arr[1], 0, 0, 3);
	test(*r->s2, 2, 3, *arr[1], 0, 1, 2);
	test(*r->s2, 2, 3, *arr[1], 0, 2, 2);
	test(*r->s2, 2, 3, *arr[1], 0, 4, 2);
	test(*r->s2, 2, 3, *arr[1], 0, 5, 2);
	test(*r->s2, 2, 3, *arr[1], 0, 6, 2);
	test(*r->s2, 2, 3, *arr[1], 1, 0, 3);
	test(*r->s2, 2, 3, *arr[1], 1, 1, 1);
	test(*r->s2, 2, 3, *arr[1], 1, 2, 1);
	test(*r->s2, 2, 3, *arr[1], 1, 3, 1);
	test(*r->s2, 2, 3, *arr[1], 1, 4, 1);
	test(*r->s2, 2, 3, *arr[1], 1, 5, 1);
	test(*r->s2, 2, 3, *arr[1], 2, 0, 3);
	test(*r->s2, 2, 3, *arr[1], 2, 1, 2);
	test(*r->s2, 2, 3, *arr[1], 2, 2, 1);
	test(*r->s2, 2, 3, *arr[1], 2, 3, 0);
	test(*r->s2, 2, 3, *arr[1], 2, 4, 0);
	test(*r->s2, 2, 3, *arr[1], 4, 0, 3);
	test(*r->s2, 2, 3, *arr[1], 4, 1, -2);
	test(*r->s2, 2, 3, *arr[1], 4, 2, -2);
	test(*r->s2, 2, 3, *arr[1], 5, 0, 3);
	test(*r->s2, 2, 3, *arr[1], 5, 1, 3);
	test(*r->s2, 2, 3, *arr[1], 6, 0, 0);
	test(*r->s2, 2, 3, *arr[2], 0, 0, 3);
	test(*r->s2, 2, 3, *arr[2], 0, 1, 2);
	test(*r->s2, 2, 3, *arr[2], 0, 5, 2);
	test(*r->s2, 2, 3, *arr[2], 0, 9, 2);
	test(*r->s2, 2, 3, *arr[2], 0, 10, 2);
	test(*r->s2, 2, 3, *arr[2], 0, 11, 2);
	test(*r->s2, 2, 3, *arr[2], 1, 0, 3);
	test(*r->s2, 2, 3, *arr[2], 1, 1, 1);
	test(*r->s2, 2, 3, *arr[2], 1, 4, 1);
	test(*r->s2, 2, 3, *arr[2], 1, 8, 1);
	test(*r->s2, 2, 3, *arr[2], 1, 9, 1);
	test(*r->s2, 2, 3, *arr[2], 1, 10, 1);
	test(*r->s2, 2, 3, *arr[2], 5, 0, 3);
	test(*r->s2, 2, 3, *arr[2], 5, 1, -3);
	test(*r->s2, 2, 3, *arr[2], 5, 2, -3);
	test(*r->s2, 2, 3, *arr[2], 5, 4, -3);
	test(*r->s2, 2, 3, *arr[2], 5, 5, -3);
	test(*r->s2, 2, 3, *arr[2], 5, 6, -3);
	test(*r->s2, 2, 3, *arr[2], 9, 0, 3);
	test(*r->s2, 2, 3, *arr[2], 9, 1, -7);
	test(*r->s2, 2, 3, *arr[2], 9, 2, -7);
	test(*r->s2, 2, 3, *arr[2], 10, 0, 3);
	test(*r->s2, 2, 3, *arr[2], 10, 1, 3);
	test(*r->s2, 2, 3, *arr[2], 11, 0, 0);
	test(*r->s2, 2, 3, *arr[3], 0, 0, 3);
	test(*r->s2, 2, 3, *arr[3], 0, 1, 2);
	test(*r->s2, 2, 3, *arr[3], 0, 10, 2);
	test(*r->s2, 2, 3, *arr[3], 0, 19, 2);
	test(*r->s2, 2, 3, *arr[3], 0, 20, 2);
	test(*r->s2, 2, 3, *arr[3], 0, 21, 2);
	test(*r->s2, 2, 3, *arr[3], 1, 0, 3);
	test(*r->s2, 2, 3, *arr[3], 1, 1, 1);
	test(*r->s2, 2, 3, *arr[3], 1, 9, 1);
	test(*r->s2, 2, 3, *arr[3], 1, 18, 1);
	test(*r->s2, 2, 3, *arr[3], 1, 19, 1);
	test(*r->s2, 2, 3, *arr[3], 1, 20, 1);
	test(*r->s2, 2, 3, *arr[3], 10, 0, 3);
	test(*r->s2, 2, 3, *arr[3], 10, 1, -8);
	test(*r->s2, 2, 3, *arr[3], 10, 5, -8);
	test(*r->s2, 2, 3, *arr[3], 10, 9, -8);
	test(*r->s2, 2, 3, *arr[3], 10, 10, -8);
	test(*r->s2, 2, 3, *arr[3], 10, 11, -8);
}

template <class S, class U>
void
test14(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 2, 3, *arr[3], 19, 0, 3);
	test(*r->s2, 2, 3, *arr[3], 19, 1, -17);
	test(*r->s2, 2, 3, *arr[3], 19, 2, -17);
	test(*r->s2, 2, 3, *arr[3], 20, 0, 3);
	test(*r->s2, 2, 3, *arr[3], 20, 1, 3);
	test(*r->s2, 2, 3, *arr[3], 21, 0, 0);
	test(*r->s2, 2, 4, *arr[0], 0, 0, 3);
	test(*r->s2, 2, 4, *arr[0], 0, 1, 3);
	test(*r->s2, 2, 4, *arr[0], 1, 0, 0);
	test(*r->s2, 2, 4, *arr[1], 0, 0, 3);
	test(*r->s2, 2, 4, *arr[1], 0, 1, 2);
	test(*r->s2, 2, 4, *arr[1], 0, 2, 2);
	test(*r->s2, 2, 4, *arr[1], 0, 4, 2);
	test(*r->s2, 2, 4, *arr[1], 0, 5, 2);
	test(*r->s2, 2, 4, *arr[1], 0, 6, 2);
	test(*r->s2, 2, 4, *arr[1], 1, 0, 3);
	test(*r->s2, 2, 4, *arr[1], 1, 1, 1);
	test(*r->s2, 2, 4, *arr[1], 1, 2, 1);
	test(*r->s2, 2, 4, *arr[1], 1, 3, 1);
	test(*r->s2, 2, 4, *arr[1], 1, 4, 1);
	test(*r->s2, 2, 4, *arr[1], 1, 5, 1);
	test(*r->s2, 2, 4, *arr[1], 2, 0, 3);
	test(*r->s2, 2, 4, *arr[1], 2, 1, 2);
	test(*r->s2, 2, 4, *arr[1], 2, 2, 1);
	test(*r->s2, 2, 4, *arr[1], 2, 3, 0);
	test(*r->s2, 2, 4, *arr[1], 2, 4, 0);
	test(*r->s2, 2, 4, *arr[1], 4, 0, 3);
	test(*r->s2, 2, 4, *arr[1], 4, 1, -2);
	test(*r->s2, 2, 4, *arr[1], 4, 2, -2);
	test(*r->s2, 2, 4, *arr[1], 5, 0, 3);
	test(*r->s2, 2, 4, *arr[1], 5, 1, 3);
	test(*r->s2, 2, 4, *arr[1], 6, 0, 0);
	test(*r->s2, 2, 4, *arr[2], 0, 0, 3);
	test(*r->s2, 2, 4, *arr[2], 0, 1, 2);
	test(*r->s2, 2, 4, *arr[2], 0, 5, 2);
	test(*r->s2, 2, 4, *arr[2], 0, 9, 2);
	test(*r->s2, 2, 4, *arr[2], 0, 10, 2);
	test(*r->s2, 2, 4, *arr[2], 0, 11, 2);
	test(*r->s2, 2, 4, *arr[2], 1, 0, 3);
	test(*r->s2, 2, 4, *arr[2], 1, 1, 1);
	test(*r->s2, 2, 4, *arr[2], 1, 4, 1);
	test(*r->s2, 2, 4, *arr[2], 1, 8, 1);
	test(*r->s2, 2, 4, *arr[2], 1, 9, 1);
	test(*r->s2, 2, 4, *arr[2], 1, 10, 1);
	test(*r->s2, 2, 4, *arr[2], 5, 0, 3);
	test(*r->s2, 2, 4, *arr[2], 5, 1, -3);
	test(*r->s2, 2, 4, *arr[2], 5, 2, -3);
	test(*r->s2, 2, 4, *arr[2], 5, 4, -3);
	test(*r->s2, 2, 4, *arr[2], 5, 5, -3);
	test(*r->s2, 2, 4, *arr[2], 5, 6, -3);
	test(*r->s2, 2, 4, *arr[2], 9, 0, 3);
	test(*r->s2, 2, 4, *arr[2], 9, 1, -7);
	test(*r->s2, 2, 4, *arr[2], 9, 2, -7);
	test(*r->s2, 2, 4, *arr[2], 10, 0, 3);
	test(*r->s2, 2, 4, *arr[2], 10, 1, 3);
	test(*r->s2, 2, 4, *arr[2], 11, 0, 0);
	test(*r->s2, 2, 4, *arr[3], 0, 0, 3);
	test(*r->s2, 2, 4, *arr[3], 0, 1, 2);
	test(*r->s2, 2, 4, *arr[3], 0, 10, 2);
	test(*r->s2, 2, 4, *arr[3], 0, 19, 2);
	test(*r->s2, 2, 4, *arr[3], 0, 20, 2);
	test(*r->s2, 2, 4, *arr[3], 0, 21, 2);
	test(*r->s2, 2, 4, *arr[3], 1, 0, 3);
	test(*r->s2, 2, 4, *arr[3], 1, 1, 1);
	test(*r->s2, 2, 4, *arr[3], 1, 9, 1);
	test(*r->s2, 2, 4, *arr[3], 1, 18, 1);
	test(*r->s2, 2, 4, *arr[3], 1, 19, 1);
	test(*r->s2, 2, 4, *arr[3], 1, 20, 1);
	test(*r->s2, 2, 4, *arr[3], 10, 0, 3);
	test(*r->s2, 2, 4, *arr[3], 10, 1, -8);
	test(*r->s2, 2, 4, *arr[3], 10, 5, -8);
	test(*r->s2, 2, 4, *arr[3], 10, 9, -8);
	test(*r->s2, 2, 4, *arr[3], 10, 10, -8);
	test(*r->s2, 2, 4, *arr[3], 10, 11, -8);
	test(*r->s2, 2, 4, *arr[3], 19, 0, 3);
	test(*r->s2, 2, 4, *arr[3], 19, 1, -17);
	test(*r->s2, 2, 4, *arr[3], 19, 2, -17);
	test(*r->s2, 2, 4, *arr[3], 20, 0, 3);
	test(*r->s2, 2, 4, *arr[3], 20, 1, 3);
	test(*r->s2, 2, 4, *arr[3], 21, 0, 0);
	test(*r->s2, 4, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 4, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 4, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 0, 1, -1);
	test(*r->s2, 4, 0, *arr[1], 0, 2, -2);
	test(*r->s2, 4, 0, *arr[1], 0, 4, -4);
	test(*r->s2, 4, 0, *arr[1], 0, 5, -5);
	test(*r->s2, 4, 0, *arr[1], 0, 6, -5);
	test(*r->s2, 4, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 1, 1, -1);
	test(*r->s2, 4, 0, *arr[1], 1, 2, -2);
	test(*r->s2, 4, 0, *arr[1], 1, 3, -3);
	test(*r->s2, 4, 0, *arr[1], 1, 4, -4);
	test(*r->s2, 4, 0, *arr[1], 1, 5, -4);
	test(*r->s2, 4, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 2, 1, -1);
	test(*r->s2, 4, 0, *arr[1], 2, 2, -2);
	test(*r->s2, 4, 0, *arr[1], 2, 3, -3);
	test(*r->s2, 4, 0, *arr[1], 2, 4, -3);
}

template <class S, class U>
void
test15(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 4, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 4, 1, -1);
	test(*r->s2, 4, 0, *arr[1], 4, 2, -1);
	test(*r->s2, 4, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 4, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 4, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 0, 1, -1);
	test(*r->s2, 4, 0, *arr[2], 0, 5, -5);
	test(*r->s2, 4, 0, *arr[2], 0, 9, -9);
	test(*r->s2, 4, 0, *arr[2], 0, 10, -10);
	test(*r->s2, 4, 0, *arr[2], 0, 11, -10);
	test(*r->s2, 4, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 1, 1, -1);
	test(*r->s2, 4, 0, *arr[2], 1, 4, -4);
	test(*r->s2, 4, 0, *arr[2], 1, 8, -8);
	test(*r->s2, 4, 0, *arr[2], 1, 9, -9);
	test(*r->s2, 4, 0, *arr[2], 1, 10, -9);
	test(*r->s2, 4, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 5, 1, -1);
	test(*r->s2, 4, 0, *arr[2], 5, 2, -2);
	test(*r->s2, 4, 0, *arr[2], 5, 4, -4);
	test(*r->s2, 4, 0, *arr[2], 5, 5, -5);
	test(*r->s2, 4, 0, *arr[2], 5, 6, -5);
	test(*r->s2, 4, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 9, 1, -1);
	test(*r->s2, 4, 0, *arr[2], 9, 2, -1);
	test(*r->s2, 4, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 4, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 4, 0, *arr[2], 11, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 0, 1, -1);
	test(*r->s2, 4, 0, *arr[3], 0, 10, -10);
	test(*r->s2, 4, 0, *arr[3], 0, 19, -19);
	test(*r->s2, 4, 0, *arr[3], 0, 20, -20);
	test(*r->s2, 4, 0, *arr[3], 0, 21, -20);
	test(*r->s2, 4, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 1, 1, -1);
	test(*r->s2, 4, 0, *arr[3], 1, 9, -9);
	test(*r->s2, 4, 0, *arr[3], 1, 18, -18);
	test(*r->s2, 4, 0, *arr[3], 1, 19, -19);
	test(*r->s2, 4, 0, *arr[3], 1, 20, -19);
	test(*r->s2, 4, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 10, 1, -1);
	test(*r->s2, 4, 0, *arr[3], 10, 5, -5);
	test(*r->s2, 4, 0, *arr[3], 10, 9, -9);
	test(*r->s2, 4, 0, *arr[3], 10, 10, -10);
	test(*r->s2, 4, 0, *arr[3], 10, 11, -10);
	test(*r->s2, 4, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 19, 1, -1);
	test(*r->s2, 4, 0, *arr[3], 19, 2, -1);
	test(*r->s2, 4, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 4, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 4, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 4, 1, *arr[0], 0, 0, 1);
	test(*r->s2, 4, 1, *arr[0], 0, 1, 1);
	test(*r->s2, 4, 1, *arr[0], 1, 0, 0);
	test(*r->s2, 4, 1, *arr[1], 0, 0, 1);
	test(*r->s2, 4, 1, *arr[1], 0, 1, 4);
	test(*r->s2, 4, 1, *arr[1], 0, 2, 4);
	test(*r->s2, 4, 1, *arr[1], 0, 4, 4);
	test(*r->s2, 4, 1, *arr[1], 0, 5, 4);
	test(*r->s2, 4, 1, *arr[1], 0, 6, 4);
	test(*r->s2, 4, 1, *arr[1], 1, 0, 1);
	test(*r->s2, 4, 1, *arr[1], 1, 1, 3);
	test(*r->s2, 4, 1, *arr[1], 1, 2, 3);
	test(*r->s2, 4, 1, *arr[1], 1, 3, 3);
	test(*r->s2, 4, 1, *arr[1], 1, 4, 3);
	test(*r->s2, 4, 1, *arr[1], 1, 5, 3);
	test(*r->s2, 4, 1, *arr[1], 2, 0, 1);
	test(*r->s2, 4, 1, *arr[1], 2, 1, 2);
	test(*r->s2, 4, 1, *arr[1], 2, 2, 2);
	test(*r->s2, 4, 1, *arr[1], 2, 3, 2);
	test(*r->s2, 4, 1, *arr[1], 2, 4, 2);
	test(*r->s2, 4, 1, *arr[1], 4, 0, 1);
	test(*r->s2, 4, 1, *arr[1], 4, 1, 0);
	test(*r->s2, 4, 1, *arr[1], 4, 2, 0);
	test(*r->s2, 4, 1, *arr[1], 5, 0, 1);
	test(*r->s2, 4, 1, *arr[1], 5, 1, 1);
	test(*r->s2, 4, 1, *arr[1], 6, 0, 0);
	test(*r->s2, 4, 1, *arr[2], 0, 0, 1);
	test(*r->s2, 4, 1, *arr[2], 0, 1, 4);
	test(*r->s2, 4, 1, *arr[2], 0, 5, 4);
	test(*r->s2, 4, 1, *arr[2], 0, 9, 4);
	test(*r->s2, 4, 1, *arr[2], 0, 10, 4);
	test(*r->s2, 4, 1, *arr[2], 0, 11, 4);
	test(*r->s2, 4, 1, *arr[2], 1, 0, 1);
	test(*r->s2, 4, 1, *arr[2], 1, 1, 3);
	test(*r->s2, 4, 1, *arr[2], 1, 4, 3);
	test(*r->s2, 4, 1, *arr[2], 1, 8, 3);
	test(*r->s2, 4, 1, *arr[2], 1, 9, 3);
	test(*r->s2, 4, 1, *arr[2], 1, 10, 3);
	test(*r->s2, 4, 1, *arr[2], 5, 0, 1);
	test(*r->s2, 4, 1, *arr[2], 5, 1, -1);
	test(*r->s2, 4, 1, *arr[2], 5, 2, -1);
	test(*r->s2, 4, 1, *arr[2], 5, 4, -1);
	test(*r->s2, 4, 1, *arr[2], 5, 5, -1);
	test(*r->s2, 4, 1, *arr[2], 5, 6, -1);
	test(*r->s2, 4, 1, *arr[2], 9, 0, 1);
	test(*r->s2, 4, 1, *arr[2], 9, 1, -5);
}

template <class S, class U>
void
test16(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 4, 1, *arr[2], 9, 2, -5);
	test(*r->s2, 4, 1, *arr[2], 10, 0, 1);
	test(*r->s2, 4, 1, *arr[2], 10, 1, 1);
	test(*r->s2, 4, 1, *arr[2], 11, 0, 0);
	test(*r->s2, 4, 1, *arr[3], 0, 0, 1);
	test(*r->s2, 4, 1, *arr[3], 0, 1, 4);
	test(*r->s2, 4, 1, *arr[3], 0, 10, 4);
	test(*r->s2, 4, 1, *arr[3], 0, 19, 4);
	test(*r->s2, 4, 1, *arr[3], 0, 20, 4);
	test(*r->s2, 4, 1, *arr[3], 0, 21, 4);
	test(*r->s2, 4, 1, *arr[3], 1, 0, 1);
	test(*r->s2, 4, 1, *arr[3], 1, 1, 3);
	test(*r->s2, 4, 1, *arr[3], 1, 9, 3);
	test(*r->s2, 4, 1, *arr[3], 1, 18, 3);
	test(*r->s2, 4, 1, *arr[3], 1, 19, 3);
	test(*r->s2, 4, 1, *arr[3], 1, 20, 3);
	test(*r->s2, 4, 1, *arr[3], 10, 0, 1);
	test(*r->s2, 4, 1, *arr[3], 10, 1, -6);
	test(*r->s2, 4, 1, *arr[3], 10, 5, -6);
	test(*r->s2, 4, 1, *arr[3], 10, 9, -6);
	test(*r->s2, 4, 1, *arr[3], 10, 10, -6);
	test(*r->s2, 4, 1, *arr[3], 10, 11, -6);
	test(*r->s2, 4, 1, *arr[3], 19, 0, 1);
	test(*r->s2, 4, 1, *arr[3], 19, 1, -15);
	test(*r->s2, 4, 1, *arr[3], 19, 2, -15);
	test(*r->s2, 4, 1, *arr[3], 20, 0, 1);
	test(*r->s2, 4, 1, *arr[3], 20, 1, 1);
	test(*r->s2, 4, 1, *arr[3], 21, 0, 0);
	test(*r->s2, 4, 2, *arr[0], 0, 0, 1);
	test(*r->s2, 4, 2, *arr[0], 0, 1, 1);
	test(*r->s2, 4, 2, *arr[0], 1, 0, 0);
	test(*r->s2, 4, 2, *arr[1], 0, 0, 1);
	test(*r->s2, 4, 2, *arr[1], 0, 1, 4);
	test(*r->s2, 4, 2, *arr[1], 0, 2, 4);
	test(*r->s2, 4, 2, *arr[1], 0, 4, 4);
	test(*r->s2, 4, 2, *arr[1], 0, 5, 4);
	test(*r->s2, 4, 2, *arr[1], 0, 6, 4);
	test(*r->s2, 4, 2, *arr[1], 1, 0, 1);
	test(*r->s2, 4, 2, *arr[1], 1, 1, 3);
	test(*r->s2, 4, 2, *arr[1], 1, 2, 3);
	test(*r->s2, 4, 2, *arr[1], 1, 3, 3);
	test(*r->s2, 4, 2, *arr[1], 1, 4, 3);
	test(*r->s2, 4, 2, *arr[1], 1, 5, 3);
	test(*r->s2, 4, 2, *arr[1], 2, 0, 1);
	test(*r->s2, 4, 2, *arr[1], 2, 1, 2);
	test(*r->s2, 4, 2, *arr[1], 2, 2, 2);
	test(*r->s2, 4, 2, *arr[1], 2, 3, 2);
	test(*r->s2, 4, 2, *arr[1], 2, 4, 2);
	test(*r->s2, 4, 2, *arr[1], 4, 0, 1);
	test(*r->s2, 4, 2, *arr[1], 4, 1, 0);
	test(*r->s2, 4, 2, *arr[1], 4, 2, 0);
	test(*r->s2, 4, 2, *arr[1], 5, 0, 1);
	test(*r->s2, 4, 2, *arr[1], 5, 1, 1);
	test(*r->s2, 4, 2, *arr[1], 6, 0, 0);
	test(*r->s2, 4, 2, *arr[2], 0, 0, 1);
	test(*r->s2, 4, 2, *arr[2], 0, 1, 4);
	test(*r->s2, 4, 2, *arr[2], 0, 5, 4);
	test(*r->s2, 4, 2, *arr[2], 0, 9, 4);
	test(*r->s2, 4, 2, *arr[2], 0, 10, 4);
	test(*r->s2, 4, 2, *arr[2], 0, 11, 4);
	test(*r->s2, 4, 2, *arr[2], 1, 0, 1);
	test(*r->s2, 4, 2, *arr[2], 1, 1, 3);
	test(*r->s2, 4, 2, *arr[2], 1, 4, 3);
	test(*r->s2, 4, 2, *arr[2], 1, 8, 3);
	test(*r->s2, 4, 2, *arr[2], 1, 9, 3);
	test(*r->s2, 4, 2, *arr[2], 1, 10, 3);
	test(*r->s2, 4, 2, *arr[2], 5, 0, 1);
	test(*r->s2, 4, 2, *arr[2], 5, 1, -1);
	test(*r->s2, 4, 2, *arr[2], 5, 2, -1);
	test(*r->s2, 4, 2, *arr[2], 5, 4, -1);
	test(*r->s2, 4, 2, *arr[2], 5, 5, -1);
	test(*r->s2, 4, 2, *arr[2], 5, 6, -1);
	test(*r->s2, 4, 2, *arr[2], 9, 0, 1);
	test(*r->s2, 4, 2, *arr[2], 9, 1, -5);
	test(*r->s2, 4, 2, *arr[2], 9, 2, -5);
	test(*r->s2, 4, 2, *arr[2], 10, 0, 1);
	test(*r->s2, 4, 2, *arr[2], 10, 1, 1);
	test(*r->s2, 4, 2, *arr[2], 11, 0, 0);
	test(*r->s2, 4, 2, *arr[3], 0, 0, 1);
	test(*r->s2, 4, 2, *arr[3], 0, 1, 4);
	test(*r->s2, 4, 2, *arr[3], 0, 10, 4);
	test(*r->s2, 4, 2, *arr[3], 0, 19, 4);
	test(*r->s2, 4, 2, *arr[3], 0, 20, 4);
	test(*r->s2, 4, 2, *arr[3], 0, 21, 4);
	test(*r->s2, 4, 2, *arr[3], 1, 0, 1);
	test(*r->s2, 4, 2, *arr[3], 1, 1, 3);
	test(*r->s2, 4, 2, *arr[3], 1, 9, 3);
	test(*r->s2, 4, 2, *arr[3], 1, 18, 3);
	test(*r->s2, 4, 2, *arr[3], 1, 19, 3);
	test(*r->s2, 4, 2, *arr[3], 1, 20, 3);
	test(*r->s2, 4, 2, *arr[3], 10, 0, 1);
	test(*r->s2, 4, 2, *arr[3], 10, 1, -6);
	test(*r->s2, 4, 2, *arr[3], 10, 5, -6);
	test(*r->s2, 4, 2, *arr[3], 10, 9, -6);
	test(*r->s2, 4, 2, *arr[3], 10, 10, -6);
	test(*r->s2, 4, 2, *arr[3], 10, 11, -6);
	test(*r->s2, 4, 2, *arr[3], 19, 0, 1);
	test(*r->s2, 4, 2, *arr[3], 19, 1, -15);
	test(*r->s2, 4, 2, *arr[3], 19, 2, -15);
	test(*r->s2, 4, 2, *arr[3], 20, 0, 1);
}

template <class S, class U>
void
test17(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 4, 2, *arr[3], 20, 1, 1);
	test(*r->s2, 4, 2, *arr[3], 21, 0, 0);
	test(*r->s2, 5, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 5, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 5, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 0, 1, -1);
	test(*r->s2, 5, 0, *arr[1], 0, 2, -2);
	test(*r->s2, 5, 0, *arr[1], 0, 4, -4);
	test(*r->s2, 5, 0, *arr[1], 0, 5, -5);
	test(*r->s2, 5, 0, *arr[1], 0, 6, -5);
	test(*r->s2, 5, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 1, 1, -1);
	test(*r->s2, 5, 0, *arr[1], 1, 2, -2);
	test(*r->s2, 5, 0, *arr[1], 1, 3, -3);
	test(*r->s2, 5, 0, *arr[1], 1, 4, -4);
	test(*r->s2, 5, 0, *arr[1], 1, 5, -4);
	test(*r->s2, 5, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 2, 1, -1);
	test(*r->s2, 5, 0, *arr[1], 2, 2, -2);
	test(*r->s2, 5, 0, *arr[1], 2, 3, -3);
	test(*r->s2, 5, 0, *arr[1], 2, 4, -3);
	test(*r->s2, 5, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 4, 1, -1);
	test(*r->s2, 5, 0, *arr[1], 4, 2, -1);
	test(*r->s2, 5, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 5, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 5, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 0, 1, -1);
	test(*r->s2, 5, 0, *arr[2], 0, 5, -5);
	test(*r->s2, 5, 0, *arr[2], 0, 9, -9);
	test(*r->s2, 5, 0, *arr[2], 0, 10, -10);
	test(*r->s2, 5, 0, *arr[2], 0, 11, -10);
	test(*r->s2, 5, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 1, 1, -1);
	test(*r->s2, 5, 0, *arr[2], 1, 4, -4);
	test(*r->s2, 5, 0, *arr[2], 1, 8, -8);
	test(*r->s2, 5, 0, *arr[2], 1, 9, -9);
	test(*r->s2, 5, 0, *arr[2], 1, 10, -9);
	test(*r->s2, 5, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 5, 1, -1);
	test(*r->s2, 5, 0, *arr[2], 5, 2, -2);
	test(*r->s2, 5, 0, *arr[2], 5, 4, -4);
	test(*r->s2, 5, 0, *arr[2], 5, 5, -5);
	test(*r->s2, 5, 0, *arr[2], 5, 6, -5);
	test(*r->s2, 5, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 9, 1, -1);
	test(*r->s2, 5, 0, *arr[2], 9, 2, -1);
	test(*r->s2, 5, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 5, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 5, 0, *arr[2], 11, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 0, 1, -1);
	test(*r->s2, 5, 0, *arr[3], 0, 10, -10);
	test(*r->s2, 5, 0, *arr[3], 0, 19, -19);
	test(*r->s2, 5, 0, *arr[3], 0, 20, -20);
	test(*r->s2, 5, 0, *arr[3], 0, 21, -20);
	test(*r->s2, 5, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 1, 1, -1);
	test(*r->s2, 5, 0, *arr[3], 1, 9, -9);
	test(*r->s2, 5, 0, *arr[3], 1, 18, -18);
	test(*r->s2, 5, 0, *arr[3], 1, 19, -19);
	test(*r->s2, 5, 0, *arr[3], 1, 20, -19);
	test(*r->s2, 5, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 10, 1, -1);
	test(*r->s2, 5, 0, *arr[3], 10, 5, -5);
	test(*r->s2, 5, 0, *arr[3], 10, 9, -9);
	test(*r->s2, 5, 0, *arr[3], 10, 10, -10);
	test(*r->s2, 5, 0, *arr[3], 10, 11, -10);
	test(*r->s2, 5, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 19, 1, -1);
	test(*r->s2, 5, 0, *arr[3], 19, 2, -1);
	test(*r->s2, 5, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 5, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 5, 0, *arr[3], 21, 0, 0);
	test(*r->s2, 5, 1, *arr[0], 0, 0, 0);
	test(*r->s2, 5, 1, *arr[0], 0, 1, 0);
	test(*r->s2, 5, 1, *arr[0], 1, 0, 0);
	test(*r->s2, 5, 1, *arr[1], 0, 0, 0);
	test(*r->s2, 5, 1, *arr[1], 0, 1, -1);
	test(*r->s2, 5, 1, *arr[1], 0, 2, -2);
	test(*r->s2, 5, 1, *arr[1], 0, 4, -4);
	test(*r->s2, 5, 1, *arr[1], 0, 5, -5);
	test(*r->s2, 5, 1, *arr[1], 0, 6, -5);
	test(*r->s2, 5, 1, *arr[1], 1, 0, 0);
	test(*r->s2, 5, 1, *arr[1], 1, 1, -1);
	test(*r->s2, 5, 1, *arr[1], 1, 2, -2);
	test(*r->s2, 5, 1, *arr[1], 1, 3, -3);
	test(*r->s2, 5, 1, *arr[1], 1, 4, -4);
	test(*r->s2, 5, 1, *arr[1], 1, 5, -4);
	test(*r->s2, 5, 1, *arr[1], 2, 0, 0);
	test(*r->s2, 5, 1, *arr[1], 2, 1, -1);
	test(*r->s2, 5, 1, *arr[1], 2, 2, -2);
	test(*r->s2, 5, 1, *arr[1], 2, 3, -3);
	test(*r->s2, 5, 1, *arr[1], 2, 4, -3);
	test(*r->s2, 5, 1, *arr[1], 4, 0, 0);
	test(*r->s2, 5, 1, *arr[1], 4, 1, -1);
	test(*r->s2, 5, 1, *arr[1], 4, 2, -1);
	test(*r->s2, 5, 1, *arr[1], 5, 0, 0);
}

template <class S, class U>
void
test18(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 5, 1, *arr[1], 5, 1, 0);
	test(*r->s2, 5, 1, *arr[1], 6, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 0, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 0, 1, -1);
	test(*r->s2, 5, 1, *arr[2], 0, 5, -5);
	test(*r->s2, 5, 1, *arr[2], 0, 9, -9);
	test(*r->s2, 5, 1, *arr[2], 0, 10, -10);
	test(*r->s2, 5, 1, *arr[2], 0, 11, -10);
	test(*r->s2, 5, 1, *arr[2], 1, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 1, 1, -1);
	test(*r->s2, 5, 1, *arr[2], 1, 4, -4);
	test(*r->s2, 5, 1, *arr[2], 1, 8, -8);
	test(*r->s2, 5, 1, *arr[2], 1, 9, -9);
	test(*r->s2, 5, 1, *arr[2], 1, 10, -9);
	test(*r->s2, 5, 1, *arr[2], 5, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 5, 1, -1);
	test(*r->s2, 5, 1, *arr[2], 5, 2, -2);
	test(*r->s2, 5, 1, *arr[2], 5, 4, -4);
	test(*r->s2, 5, 1, *arr[2], 5, 5, -5);
	test(*r->s2, 5, 1, *arr[2], 5, 6, -5);
	test(*r->s2, 5, 1, *arr[2], 9, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 9, 1, -1);
	test(*r->s2, 5, 1, *arr[2], 9, 2, -1);
	test(*r->s2, 5, 1, *arr[2], 10, 0, 0);
	test(*r->s2, 5, 1, *arr[2], 10, 1, 0);
	test(*r->s2, 5, 1, *arr[2], 11, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 0, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 0, 1, -1);
	test(*r->s2, 5, 1, *arr[3], 0, 10, -10);
	test(*r->s2, 5, 1, *arr[3], 0, 19, -19);
	test(*r->s2, 5, 1, *arr[3], 0, 20, -20);
	test(*r->s2, 5, 1, *arr[3], 0, 21, -20);
	test(*r->s2, 5, 1, *arr[3], 1, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 1, 1, -1);
	test(*r->s2, 5, 1, *arr[3], 1, 9, -9);
	test(*r->s2, 5, 1, *arr[3], 1, 18, -18);
	test(*r->s2, 5, 1, *arr[3], 1, 19, -19);
	test(*r->s2, 5, 1, *arr[3], 1, 20, -19);
	test(*r->s2, 5, 1, *arr[3], 10, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 10, 1, -1);
	test(*r->s2, 5, 1, *arr[3], 10, 5, -5);
	test(*r->s2, 5, 1, *arr[3], 10, 9, -9);
	test(*r->s2, 5, 1, *arr[3], 10, 10, -10);
	test(*r->s2, 5, 1, *arr[3], 10, 11, -10);
	test(*r->s2, 5, 1, *arr[3], 19, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 19, 1, -1);
	test(*r->s2, 5, 1, *arr[3], 19, 2, -1);
	test(*r->s2, 5, 1, *arr[3], 20, 0, 0);
	test(*r->s2, 5, 1, *arr[3], 20, 1, 0);
	test(*r->s2, 5, 1, *arr[3], 21, 0, 0);
	test(*r->s2, 6, 0, *arr[0], 0, 0, 0);
	test(*r->s2, 6, 0, *arr[0], 0, 1, 0);
	test(*r->s2, 6, 0, *arr[0], 1, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 1, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 2, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 4, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 5, 0);
	test(*r->s2, 6, 0, *arr[1], 0, 6, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 1, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 2, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 3, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 4, 0);
	test(*r->s2, 6, 0, *arr[1], 1, 5, 0);
	test(*r->s2, 6, 0, *arr[1], 2, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 2, 1, 0);
	test(*r->s2, 6, 0, *arr[1], 2, 2, 0);
	test(*r->s2, 6, 0, *arr[1], 2, 3, 0);
	test(*r->s2, 6, 0, *arr[1], 2, 4, 0);
	test(*r->s2, 6, 0, *arr[1], 4, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 4, 1, 0);
	test(*r->s2, 6, 0, *arr[1], 4, 2, 0);
	test(*r->s2, 6, 0, *arr[1], 5, 0, 0);
	test(*r->s2, 6, 0, *arr[1], 5, 1, 0);
	test(*r->s2, 6, 0, *arr[1], 6, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 1, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 5, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 9, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 10, 0);
	test(*r->s2, 6, 0, *arr[2], 0, 11, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 1, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 4, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 8, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 9, 0);
	test(*r->s2, 6, 0, *arr[2], 1, 10, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 1, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 2, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 4, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 5, 0);
	test(*r->s2, 6, 0, *arr[2], 5, 6, 0);
	test(*r->s2, 6, 0, *arr[2], 9, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 9, 1, 0);
	test(*r->s2, 6, 0, *arr[2], 9, 2, 0);
	test(*r->s2, 6, 0, *arr[2], 10, 0, 0);
	test(*r->s2, 6, 0, *arr[2], 10, 1, 0);
	test(*r->s2, 6, 0, *arr[2], 11, 0, 0);
}

template <class S, class U>
void
test19(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s2, 6, 0, *arr[3], 0, 0, 0);
	test(*r->s2, 6, 0, *arr[3], 0, 1, 0);
	test(*r->s2, 6, 0, *arr[3], 0, 10, 0);
	test(*r->s2, 6, 0, *arr[3], 0, 19, 0);
	test(*r->s2, 6, 0, *arr[3], 0, 20, 0);
	test(*r->s2, 6, 0, *arr[3], 0, 21, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 0, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 1, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 9, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 18, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 19, 0);
	test(*r->s2, 6, 0, *arr[3], 1, 20, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 0, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 1, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 5, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 9, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 10, 0);
	test(*r->s2, 6, 0, *arr[3], 10, 11, 0);
	test(*r->s2, 6, 0, *arr[3], 19, 0, 0);
	test(*r->s2, 6, 0, *arr[3], 19, 1, 0);
	test(*r->s2, 6, 0, *arr[3], 19, 2, 0);
	test(*r->s2, 6, 0, *arr[3], 20, 0, 0);
	test(*r->s2, 6, 0, *arr[3], 20, 1, 0);
	test(*r->s2, 6, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 0, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 0, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 0, 1, -1);
	test(*r->s3, 0, 0, *arr[1], 0, 2, -2);
	test(*r->s3, 0, 0, *arr[1], 0, 4, -4);
	test(*r->s3, 0, 0, *arr[1], 0, 5, -5);
	test(*r->s3, 0, 0, *arr[1], 0, 6, -5);
	test(*r->s3, 0, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 0, *arr[1], 1, 2, -2);
	test(*r->s3, 0, 0, *arr[1], 1, 3, -3);
	test(*r->s3, 0, 0, *arr[1], 1, 4, -4);
	test(*r->s3, 0, 0, *arr[1], 1, 5, -4);
	test(*r->s3, 0, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 2, 1, -1);
	test(*r->s3, 0, 0, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 0, *arr[1], 2, 3, -3);
	test(*r->s3, 0, 0, *arr[1], 2, 4, -3);
	test(*r->s3, 0, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 4, 1, -1);
	test(*r->s3, 0, 0, *arr[1], 4, 2, -1);
	test(*r->s3, 0, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 0, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 0, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 0, 1, -1);
	test(*r->s3, 0, 0, *arr[2], 0, 5, -5);
	test(*r->s3, 0, 0, *arr[2], 0, 9, -9);
	test(*r->s3, 0, 0, *arr[2], 0, 10, -10);
	test(*r->s3, 0, 0, *arr[2], 0, 11, -10);
	test(*r->s3, 0, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 0, *arr[2], 1, 4, -4);
	test(*r->s3, 0, 0, *arr[2], 1, 8, -8);
	test(*r->s3, 0, 0, *arr[2], 1, 9, -9);
	test(*r->s3, 0, 0, *arr[2], 1, 10, -9);
	test(*r->s3, 0, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 5, 1, -1);
	test(*r->s3, 0, 0, *arr[2], 5, 2, -2);
	test(*r->s3, 0, 0, *arr[2], 5, 4, -4);
	test(*r->s3, 0, 0, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 0, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 9, 1, -1);
	test(*r->s3, 0, 0, *arr[2], 9, 2, -1);
	test(*r->s3, 0, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 0, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 0, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 0, 1, -1);
	test(*r->s3, 0, 0, *arr[3], 0, 10, -10);
	test(*r->s3, 0, 0, *arr[3], 0, 19, -19);
	test(*r->s3, 0, 0, *arr[3], 0, 20, -20);
	test(*r->s3, 0, 0, *arr[3], 0, 21, -20);
	test(*r->s3, 0, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 0, *arr[3], 1, 9, -9);
	test(*r->s3, 0, 0, *arr[3], 1, 18, -18);
	test(*r->s3, 0, 0, *arr[3], 1, 19, -19);
	test(*r->s3, 0, 0, *arr[3], 1, 20, -19);
	test(*r->s3, 0, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 10, 1, -1);
	test(*r->s3, 0, 0, *arr[3], 10, 5, -5);
	test(*r->s3, 0, 0, *arr[3], 10, 9, -9);
	test(*r->s3, 0, 0, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 0, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 19, 1, -1);
	test(*r->s3, 0, 0, *arr[3], 19, 2, -1);
	test(*r->s3, 0, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 0, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 0, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 1, *arr[0], 0, 0, 1);
	test(*r->s3, 0, 1, *arr[0], 0, 1, 1);
}

template <class S, class U>
void
test20(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 0, 1, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 1, *arr[1], 0, 0, 1);
	test(*r->s3, 0, 1, *arr[1], 0, 1, 0);
	test(*r->s3, 0, 1, *arr[1], 0, 2, -1);
	test(*r->s3, 0, 1, *arr[1], 0, 4, -3);
	test(*r->s3, 0, 1, *arr[1], 0, 5, -4);
	test(*r->s3, 0, 1, *arr[1], 0, 6, -4);
	test(*r->s3, 0, 1, *arr[1], 1, 0, 1);
	test(*r->s3, 0, 1, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 1, *arr[1], 1, 2, -1);
	test(*r->s3, 0, 1, *arr[1], 1, 3, -1);
	test(*r->s3, 0, 1, *arr[1], 1, 4, -1);
	test(*r->s3, 0, 1, *arr[1], 1, 5, -1);
	test(*r->s3, 0, 1, *arr[1], 2, 0, 1);
	test(*r->s3, 0, 1, *arr[1], 2, 1, -2);
	test(*r->s3, 0, 1, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 1, *arr[1], 2, 3, -2);
	test(*r->s3, 0, 1, *arr[1], 2, 4, -2);
	test(*r->s3, 0, 1, *arr[1], 4, 0, 1);
	test(*r->s3, 0, 1, *arr[1], 4, 1, -4);
	test(*r->s3, 0, 1, *arr[1], 4, 2, -4);
	test(*r->s3, 0, 1, *arr[1], 5, 0, 1);
	test(*r->s3, 0, 1, *arr[1], 5, 1, 1);
	test(*r->s3, 0, 1, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 1, *arr[2], 0, 0, 1);
	test(*r->s3, 0, 1, *arr[2], 0, 1, 0);
	test(*r->s3, 0, 1, *arr[2], 0, 5, -4);
	test(*r->s3, 0, 1, *arr[2], 0, 9, -8);
	test(*r->s3, 0, 1, *arr[2], 0, 10, -9);
	test(*r->s3, 0, 1, *arr[2], 0, 11, -9);
	test(*r->s3, 0, 1, *arr[2], 1, 0, 1);
	test(*r->s3, 0, 1, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 1, *arr[2], 1, 4, -1);
	test(*r->s3, 0, 1, *arr[2], 1, 8, -1);
	test(*r->s3, 0, 1, *arr[2], 1, 9, -1);
	test(*r->s3, 0, 1, *arr[2], 1, 10, -1);
	test(*r->s3, 0, 1, *arr[2], 5, 0, 1);
	test(*r->s3, 0, 1, *arr[2], 5, 1, -5);
	test(*r->s3, 0, 1, *arr[2], 5, 2, -5);
	test(*r->s3, 0, 1, *arr[2], 5, 4, -5);
	test(*r->s3, 0, 1, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 1, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 1, *arr[2], 9, 0, 1);
	test(*r->s3, 0, 1, *arr[2], 9, 1, -9);
	test(*r->s3, 0, 1, *arr[2], 9, 2, -9);
	test(*r->s3, 0, 1, *arr[2], 10, 0, 1);
	test(*r->s3, 0, 1, *arr[2], 10, 1, 1);
	test(*r->s3, 0, 1, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 1, *arr[3], 0, 0, 1);
	test(*r->s3, 0, 1, *arr[3], 0, 1, 0);
	test(*r->s3, 0, 1, *arr[3], 0, 10, -9);
	test(*r->s3, 0, 1, *arr[3], 0, 19, -18);
	test(*r->s3, 0, 1, *arr[3], 0, 20, -19);
	test(*r->s3, 0, 1, *arr[3], 0, 21, -19);
	test(*r->s3, 0, 1, *arr[3], 1, 0, 1);
	test(*r->s3, 0, 1, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 1, *arr[3], 1, 9, -1);
	test(*r->s3, 0, 1, *arr[3], 1, 18, -1);
	test(*r->s3, 0, 1, *arr[3], 1, 19, -1);
	test(*r->s3, 0, 1, *arr[3], 1, 20, -1);
	test(*r->s3, 0, 1, *arr[3], 10, 0, 1);
	test(*r->s3, 0, 1, *arr[3], 10, 1, -10);
	test(*r->s3, 0, 1, *arr[3], 10, 5, -10);
	test(*r->s3, 0, 1, *arr[3], 10, 9, -10);
	test(*r->s3, 0, 1, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 1, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 1, *arr[3], 19, 0, 1);
	test(*r->s3, 0, 1, *arr[3], 19, 1, -19);
	test(*r->s3, 0, 1, *arr[3], 19, 2, -19);
	test(*r->s3, 0, 1, *arr[3], 20, 0, 1);
	test(*r->s3, 0, 1, *arr[3], 20, 1, 1);
	test(*r->s3, 0, 1, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 5, *arr[0], 0, 0, 5);
	test(*r->s3, 0, 5, *arr[0], 0, 1, 5);
	test(*r->s3, 0, 5, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 5, *arr[1], 0, 0, 5);
	test(*r->s3, 0, 5, *arr[1], 0, 1, 4);
	test(*r->s3, 0, 5, *arr[1], 0, 2, 3);
	test(*r->s3, 0, 5, *arr[1], 0, 4, 1);
	test(*r->s3, 0, 5, *arr[1], 0, 5, 0);
	test(*r->s3, 0, 5, *arr[1], 0, 6, 0);
	test(*r->s3, 0, 5, *arr[1], 1, 0, 5);
	test(*r->s3, 0, 5, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 5, *arr[1], 1, 2, -1);
	test(*r->s3, 0, 5, *arr[1], 1, 3, -1);
	test(*r->s3, 0, 5, *arr[1], 1, 4, -1);
	test(*r->s3, 0, 5, *arr[1], 1, 5, -1);
	test(*r->s3, 0, 5, *arr[1], 2, 0, 5);
	test(*r->s3, 0, 5, *arr[1], 2, 1, -2);
	test(*r->s3, 0, 5, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 5, *arr[1], 2, 3, -2);
	test(*r->s3, 0, 5, *arr[1], 2, 4, -2);
	test(*r->s3, 0, 5, *arr[1], 4, 0, 5);
	test(*r->s3, 0, 5, *arr[1], 4, 1, -4);
	test(*r->s3, 0, 5, *arr[1], 4, 2, -4);
	test(*r->s3, 0, 5, *arr[1], 5, 0, 5);
	test(*r->s3, 0, 5, *arr[1], 5, 1, 5);
	test(*r->s3, 0, 5, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 5, *arr[2], 0, 0, 5);
	test(*r->s3, 0, 5, *arr[2], 0, 1, 4);
}

template <class S, class U>
void
test21(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 0, 5, *arr[2], 0, 5, 0);
	test(*r->s3, 0, 5, *arr[2], 0, 9, -4);
	test(*r->s3, 0, 5, *arr[2], 0, 10, -5);
	test(*r->s3, 0, 5, *arr[2], 0, 11, -5);
	test(*r->s3, 0, 5, *arr[2], 1, 0, 5);
	test(*r->s3, 0, 5, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 5, *arr[2], 1, 4, -1);
	test(*r->s3, 0, 5, *arr[2], 1, 8, -1);
	test(*r->s3, 0, 5, *arr[2], 1, 9, -1);
	test(*r->s3, 0, 5, *arr[2], 1, 10, -1);
	test(*r->s3, 0, 5, *arr[2], 5, 0, 5);
	test(*r->s3, 0, 5, *arr[2], 5, 1, -5);
	test(*r->s3, 0, 5, *arr[2], 5, 2, -5);
	test(*r->s3, 0, 5, *arr[2], 5, 4, -5);
	test(*r->s3, 0, 5, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 5, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 5, *arr[2], 9, 0, 5);
	test(*r->s3, 0, 5, *arr[2], 9, 1, -9);
	test(*r->s3, 0, 5, *arr[2], 9, 2, -9);
	test(*r->s3, 0, 5, *arr[2], 10, 0, 5);
	test(*r->s3, 0, 5, *arr[2], 10, 1, 5);
	test(*r->s3, 0, 5, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 5, *arr[3], 0, 0, 5);
	test(*r->s3, 0, 5, *arr[3], 0, 1, 4);
	test(*r->s3, 0, 5, *arr[3], 0, 10, -5);
	test(*r->s3, 0, 5, *arr[3], 0, 19, -14);
	test(*r->s3, 0, 5, *arr[3], 0, 20, -15);
	test(*r->s3, 0, 5, *arr[3], 0, 21, -15);
	test(*r->s3, 0, 5, *arr[3], 1, 0, 5);
	test(*r->s3, 0, 5, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 5, *arr[3], 1, 9, -1);
	test(*r->s3, 0, 5, *arr[3], 1, 18, -1);
	test(*r->s3, 0, 5, *arr[3], 1, 19, -1);
	test(*r->s3, 0, 5, *arr[3], 1, 20, -1);
	test(*r->s3, 0, 5, *arr[3], 10, 0, 5);
	test(*r->s3, 0, 5, *arr[3], 10, 1, -10);
	test(*r->s3, 0, 5, *arr[3], 10, 5, -10);
	test(*r->s3, 0, 5, *arr[3], 10, 9, -10);
	test(*r->s3, 0, 5, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 5, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 5, *arr[3], 19, 0, 5);
	test(*r->s3, 0, 5, *arr[3], 19, 1, -19);
	test(*r->s3, 0, 5, *arr[3], 19, 2, -19);
	test(*r->s3, 0, 5, *arr[3], 20, 0, 5);
	test(*r->s3, 0, 5, *arr[3], 20, 1, 5);
	test(*r->s3, 0, 5, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 9, *arr[0], 0, 0, 9);
	test(*r->s3, 0, 9, *arr[0], 0, 1, 9);
	test(*r->s3, 0, 9, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 9, *arr[1], 0, 0, 9);
	test(*r->s3, 0, 9, *arr[1], 0, 1, 8);
	test(*r->s3, 0, 9, *arr[1], 0, 2, 7);
	test(*r->s3, 0, 9, *arr[1], 0, 4, 5);
	test(*r->s3, 0, 9, *arr[1], 0, 5, 4);
	test(*r->s3, 0, 9, *arr[1], 0, 6, 4);
	test(*r->s3, 0, 9, *arr[1], 1, 0, 9);
	test(*r->s3, 0, 9, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 9, *arr[1], 1, 2, -1);
	test(*r->s3, 0, 9, *arr[1], 1, 3, -1);
	test(*r->s3, 0, 9, *arr[1], 1, 4, -1);
	test(*r->s3, 0, 9, *arr[1], 1, 5, -1);
	test(*r->s3, 0, 9, *arr[1], 2, 0, 9);
	test(*r->s3, 0, 9, *arr[1], 2, 1, -2);
	test(*r->s3, 0, 9, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 9, *arr[1], 2, 3, -2);
	test(*r->s3, 0, 9, *arr[1], 2, 4, -2);
	test(*r->s3, 0, 9, *arr[1], 4, 0, 9);
	test(*r->s3, 0, 9, *arr[1], 4, 1, -4);
	test(*r->s3, 0, 9, *arr[1], 4, 2, -4);
	test(*r->s3, 0, 9, *arr[1], 5, 0, 9);
	test(*r->s3, 0, 9, *arr[1], 5, 1, 9);
	test(*r->s3, 0, 9, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 9, *arr[2], 0, 0, 9);
	test(*r->s3, 0, 9, *arr[2], 0, 1, 8);
	test(*r->s3, 0, 9, *arr[2], 0, 5, 4);
	test(*r->s3, 0, 9, *arr[2], 0, 9, 0);
	test(*r->s3, 0, 9, *arr[2], 0, 10, -1);
	test(*r->s3, 0, 9, *arr[2], 0, 11, -1);
	test(*r->s3, 0, 9, *arr[2], 1, 0, 9);
	test(*r->s3, 0, 9, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 9, *arr[2], 1, 4, -1);
	test(*r->s3, 0, 9, *arr[2], 1, 8, -1);
	test(*r->s3, 0, 9, *arr[2], 1, 9, -1);
	test(*r->s3, 0, 9, *arr[2], 1, 10, -1);
	test(*r->s3, 0, 9, *arr[2], 5, 0, 9);
	test(*r->s3, 0, 9, *arr[2], 5, 1, -5);
	test(*r->s3, 0, 9, *arr[2], 5, 2, -5);
	test(*r->s3, 0, 9, *arr[2], 5, 4, -5);
	test(*r->s3, 0, 9, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 9, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 9, *arr[2], 9, 0, 9);
	test(*r->s3, 0, 9, *arr[2], 9, 1, -9);
	test(*r->s3, 0, 9, *arr[2], 9, 2, -9);
	test(*r->s3, 0, 9, *arr[2], 10, 0, 9);
	test(*r->s3, 0, 9, *arr[2], 10, 1, 9);
	test(*r->s3, 0, 9, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 9, *arr[3], 0, 0, 9);
	test(*r->s3, 0, 9, *arr[3], 0, 1, 8);
	test(*r->s3, 0, 9, *arr[3], 0, 10, -1);
	test(*r->s3, 0, 9, *arr[3], 0, 19, -10);
}

template <class S, class U>
void
test22(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 0, 9, *arr[3], 0, 20, -11);
	test(*r->s3, 0, 9, *arr[3], 0, 21, -11);
	test(*r->s3, 0, 9, *arr[3], 1, 0, 9);
	test(*r->s3, 0, 9, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 9, *arr[3], 1, 9, -1);
	test(*r->s3, 0, 9, *arr[3], 1, 18, -1);
	test(*r->s3, 0, 9, *arr[3], 1, 19, -1);
	test(*r->s3, 0, 9, *arr[3], 1, 20, -1);
	test(*r->s3, 0, 9, *arr[3], 10, 0, 9);
	test(*r->s3, 0, 9, *arr[3], 10, 1, -10);
	test(*r->s3, 0, 9, *arr[3], 10, 5, -10);
	test(*r->s3, 0, 9, *arr[3], 10, 9, -10);
	test(*r->s3, 0, 9, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 9, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 9, *arr[3], 19, 0, 9);
	test(*r->s3, 0, 9, *arr[3], 19, 1, -19);
	test(*r->s3, 0, 9, *arr[3], 19, 2, -19);
	test(*r->s3, 0, 9, *arr[3], 20, 0, 9);
	test(*r->s3, 0, 9, *arr[3], 20, 1, 9);
	test(*r->s3, 0, 9, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 10, *arr[0], 0, 0, 10);
	test(*r->s3, 0, 10, *arr[0], 0, 1, 10);
	test(*r->s3, 0, 10, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 10, *arr[1], 0, 0, 10);
	test(*r->s3, 0, 10, *arr[1], 0, 1, 9);
	test(*r->s3, 0, 10, *arr[1], 0, 2, 8);
	test(*r->s3, 0, 10, *arr[1], 0, 4, 6);
	test(*r->s3, 0, 10, *arr[1], 0, 5, 5);
	test(*r->s3, 0, 10, *arr[1], 0, 6, 5);
	test(*r->s3, 0, 10, *arr[1], 1, 0, 10);
	test(*r->s3, 0, 10, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 10, *arr[1], 1, 2, -1);
	test(*r->s3, 0, 10, *arr[1], 1, 3, -1);
	test(*r->s3, 0, 10, *arr[1], 1, 4, -1);
	test(*r->s3, 0, 10, *arr[1], 1, 5, -1);
	test(*r->s3, 0, 10, *arr[1], 2, 0, 10);
	test(*r->s3, 0, 10, *arr[1], 2, 1, -2);
	test(*r->s3, 0, 10, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 10, *arr[1], 2, 3, -2);
	test(*r->s3, 0, 10, *arr[1], 2, 4, -2);
	test(*r->s3, 0, 10, *arr[1], 4, 0, 10);
	test(*r->s3, 0, 10, *arr[1], 4, 1, -4);
	test(*r->s3, 0, 10, *arr[1], 4, 2, -4);
	test(*r->s3, 0, 10, *arr[1], 5, 0, 10);
	test(*r->s3, 0, 10, *arr[1], 5, 1, 10);
	test(*r->s3, 0, 10, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 10, *arr[2], 0, 0, 10);
	test(*r->s3, 0, 10, *arr[2], 0, 1, 9);
	test(*r->s3, 0, 10, *arr[2], 0, 5, 5);
	test(*r->s3, 0, 10, *arr[2], 0, 9, 1);
	test(*r->s3, 0, 10, *arr[2], 0, 10, 0);
	test(*r->s3, 0, 10, *arr[2], 0, 11, 0);
	test(*r->s3, 0, 10, *arr[2], 1, 0, 10);
	test(*r->s3, 0, 10, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 10, *arr[2], 1, 4, -1);
	test(*r->s3, 0, 10, *arr[2], 1, 8, -1);
	test(*r->s3, 0, 10, *arr[2], 1, 9, -1);
	test(*r->s3, 0, 10, *arr[2], 1, 10, -1);
	test(*r->s3, 0, 10, *arr[2], 5, 0, 10);
	test(*r->s3, 0, 10, *arr[2], 5, 1, -5);
	test(*r->s3, 0, 10, *arr[2], 5, 2, -5);
	test(*r->s3, 0, 10, *arr[2], 5, 4, -5);
	test(*r->s3, 0, 10, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 10, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 10, *arr[2], 9, 0, 10);
	test(*r->s3, 0, 10, *arr[2], 9, 1, -9);
	test(*r->s3, 0, 10, *arr[2], 9, 2, -9);
	test(*r->s3, 0, 10, *arr[2], 10, 0, 10);
	test(*r->s3, 0, 10, *arr[2], 10, 1, 10);
	test(*r->s3, 0, 10, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 10, *arr[3], 0, 0, 10);
	test(*r->s3, 0, 10, *arr[3], 0, 1, 9);
	test(*r->s3, 0, 10, *arr[3], 0, 10, 0);
	test(*r->s3, 0, 10, *arr[3], 0, 19, -9);
	test(*r->s3, 0, 10, *arr[3], 0, 20, -10);
	test(*r->s3, 0, 10, *arr[3], 0, 21, -10);
	test(*r->s3, 0, 10, *arr[3], 1, 0, 10);
	test(*r->s3, 0, 10, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 10, *arr[3], 1, 9, -1);
	test(*r->s3, 0, 10, *arr[3], 1, 18, -1);
	test(*r->s3, 0, 10, *arr[3], 1, 19, -1);
	test(*r->s3, 0, 10, *arr[3], 1, 20, -1);
	test(*r->s3, 0, 10, *arr[3], 10, 0, 10);
	test(*r->s3, 0, 10, *arr[3], 10, 1, -10);
	test(*r->s3, 0, 10, *arr[3], 10, 5, -10);
	test(*r->s3, 0, 10, *arr[3], 10, 9, -10);
	test(*r->s3, 0, 10, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 10, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 10, *arr[3], 19, 0, 10);
	test(*r->s3, 0, 10, *arr[3], 19, 1, -19);
	test(*r->s3, 0, 10, *arr[3], 19, 2, -19);
	test(*r->s3, 0, 10, *arr[3], 20, 0, 10);
	test(*r->s3, 0, 10, *arr[3], 20, 1, 10);
	test(*r->s3, 0, 10, *arr[3], 21, 0, 0);
	test(*r->s3, 0, 11, *arr[0], 0, 0, 10);
	test(*r->s3, 0, 11, *arr[0], 0, 1, 10);
	test(*r->s3, 0, 11, *arr[0], 1, 0, 0);
	test(*r->s3, 0, 11, *arr[1], 0, 0, 10);
	test(*r->s3, 0, 11, *arr[1], 0, 1, 9);
	test(*r->s3, 0, 11, *arr[1], 0, 2, 8);
}

template <class S, class U>
void
test23(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 0, 11, *arr[1], 0, 4, 6);
	test(*r->s3, 0, 11, *arr[1], 0, 5, 5);
	test(*r->s3, 0, 11, *arr[1], 0, 6, 5);
	test(*r->s3, 0, 11, *arr[1], 1, 0, 10);
	test(*r->s3, 0, 11, *arr[1], 1, 1, -1);
	test(*r->s3, 0, 11, *arr[1], 1, 2, -1);
	test(*r->s3, 0, 11, *arr[1], 1, 3, -1);
	test(*r->s3, 0, 11, *arr[1], 1, 4, -1);
	test(*r->s3, 0, 11, *arr[1], 1, 5, -1);
	test(*r->s3, 0, 11, *arr[1], 2, 0, 10);
	test(*r->s3, 0, 11, *arr[1], 2, 1, -2);
	test(*r->s3, 0, 11, *arr[1], 2, 2, -2);
	test(*r->s3, 0, 11, *arr[1], 2, 3, -2);
	test(*r->s3, 0, 11, *arr[1], 2, 4, -2);
	test(*r->s3, 0, 11, *arr[1], 4, 0, 10);
	test(*r->s3, 0, 11, *arr[1], 4, 1, -4);
	test(*r->s3, 0, 11, *arr[1], 4, 2, -4);
	test(*r->s3, 0, 11, *arr[1], 5, 0, 10);
	test(*r->s3, 0, 11, *arr[1], 5, 1, 10);
	test(*r->s3, 0, 11, *arr[1], 6, 0, 0);
	test(*r->s3, 0, 11, *arr[2], 0, 0, 10);
	test(*r->s3, 0, 11, *arr[2], 0, 1, 9);
	test(*r->s3, 0, 11, *arr[2], 0, 5, 5);
	test(*r->s3, 0, 11, *arr[2], 0, 9, 1);
	test(*r->s3, 0, 11, *arr[2], 0, 10, 0);
	test(*r->s3, 0, 11, *arr[2], 0, 11, 0);
	test(*r->s3, 0, 11, *arr[2], 1, 0, 10);
	test(*r->s3, 0, 11, *arr[2], 1, 1, -1);
	test(*r->s3, 0, 11, *arr[2], 1, 4, -1);
	test(*r->s3, 0, 11, *arr[2], 1, 8, -1);
	test(*r->s3, 0, 11, *arr[2], 1, 9, -1);
	test(*r->s3, 0, 11, *arr[2], 1, 10, -1);
	test(*r->s3, 0, 11, *arr[2], 5, 0, 10);
	test(*r->s3, 0, 11, *arr[2], 5, 1, -5);
	test(*r->s3, 0, 11, *arr[2], 5, 2, -5);
	test(*r->s3, 0, 11, *arr[2], 5, 4, -5);
	test(*r->s3, 0, 11, *arr[2], 5, 5, -5);
	test(*r->s3, 0, 11, *arr[2], 5, 6, -5);
	test(*r->s3, 0, 11, *arr[2], 9, 0, 10);
	test(*r->s3, 0, 11, *arr[2], 9, 1, -9);
	test(*r->s3, 0, 11, *arr[2], 9, 2, -9);
	test(*r->s3, 0, 11, *arr[2], 10, 0, 10);
	test(*r->s3, 0, 11, *arr[2], 10, 1, 10);
	test(*r->s3, 0, 11, *arr[2], 11, 0, 0);
	test(*r->s3, 0, 11, *arr[3], 0, 0, 10);
	test(*r->s3, 0, 11, *arr[3], 0, 1, 9);
	test(*r->s3, 0, 11, *arr[3], 0, 10, 0);
	test(*r->s3, 0, 11, *arr[3], 0, 19, -9);
	test(*r->s3, 0, 11, *arr[3], 0, 20, -10);
	test(*r->s3, 0, 11, *arr[3], 0, 21, -10);
	test(*r->s3, 0, 11, *arr[3], 1, 0, 10);
	test(*r->s3, 0, 11, *arr[3], 1, 1, -1);
	test(*r->s3, 0, 11, *arr[3], 1, 9, -1);
	test(*r->s3, 0, 11, *arr[3], 1, 18, -1);
	test(*r->s3, 0, 11, *arr[3], 1, 19, -1);
	test(*r->s3, 0, 11, *arr[3], 1, 20, -1);
	test(*r->s3, 0, 11, *arr[3], 10, 0, 10);
	test(*r->s3, 0, 11, *arr[3], 10, 1, -10);
	test(*r->s3, 0, 11, *arr[3], 10, 5, -10);
	test(*r->s3, 0, 11, *arr[3], 10, 9, -10);
	test(*r->s3, 0, 11, *arr[3], 10, 10, -10);
	test(*r->s3, 0, 11, *arr[3], 10, 11, -10);
	test(*r->s3, 0, 11, *arr[3], 19, 0, 10);
	test(*r->s3, 0, 11, *arr[3], 19, 1, -19);
	test(*r->s3, 0, 11, *arr[3], 19, 2, -19);
	test(*r->s3, 0, 11, *arr[3], 20, 0, 10);
	test(*r->s3, 0, 11, *arr[3], 20, 1, 10);
	test(*r->s3, 0, 11, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 1, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 1, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 0, 1, -1);
	test(*r->s3, 1, 0, *arr[1], 0, 2, -2);
	test(*r->s3, 1, 0, *arr[1], 0, 4, -4);
	test(*r->s3, 1, 0, *arr[1], 0, 5, -5);
	test(*r->s3, 1, 0, *arr[1], 0, 6, -5);
	test(*r->s3, 1, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 1, 1, -1);
	test(*r->s3, 1, 0, *arr[1], 1, 2, -2);
	test(*r->s3, 1, 0, *arr[1], 1, 3, -3);
	test(*r->s3, 1, 0, *arr[1], 1, 4, -4);
	test(*r->s3, 1, 0, *arr[1], 1, 5, -4);
	test(*r->s3, 1, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 0, *arr[1], 2, 2, -2);
	test(*r->s3, 1, 0, *arr[1], 2, 3, -3);
	test(*r->s3, 1, 0, *arr[1], 2, 4, -3);
	test(*r->s3, 1, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 4, 1, -1);
	test(*r->s3, 1, 0, *arr[1], 4, 2, -1);
	test(*r->s3, 1, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 1, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 1, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 0, 1, -1);
	test(*r->s3, 1, 0, *arr[2], 0, 5, -5);
	test(*r->s3, 1, 0, *arr[2], 0, 9, -9);
	test(*r->s3, 1, 0, *arr[2], 0, 10, -10);
	test(*r->s3, 1, 0, *arr[2], 0, 11, -10);
}

template <class S, class U>
void
test24(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 1, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 1, 1, -1);
	test(*r->s3, 1, 0, *arr[2], 1, 4, -4);
	test(*r->s3, 1, 0, *arr[2], 1, 8, -8);
	test(*r->s3, 1, 0, *arr[2], 1, 9, -9);
	test(*r->s3, 1, 0, *arr[2], 1, 10, -9);
	test(*r->s3, 1, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 5, 1, -1);
	test(*r->s3, 1, 0, *arr[2], 5, 2, -2);
	test(*r->s3, 1, 0, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 0, *arr[2], 5, 5, -5);
	test(*r->s3, 1, 0, *arr[2], 5, 6, -5);
	test(*r->s3, 1, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 9, 1, -1);
	test(*r->s3, 1, 0, *arr[2], 9, 2, -1);
	test(*r->s3, 1, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 1, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 1, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 0, 1, -1);
	test(*r->s3, 1, 0, *arr[3], 0, 10, -10);
	test(*r->s3, 1, 0, *arr[3], 0, 19, -19);
	test(*r->s3, 1, 0, *arr[3], 0, 20, -20);
	test(*r->s3, 1, 0, *arr[3], 0, 21, -20);
	test(*r->s3, 1, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 1, 1, -1);
	test(*r->s3, 1, 0, *arr[3], 1, 9, -9);
	test(*r->s3, 1, 0, *arr[3], 1, 18, -18);
	test(*r->s3, 1, 0, *arr[3], 1, 19, -19);
	test(*r->s3, 1, 0, *arr[3], 1, 20, -19);
	test(*r->s3, 1, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 10, 1, -1);
	test(*r->s3, 1, 0, *arr[3], 10, 5, -5);
	test(*r->s3, 1, 0, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 0, *arr[3], 10, 10, -10);
	test(*r->s3, 1, 0, *arr[3], 10, 11, -10);
	test(*r->s3, 1, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 19, 1, -1);
	test(*r->s3, 1, 0, *arr[3], 19, 2, -1);
	test(*r->s3, 1, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 1, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 1, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 1, *arr[0], 0, 0, 1);
	test(*r->s3, 1, 1, *arr[0], 0, 1, 1);
	test(*r->s3, 1, 1, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 1, *arr[1], 0, 0, 1);
	test(*r->s3, 1, 1, *arr[1], 0, 1, 1);
	test(*r->s3, 1, 1, *arr[1], 0, 2, 1);
	test(*r->s3, 1, 1, *arr[1], 0, 4, 1);
	test(*r->s3, 1, 1, *arr[1], 0, 5, 1);
	test(*r->s3, 1, 1, *arr[1], 0, 6, 1);
	test(*r->s3, 1, 1, *arr[1], 1, 0, 1);
	test(*r->s3, 1, 1, *arr[1], 1, 1, 0);
	test(*r->s3, 1, 1, *arr[1], 1, 2, -1);
	test(*r->s3, 1, 1, *arr[1], 1, 3, -2);
	test(*r->s3, 1, 1, *arr[1], 1, 4, -3);
	test(*r->s3, 1, 1, *arr[1], 1, 5, -3);
	test(*r->s3, 1, 1, *arr[1], 2, 0, 1);
	test(*r->s3, 1, 1, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 1, *arr[1], 2, 2, -1);
	test(*r->s3, 1, 1, *arr[1], 2, 3, -1);
	test(*r->s3, 1, 1, *arr[1], 2, 4, -1);
	test(*r->s3, 1, 1, *arr[1], 4, 0, 1);
	test(*r->s3, 1, 1, *arr[1], 4, 1, -3);
	test(*r->s3, 1, 1, *arr[1], 4, 2, -3);
	test(*r->s3, 1, 1, *arr[1], 5, 0, 1);
	test(*r->s3, 1, 1, *arr[1], 5, 1, 1);
	test(*r->s3, 1, 1, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 1, *arr[2], 0, 0, 1);
	test(*r->s3, 1, 1, *arr[2], 0, 1, 1);
	test(*r->s3, 1, 1, *arr[2], 0, 5, 1);
	test(*r->s3, 1, 1, *arr[2], 0, 9, 1);
	test(*r->s3, 1, 1, *arr[2], 0, 10, 1);
	test(*r->s3, 1, 1, *arr[2], 0, 11, 1);
	test(*r->s3, 1, 1, *arr[2], 1, 0, 1);
	test(*r->s3, 1, 1, *arr[2], 1, 1, 0);
	test(*r->s3, 1, 1, *arr[2], 1, 4, -3);
	test(*r->s3, 1, 1, *arr[2], 1, 8, -7);
	test(*r->s3, 1, 1, *arr[2], 1, 9, -8);
	test(*r->s3, 1, 1, *arr[2], 1, 10, -8);
	test(*r->s3, 1, 1, *arr[2], 5, 0, 1);
	test(*r->s3, 1, 1, *arr[2], 5, 1, -4);
	test(*r->s3, 1, 1, *arr[2], 5, 2, -4);
	test(*r->s3, 1, 1, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 1, *arr[2], 5, 5, -4);
	test(*r->s3, 1, 1, *arr[2], 5, 6, -4);
	test(*r->s3, 1, 1, *arr[2], 9, 0, 1);
	test(*r->s3, 1, 1, *arr[2], 9, 1, -8);
	test(*r->s3, 1, 1, *arr[2], 9, 2, -8);
	test(*r->s3, 1, 1, *arr[2], 10, 0, 1);
	test(*r->s3, 1, 1, *arr[2], 10, 1, 1);
	test(*r->s3, 1, 1, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 1, *arr[3], 0, 0, 1);
	test(*r->s3, 1, 1, *arr[3], 0, 1, 1);
	test(*r->s3, 1, 1, *arr[3], 0, 10, 1);
	test(*r->s3, 1, 1, *arr[3], 0, 19, 1);
	test(*r->s3, 1, 1, *arr[3], 0, 20, 1);
	test(*r->s3, 1, 1, *arr[3], 0, 21, 1);
	test(*r->s3, 1, 1, *arr[3], 1, 0, 1);
	test(*r->s3, 1, 1, *arr[3], 1, 1, 0);
}

template <class S, class U>
void
test25(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 1, 1, *arr[3], 1, 9, -8);
	test(*r->s3, 1, 1, *arr[3], 1, 18, -17);
	test(*r->s3, 1, 1, *arr[3], 1, 19, -18);
	test(*r->s3, 1, 1, *arr[3], 1, 20, -18);
	test(*r->s3, 1, 1, *arr[3], 10, 0, 1);
	test(*r->s3, 1, 1, *arr[3], 10, 1, -9);
	test(*r->s3, 1, 1, *arr[3], 10, 5, -9);
	test(*r->s3, 1, 1, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 1, *arr[3], 10, 10, -9);
	test(*r->s3, 1, 1, *arr[3], 10, 11, -9);
	test(*r->s3, 1, 1, *arr[3], 19, 0, 1);
	test(*r->s3, 1, 1, *arr[3], 19, 1, -18);
	test(*r->s3, 1, 1, *arr[3], 19, 2, -18);
	test(*r->s3, 1, 1, *arr[3], 20, 0, 1);
	test(*r->s3, 1, 1, *arr[3], 20, 1, 1);
	test(*r->s3, 1, 1, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 4, *arr[0], 0, 0, 4);
	test(*r->s3, 1, 4, *arr[0], 0, 1, 4);
	test(*r->s3, 1, 4, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 4, *arr[1], 0, 0, 4);
	test(*r->s3, 1, 4, *arr[1], 0, 1, 1);
	test(*r->s3, 1, 4, *arr[1], 0, 2, 1);
	test(*r->s3, 1, 4, *arr[1], 0, 4, 1);
	test(*r->s3, 1, 4, *arr[1], 0, 5, 1);
	test(*r->s3, 1, 4, *arr[1], 0, 6, 1);
	test(*r->s3, 1, 4, *arr[1], 1, 0, 4);
	test(*r->s3, 1, 4, *arr[1], 1, 1, 3);
	test(*r->s3, 1, 4, *arr[1], 1, 2, 2);
	test(*r->s3, 1, 4, *arr[1], 1, 3, 1);
	test(*r->s3, 1, 4, *arr[1], 1, 4, 0);
	test(*r->s3, 1, 4, *arr[1], 1, 5, 0);
	test(*r->s3, 1, 4, *arr[1], 2, 0, 4);
	test(*r->s3, 1, 4, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 4, *arr[1], 2, 2, -1);
	test(*r->s3, 1, 4, *arr[1], 2, 3, -1);
	test(*r->s3, 1, 4, *arr[1], 2, 4, -1);
	test(*r->s3, 1, 4, *arr[1], 4, 0, 4);
	test(*r->s3, 1, 4, *arr[1], 4, 1, -3);
	test(*r->s3, 1, 4, *arr[1], 4, 2, -3);
	test(*r->s3, 1, 4, *arr[1], 5, 0, 4);
	test(*r->s3, 1, 4, *arr[1], 5, 1, 4);
	test(*r->s3, 1, 4, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 4, *arr[2], 0, 0, 4);
	test(*r->s3, 1, 4, *arr[2], 0, 1, 1);
	test(*r->s3, 1, 4, *arr[2], 0, 5, 1);
	test(*r->s3, 1, 4, *arr[2], 0, 9, 1);
	test(*r->s3, 1, 4, *arr[2], 0, 10, 1);
	test(*r->s3, 1, 4, *arr[2], 0, 11, 1);
	test(*r->s3, 1, 4, *arr[2], 1, 0, 4);
	test(*r->s3, 1, 4, *arr[2], 1, 1, 3);
	test(*r->s3, 1, 4, *arr[2], 1, 4, 0);
	test(*r->s3, 1, 4, *arr[2], 1, 8, -4);
	test(*r->s3, 1, 4, *arr[2], 1, 9, -5);
	test(*r->s3, 1, 4, *arr[2], 1, 10, -5);
	test(*r->s3, 1, 4, *arr[2], 5, 0, 4);
	test(*r->s3, 1, 4, *arr[2], 5, 1, -4);
	test(*r->s3, 1, 4, *arr[2], 5, 2, -4);
	test(*r->s3, 1, 4, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 4, *arr[2], 5, 5, -4);
	test(*r->s3, 1, 4, *arr[2], 5, 6, -4);
	test(*r->s3, 1, 4, *arr[2], 9, 0, 4);
	test(*r->s3, 1, 4, *arr[2], 9, 1, -8);
	test(*r->s3, 1, 4, *arr[2], 9, 2, -8);
	test(*r->s3, 1, 4, *arr[2], 10, 0, 4);
	test(*r->s3, 1, 4, *arr[2], 10, 1, 4);
	test(*r->s3, 1, 4, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 4, *arr[3], 0, 0, 4);
	test(*r->s3, 1, 4, *arr[3], 0, 1, 1);
	test(*r->s3, 1, 4, *arr[3], 0, 10, 1);
	test(*r->s3, 1, 4, *arr[3], 0, 19, 1);
	test(*r->s3, 1, 4, *arr[3], 0, 20, 1);
	test(*r->s3, 1, 4, *arr[3], 0, 21, 1);
	test(*r->s3, 1, 4, *arr[3], 1, 0, 4);
	test(*r->s3, 1, 4, *arr[3], 1, 1, 3);
	test(*r->s3, 1, 4, *arr[3], 1, 9, -5);
	test(*r->s3, 1, 4, *arr[3], 1, 18, -14);
	test(*r->s3, 1, 4, *arr[3], 1, 19, -15);
	test(*r->s3, 1, 4, *arr[3], 1, 20, -15);
	test(*r->s3, 1, 4, *arr[3], 10, 0, 4);
	test(*r->s3, 1, 4, *arr[3], 10, 1, -9);
	test(*r->s3, 1, 4, *arr[3], 10, 5, -9);
	test(*r->s3, 1, 4, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 4, *arr[3], 10, 10, -9);
	test(*r->s3, 1, 4, *arr[3], 10, 11, -9);
	test(*r->s3, 1, 4, *arr[3], 19, 0, 4);
	test(*r->s3, 1, 4, *arr[3], 19, 1, -18);
	test(*r->s3, 1, 4, *arr[3], 19, 2, -18);
	test(*r->s3, 1, 4, *arr[3], 20, 0, 4);
	test(*r->s3, 1, 4, *arr[3], 20, 1, 4);
	test(*r->s3, 1, 4, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 8, *arr[0], 0, 0, 8);
	test(*r->s3, 1, 8, *arr[0], 0, 1, 8);
	test(*r->s3, 1, 8, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 8, *arr[1], 0, 0, 8);
	test(*r->s3, 1, 8, *arr[1], 0, 1, 1);
	test(*r->s3, 1, 8, *arr[1], 0, 2, 1);
	test(*r->s3, 1, 8, *arr[1], 0, 4, 1);
	test(*r->s3, 1, 8, *arr[1], 0, 5, 1);
	test(*r->s3, 1, 8, *arr[1], 0, 6, 1);
	test(*r->s3, 1, 8, *arr[1], 1, 0, 8);
}

template <class S, class U>
void
test26(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 1, 8, *arr[1], 1, 1, 7);
	test(*r->s3, 1, 8, *arr[1], 1, 2, 6);
	test(*r->s3, 1, 8, *arr[1], 1, 3, 5);
	test(*r->s3, 1, 8, *arr[1], 1, 4, 4);
	test(*r->s3, 1, 8, *arr[1], 1, 5, 4);
	test(*r->s3, 1, 8, *arr[1], 2, 0, 8);
	test(*r->s3, 1, 8, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 8, *arr[1], 2, 2, -1);
	test(*r->s3, 1, 8, *arr[1], 2, 3, -1);
	test(*r->s3, 1, 8, *arr[1], 2, 4, -1);
	test(*r->s3, 1, 8, *arr[1], 4, 0, 8);
	test(*r->s3, 1, 8, *arr[1], 4, 1, -3);
	test(*r->s3, 1, 8, *arr[1], 4, 2, -3);
	test(*r->s3, 1, 8, *arr[1], 5, 0, 8);
	test(*r->s3, 1, 8, *arr[1], 5, 1, 8);
	test(*r->s3, 1, 8, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 8, *arr[2], 0, 0, 8);
	test(*r->s3, 1, 8, *arr[2], 0, 1, 1);
	test(*r->s3, 1, 8, *arr[2], 0, 5, 1);
	test(*r->s3, 1, 8, *arr[2], 0, 9, 1);
	test(*r->s3, 1, 8, *arr[2], 0, 10, 1);
	test(*r->s3, 1, 8, *arr[2], 0, 11, 1);
	test(*r->s3, 1, 8, *arr[2], 1, 0, 8);
	test(*r->s3, 1, 8, *arr[2], 1, 1, 7);
	test(*r->s3, 1, 8, *arr[2], 1, 4, 4);
	test(*r->s3, 1, 8, *arr[2], 1, 8, 0);
	test(*r->s3, 1, 8, *arr[2], 1, 9, -1);
	test(*r->s3, 1, 8, *arr[2], 1, 10, -1);
	test(*r->s3, 1, 8, *arr[2], 5, 0, 8);
	test(*r->s3, 1, 8, *arr[2], 5, 1, -4);
	test(*r->s3, 1, 8, *arr[2], 5, 2, -4);
	test(*r->s3, 1, 8, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 8, *arr[2], 5, 5, -4);
	test(*r->s3, 1, 8, *arr[2], 5, 6, -4);
	test(*r->s3, 1, 8, *arr[2], 9, 0, 8);
	test(*r->s3, 1, 8, *arr[2], 9, 1, -8);
	test(*r->s3, 1, 8, *arr[2], 9, 2, -8);
	test(*r->s3, 1, 8, *arr[2], 10, 0, 8);
	test(*r->s3, 1, 8, *arr[2], 10, 1, 8);
	test(*r->s3, 1, 8, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 8, *arr[3], 0, 0, 8);
	test(*r->s3, 1, 8, *arr[3], 0, 1, 1);
	test(*r->s3, 1, 8, *arr[3], 0, 10, 1);
	test(*r->s3, 1, 8, *arr[3], 0, 19, 1);
	test(*r->s3, 1, 8, *arr[3], 0, 20, 1);
	test(*r->s3, 1, 8, *arr[3], 0, 21, 1);
	test(*r->s3, 1, 8, *arr[3], 1, 0, 8);
	test(*r->s3, 1, 8, *arr[3], 1, 1, 7);
	test(*r->s3, 1, 8, *arr[3], 1, 9, -1);
	test(*r->s3, 1, 8, *arr[3], 1, 18, -10);
	test(*r->s3, 1, 8, *arr[3], 1, 19, -11);
	test(*r->s3, 1, 8, *arr[3], 1, 20, -11);
	test(*r->s3, 1, 8, *arr[3], 10, 0, 8);
	test(*r->s3, 1, 8, *arr[3], 10, 1, -9);
	test(*r->s3, 1, 8, *arr[3], 10, 5, -9);
	test(*r->s3, 1, 8, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 8, *arr[3], 10, 10, -9);
	test(*r->s3, 1, 8, *arr[3], 10, 11, -9);
	test(*r->s3, 1, 8, *arr[3], 19, 0, 8);
	test(*r->s3, 1, 8, *arr[3], 19, 1, -18);
	test(*r->s3, 1, 8, *arr[3], 19, 2, -18);
	test(*r->s3, 1, 8, *arr[3], 20, 0, 8);
	test(*r->s3, 1, 8, *arr[3], 20, 1, 8);
	test(*r->s3, 1, 8, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 9, *arr[0], 0, 0, 9);
	test(*r->s3, 1, 9, *arr[0], 0, 1, 9);
	test(*r->s3, 1, 9, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 9, *arr[1], 0, 0, 9);
	test(*r->s3, 1, 9, *arr[1], 0, 1, 1);
	test(*r->s3, 1, 9, *arr[1], 0, 2, 1);
	test(*r->s3, 1, 9, *arr[1], 0, 4, 1);
	test(*r->s3, 1, 9, *arr[1], 0, 5, 1);
	test(*r->s3, 1, 9, *arr[1], 0, 6, 1);
	test(*r->s3, 1, 9, *arr[1], 1, 0, 9);
	test(*r->s3, 1, 9, *arr[1], 1, 1, 8);
	test(*r->s3, 1, 9, *arr[1], 1, 2, 7);
	test(*r->s3, 1, 9, *arr[1], 1, 3, 6);
	test(*r->s3, 1, 9, *arr[1], 1, 4, 5);
	test(*r->s3, 1, 9, *arr[1], 1, 5, 5);
	test(*r->s3, 1, 9, *arr[1], 2, 0, 9);
	test(*r->s3, 1, 9, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 9, *arr[1], 2, 2, -1);
	test(*r->s3, 1, 9, *arr[1], 2, 3, -1);
	test(*r->s3, 1, 9, *arr[1], 2, 4, -1);
	test(*r->s3, 1, 9, *arr[1], 4, 0, 9);
	test(*r->s3, 1, 9, *arr[1], 4, 1, -3);
	test(*r->s3, 1, 9, *arr[1], 4, 2, -3);
	test(*r->s3, 1, 9, *arr[1], 5, 0, 9);
	test(*r->s3, 1, 9, *arr[1], 5, 1, 9);
	test(*r->s3, 1, 9, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 9, *arr[2], 0, 0, 9);
	test(*r->s3, 1, 9, *arr[2], 0, 1, 1);
	test(*r->s3, 1, 9, *arr[2], 0, 5, 1);
	test(*r->s3, 1, 9, *arr[2], 0, 9, 1);
	test(*r->s3, 1, 9, *arr[2], 0, 10, 1);
	test(*r->s3, 1, 9, *arr[2], 0, 11, 1);
	test(*r->s3, 1, 9, *arr[2], 1, 0, 9);
	test(*r->s3, 1, 9, *arr[2], 1, 1, 8);
	test(*r->s3, 1, 9, *arr[2], 1, 4, 5);
	test(*r->s3, 1, 9, *arr[2], 1, 8, 1);
}

template <class S, class U>
void
test27(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 1, 9, *arr[2], 1, 9, 0);
	test(*r->s3, 1, 9, *arr[2], 1, 10, 0);
	test(*r->s3, 1, 9, *arr[2], 5, 0, 9);
	test(*r->s3, 1, 9, *arr[2], 5, 1, -4);
	test(*r->s3, 1, 9, *arr[2], 5, 2, -4);
	test(*r->s3, 1, 9, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 9, *arr[2], 5, 5, -4);
	test(*r->s3, 1, 9, *arr[2], 5, 6, -4);
	test(*r->s3, 1, 9, *arr[2], 9, 0, 9);
	test(*r->s3, 1, 9, *arr[2], 9, 1, -8);
	test(*r->s3, 1, 9, *arr[2], 9, 2, -8);
	test(*r->s3, 1, 9, *arr[2], 10, 0, 9);
	test(*r->s3, 1, 9, *arr[2], 10, 1, 9);
	test(*r->s3, 1, 9, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 9, *arr[3], 0, 0, 9);
	test(*r->s3, 1, 9, *arr[3], 0, 1, 1);
	test(*r->s3, 1, 9, *arr[3], 0, 10, 1);
	test(*r->s3, 1, 9, *arr[3], 0, 19, 1);
	test(*r->s3, 1, 9, *arr[3], 0, 20, 1);
	test(*r->s3, 1, 9, *arr[3], 0, 21, 1);
	test(*r->s3, 1, 9, *arr[3], 1, 0, 9);
	test(*r->s3, 1, 9, *arr[3], 1, 1, 8);
	test(*r->s3, 1, 9, *arr[3], 1, 9, 0);
	test(*r->s3, 1, 9, *arr[3], 1, 18, -9);
	test(*r->s3, 1, 9, *arr[3], 1, 19, -10);
	test(*r->s3, 1, 9, *arr[3], 1, 20, -10);
	test(*r->s3, 1, 9, *arr[3], 10, 0, 9);
	test(*r->s3, 1, 9, *arr[3], 10, 1, -9);
	test(*r->s3, 1, 9, *arr[3], 10, 5, -9);
	test(*r->s3, 1, 9, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 9, *arr[3], 10, 10, -9);
	test(*r->s3, 1, 9, *arr[3], 10, 11, -9);
	test(*r->s3, 1, 9, *arr[3], 19, 0, 9);
	test(*r->s3, 1, 9, *arr[3], 19, 1, -18);
	test(*r->s3, 1, 9, *arr[3], 19, 2, -18);
	test(*r->s3, 1, 9, *arr[3], 20, 0, 9);
	test(*r->s3, 1, 9, *arr[3], 20, 1, 9);
	test(*r->s3, 1, 9, *arr[3], 21, 0, 0);
	test(*r->s3, 1, 10, *arr[0], 0, 0, 9);
	test(*r->s3, 1, 10, *arr[0], 0, 1, 9);
	test(*r->s3, 1, 10, *arr[0], 1, 0, 0);
	test(*r->s3, 1, 10, *arr[1], 0, 0, 9);
	test(*r->s3, 1, 10, *arr[1], 0, 1, 1);
	test(*r->s3, 1, 10, *arr[1], 0, 2, 1);
	test(*r->s3, 1, 10, *arr[1], 0, 4, 1);
	test(*r->s3, 1, 10, *arr[1], 0, 5, 1);
	test(*r->s3, 1, 10, *arr[1], 0, 6, 1);
	test(*r->s3, 1, 10, *arr[1], 1, 0, 9);
	test(*r->s3, 1, 10, *arr[1], 1, 1, 8);
	test(*r->s3, 1, 10, *arr[1], 1, 2, 7);
	test(*r->s3, 1, 10, *arr[1], 1, 3, 6);
	test(*r->s3, 1, 10, *arr[1], 1, 4, 5);
	test(*r->s3, 1, 10, *arr[1], 1, 5, 5);
	test(*r->s3, 1, 10, *arr[1], 2, 0, 9);
	test(*r->s3, 1, 10, *arr[1], 2, 1, -1);
	test(*r->s3, 1, 10, *arr[1], 2, 2, -1);
	test(*r->s3, 1, 10, *arr[1], 2, 3, -1);
	test(*r->s3, 1, 10, *arr[1], 2, 4, -1);
	test(*r->s3, 1, 10, *arr[1], 4, 0, 9);
	test(*r->s3, 1, 10, *arr[1], 4, 1, -3);
	test(*r->s3, 1, 10, *arr[1], 4, 2, -3);
	test(*r->s3, 1, 10, *arr[1], 5, 0, 9);
	test(*r->s3, 1, 10, *arr[1], 5, 1, 9);
	test(*r->s3, 1, 10, *arr[1], 6, 0, 0);
	test(*r->s3, 1, 10, *arr[2], 0, 0, 9);
	test(*r->s3, 1, 10, *arr[2], 0, 1, 1);
	test(*r->s3, 1, 10, *arr[2], 0, 5, 1);
	test(*r->s3, 1, 10, *arr[2], 0, 9, 1);
	test(*r->s3, 1, 10, *arr[2], 0, 10, 1);
	test(*r->s3, 1, 10, *arr[2], 0, 11, 1);
	test(*r->s3, 1, 10, *arr[2], 1, 0, 9);
	test(*r->s3, 1, 10, *arr[2], 1, 1, 8);
	test(*r->s3, 1, 10, *arr[2], 1, 4, 5);
	test(*r->s3, 1, 10, *arr[2], 1, 8, 1);
	test(*r->s3, 1, 10, *arr[2], 1, 9, 0);
	test(*r->s3, 1, 10, *arr[2], 1, 10, 0);
	test(*r->s3, 1, 10, *arr[2], 5, 0, 9);
	test(*r->s3, 1, 10, *arr[2], 5, 1, -4);
	test(*r->s3, 1, 10, *arr[2], 5, 2, -4);
	test(*r->s3, 1, 10, *arr[2], 5, 4, -4);
	test(*r->s3, 1, 10, *arr[2], 5, 5, -4);
	test(*r->s3, 1, 10, *arr[2], 5, 6, -4);
	test(*r->s3, 1, 10, *arr[2], 9, 0, 9);
	test(*r->s3, 1, 10, *arr[2], 9, 1, -8);
	test(*r->s3, 1, 10, *arr[2], 9, 2, -8);
	test(*r->s3, 1, 10, *arr[2], 10, 0, 9);
	test(*r->s3, 1, 10, *arr[2], 10, 1, 9);
	test(*r->s3, 1, 10, *arr[2], 11, 0, 0);
	test(*r->s3, 1, 10, *arr[3], 0, 0, 9);
	test(*r->s3, 1, 10, *arr[3], 0, 1, 1);
	test(*r->s3, 1, 10, *arr[3], 0, 10, 1);
	test(*r->s3, 1, 10, *arr[3], 0, 19, 1);
	test(*r->s3, 1, 10, *arr[3], 0, 20, 1);
	test(*r->s3, 1, 10, *arr[3], 0, 21, 1);
	test(*r->s3, 1, 10, *arr[3], 1, 0, 9);
	test(*r->s3, 1, 10, *arr[3], 1, 1, 8);
	test(*r->s3, 1, 10, *arr[3], 1, 9, 0);
	test(*r->s3, 1, 10, *arr[3], 1, 18, -9);
	test(*r->s3, 1, 10, *arr[3], 1, 19, -10);
	test(*r->s3, 1, 10, *arr[3], 1, 20, -10);
}

template <class S, class U>
void
test28(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 1, 10, *arr[3], 10, 0, 9);
	test(*r->s3, 1, 10, *arr[3], 10, 1, -9);
	test(*r->s3, 1, 10, *arr[3], 10, 5, -9);
	test(*r->s3, 1, 10, *arr[3], 10, 9, -9);
	test(*r->s3, 1, 10, *arr[3], 10, 10, -9);
	test(*r->s3, 1, 10, *arr[3], 10, 11, -9);
	test(*r->s3, 1, 10, *arr[3], 19, 0, 9);
	test(*r->s3, 1, 10, *arr[3], 19, 1, -18);
	test(*r->s3, 1, 10, *arr[3], 19, 2, -18);
	test(*r->s3, 1, 10, *arr[3], 20, 0, 9);
	test(*r->s3, 1, 10, *arr[3], 20, 1, 9);
	test(*r->s3, 1, 10, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 5, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 5, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 0, 1, -1);
	test(*r->s3, 5, 0, *arr[1], 0, 2, -2);
	test(*r->s3, 5, 0, *arr[1], 0, 4, -4);
	test(*r->s3, 5, 0, *arr[1], 0, 5, -5);
	test(*r->s3, 5, 0, *arr[1], 0, 6, -5);
	test(*r->s3, 5, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 1, 1, -1);
	test(*r->s3, 5, 0, *arr[1], 1, 2, -2);
	test(*r->s3, 5, 0, *arr[1], 1, 3, -3);
	test(*r->s3, 5, 0, *arr[1], 1, 4, -4);
	test(*r->s3, 5, 0, *arr[1], 1, 5, -4);
	test(*r->s3, 5, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 2, 1, -1);
	test(*r->s3, 5, 0, *arr[1], 2, 2, -2);
	test(*r->s3, 5, 0, *arr[1], 2, 3, -3);
	test(*r->s3, 5, 0, *arr[1], 2, 4, -3);
	test(*r->s3, 5, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 4, 1, -1);
	test(*r->s3, 5, 0, *arr[1], 4, 2, -1);
	test(*r->s3, 5, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 5, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 5, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 0, 1, -1);
	test(*r->s3, 5, 0, *arr[2], 0, 5, -5);
	test(*r->s3, 5, 0, *arr[2], 0, 9, -9);
	test(*r->s3, 5, 0, *arr[2], 0, 10, -10);
	test(*r->s3, 5, 0, *arr[2], 0, 11, -10);
	test(*r->s3, 5, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 1, 1, -1);
	test(*r->s3, 5, 0, *arr[2], 1, 4, -4);
	test(*r->s3, 5, 0, *arr[2], 1, 8, -8);
	test(*r->s3, 5, 0, *arr[2], 1, 9, -9);
	test(*r->s3, 5, 0, *arr[2], 1, 10, -9);
	test(*r->s3, 5, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 5, 1, -1);
	test(*r->s3, 5, 0, *arr[2], 5, 2, -2);
	test(*r->s3, 5, 0, *arr[2], 5, 4, -4);
	test(*r->s3, 5, 0, *arr[2], 5, 5, -5);
	test(*r->s3, 5, 0, *arr[2], 5, 6, -5);
	test(*r->s3, 5, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 9, 1, -1);
	test(*r->s3, 5, 0, *arr[2], 9, 2, -1);
	test(*r->s3, 5, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 5, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 5, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 0, 1, -1);
	test(*r->s3, 5, 0, *arr[3], 0, 10, -10);
	test(*r->s3, 5, 0, *arr[3], 0, 19, -19);
	test(*r->s3, 5, 0, *arr[3], 0, 20, -20);
	test(*r->s3, 5, 0, *arr[3], 0, 21, -20);
	test(*r->s3, 5, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 1, 1, -1);
	test(*r->s3, 5, 0, *arr[3], 1, 9, -9);
	test(*r->s3, 5, 0, *arr[3], 1, 18, -18);
	test(*r->s3, 5, 0, *arr[3], 1, 19, -19);
	test(*r->s3, 5, 0, *arr[3], 1, 20, -19);
	test(*r->s3, 5, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 10, 1, -1);
	test(*r->s3, 5, 0, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 0, *arr[3], 10, 9, -9);
	test(*r->s3, 5, 0, *arr[3], 10, 10, -10);
	test(*r->s3, 5, 0, *arr[3], 10, 11, -10);
	test(*r->s3, 5, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 19, 1, -1);
	test(*r->s3, 5, 0, *arr[3], 19, 2, -1);
	test(*r->s3, 5, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 5, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 5, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 1, *arr[0], 0, 0, 1);
	test(*r->s3, 5, 1, *arr[0], 0, 1, 1);
	test(*r->s3, 5, 1, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 1, *arr[1], 0, 0, 1);
	test(*r->s3, 5, 1, *arr[1], 0, 1, 5);
	test(*r->s3, 5, 1, *arr[1], 0, 2, 5);
	test(*r->s3, 5, 1, *arr[1], 0, 4, 5);
	test(*r->s3, 5, 1, *arr[1], 0, 5, 5);
	test(*r->s3, 5, 1, *arr[1], 0, 6, 5);
	test(*r->s3, 5, 1, *arr[1], 1, 0, 1);
	test(*r->s3, 5, 1, *arr[1], 1, 1, 4);
	test(*r->s3, 5, 1, *arr[1], 1, 2, 4);
	test(*r->s3, 5, 1, *arr[1], 1, 3, 4);
	test(*r->s3, 5, 1, *arr[1], 1, 4, 4);
}

template <class S, class U>
void
test29(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 5, 1, *arr[1], 1, 5, 4);
	test(*r->s3, 5, 1, *arr[1], 2, 0, 1);
	test(*r->s3, 5, 1, *arr[1], 2, 1, 3);
	test(*r->s3, 5, 1, *arr[1], 2, 2, 3);
	test(*r->s3, 5, 1, *arr[1], 2, 3, 3);
	test(*r->s3, 5, 1, *arr[1], 2, 4, 3);
	test(*r->s3, 5, 1, *arr[1], 4, 0, 1);
	test(*r->s3, 5, 1, *arr[1], 4, 1, 1);
	test(*r->s3, 5, 1, *arr[1], 4, 2, 1);
	test(*r->s3, 5, 1, *arr[1], 5, 0, 1);
	test(*r->s3, 5, 1, *arr[1], 5, 1, 1);
	test(*r->s3, 5, 1, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 1, *arr[2], 0, 0, 1);
	test(*r->s3, 5, 1, *arr[2], 0, 1, 5);
	test(*r->s3, 5, 1, *arr[2], 0, 5, 5);
	test(*r->s3, 5, 1, *arr[2], 0, 9, 5);
	test(*r->s3, 5, 1, *arr[2], 0, 10, 5);
	test(*r->s3, 5, 1, *arr[2], 0, 11, 5);
	test(*r->s3, 5, 1, *arr[2], 1, 0, 1);
	test(*r->s3, 5, 1, *arr[2], 1, 1, 4);
	test(*r->s3, 5, 1, *arr[2], 1, 4, 4);
	test(*r->s3, 5, 1, *arr[2], 1, 8, 4);
	test(*r->s3, 5, 1, *arr[2], 1, 9, 4);
	test(*r->s3, 5, 1, *arr[2], 1, 10, 4);
	test(*r->s3, 5, 1, *arr[2], 5, 0, 1);
	test(*r->s3, 5, 1, *arr[2], 5, 1, 0);
	test(*r->s3, 5, 1, *arr[2], 5, 2, -1);
	test(*r->s3, 5, 1, *arr[2], 5, 4, -3);
	test(*r->s3, 5, 1, *arr[2], 5, 5, -4);
	test(*r->s3, 5, 1, *arr[2], 5, 6, -4);
	test(*r->s3, 5, 1, *arr[2], 9, 0, 1);
	test(*r->s3, 5, 1, *arr[2], 9, 1, -4);
	test(*r->s3, 5, 1, *arr[2], 9, 2, -4);
	test(*r->s3, 5, 1, *arr[2], 10, 0, 1);
	test(*r->s3, 5, 1, *arr[2], 10, 1, 1);
	test(*r->s3, 5, 1, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 1, *arr[3], 0, 0, 1);
	test(*r->s3, 5, 1, *arr[3], 0, 1, 5);
	test(*r->s3, 5, 1, *arr[3], 0, 10, 5);
	test(*r->s3, 5, 1, *arr[3], 0, 19, 5);
	test(*r->s3, 5, 1, *arr[3], 0, 20, 5);
	test(*r->s3, 5, 1, *arr[3], 0, 21, 5);
	test(*r->s3, 5, 1, *arr[3], 1, 0, 1);
	test(*r->s3, 5, 1, *arr[3], 1, 1, 4);
	test(*r->s3, 5, 1, *arr[3], 1, 9, 4);
	test(*r->s3, 5, 1, *arr[3], 1, 18, 4);
	test(*r->s3, 5, 1, *arr[3], 1, 19, 4);
	test(*r->s3, 5, 1, *arr[3], 1, 20, 4);
	test(*r->s3, 5, 1, *arr[3], 10, 0, 1);
	test(*r->s3, 5, 1, *arr[3], 10, 1, -5);
	test(*r->s3, 5, 1, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 1, *arr[3], 10, 9, -5);
	test(*r->s3, 5, 1, *arr[3], 10, 10, -5);
	test(*r->s3, 5, 1, *arr[3], 10, 11, -5);
	test(*r->s3, 5, 1, *arr[3], 19, 0, 1);
	test(*r->s3, 5, 1, *arr[3], 19, 1, -14);
	test(*r->s3, 5, 1, *arr[3], 19, 2, -14);
	test(*r->s3, 5, 1, *arr[3], 20, 0, 1);
	test(*r->s3, 5, 1, *arr[3], 20, 1, 1);
	test(*r->s3, 5, 1, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 2, *arr[0], 0, 0, 2);
	test(*r->s3, 5, 2, *arr[0], 0, 1, 2);
	test(*r->s3, 5, 2, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 2, *arr[1], 0, 0, 2);
	test(*r->s3, 5, 2, *arr[1], 0, 1, 5);
	test(*r->s3, 5, 2, *arr[1], 0, 2, 5);
	test(*r->s3, 5, 2, *arr[1], 0, 4, 5);
	test(*r->s3, 5, 2, *arr[1], 0, 5, 5);
	test(*r->s3, 5, 2, *arr[1], 0, 6, 5);
	test(*r->s3, 5, 2, *arr[1], 1, 0, 2);
	test(*r->s3, 5, 2, *arr[1], 1, 1, 4);
	test(*r->s3, 5, 2, *arr[1], 1, 2, 4);
	test(*r->s3, 5, 2, *arr[1], 1, 3, 4);
	test(*r->s3, 5, 2, *arr[1], 1, 4, 4);
	test(*r->s3, 5, 2, *arr[1], 1, 5, 4);
	test(*r->s3, 5, 2, *arr[1], 2, 0, 2);
	test(*r->s3, 5, 2, *arr[1], 2, 1, 3);
	test(*r->s3, 5, 2, *arr[1], 2, 2, 3);
	test(*r->s3, 5, 2, *arr[1], 2, 3, 3);
	test(*r->s3, 5, 2, *arr[1], 2, 4, 3);
	test(*r->s3, 5, 2, *arr[1], 4, 0, 2);
	test(*r->s3, 5, 2, *arr[1], 4, 1, 1);
	test(*r->s3, 5, 2, *arr[1], 4, 2, 1);
	test(*r->s3, 5, 2, *arr[1], 5, 0, 2);
	test(*r->s3, 5, 2, *arr[1], 5, 1, 2);
	test(*r->s3, 5, 2, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 2, *arr[2], 0, 0, 2);
	test(*r->s3, 5, 2, *arr[2], 0, 1, 5);
	test(*r->s3, 5, 2, *arr[2], 0, 5, 5);
	test(*r->s3, 5, 2, *arr[2], 0, 9, 5);
	test(*r->s3, 5, 2, *arr[2], 0, 10, 5);
	test(*r->s3, 5, 2, *arr[2], 0, 11, 5);
	test(*r->s3, 5, 2, *arr[2], 1, 0, 2);
	test(*r->s3, 5, 2, *arr[2], 1, 1, 4);
	test(*r->s3, 5, 2, *arr[2], 1, 4, 4);
	test(*r->s3, 5, 2, *arr[2], 1, 8, 4);
	test(*r->s3, 5, 2, *arr[2], 1, 9, 4);
	test(*r->s3, 5, 2, *arr[2], 1, 10, 4);
	test(*r->s3, 5, 2, *arr[2], 5, 0, 2);
	test(*r->s3, 5, 2, *arr[2], 5, 1, 1);
}

template <class S, class U>
void
test30(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 5, 2, *arr[2], 5, 2, 0);
	test(*r->s3, 5, 2, *arr[2], 5, 4, -2);
	test(*r->s3, 5, 2, *arr[2], 5, 5, -3);
	test(*r->s3, 5, 2, *arr[2], 5, 6, -3);
	test(*r->s3, 5, 2, *arr[2], 9, 0, 2);
	test(*r->s3, 5, 2, *arr[2], 9, 1, -4);
	test(*r->s3, 5, 2, *arr[2], 9, 2, -4);
	test(*r->s3, 5, 2, *arr[2], 10, 0, 2);
	test(*r->s3, 5, 2, *arr[2], 10, 1, 2);
	test(*r->s3, 5, 2, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 2, *arr[3], 0, 0, 2);
	test(*r->s3, 5, 2, *arr[3], 0, 1, 5);
	test(*r->s3, 5, 2, *arr[3], 0, 10, 5);
	test(*r->s3, 5, 2, *arr[3], 0, 19, 5);
	test(*r->s3, 5, 2, *arr[3], 0, 20, 5);
	test(*r->s3, 5, 2, *arr[3], 0, 21, 5);
	test(*r->s3, 5, 2, *arr[3], 1, 0, 2);
	test(*r->s3, 5, 2, *arr[3], 1, 1, 4);
	test(*r->s3, 5, 2, *arr[3], 1, 9, 4);
	test(*r->s3, 5, 2, *arr[3], 1, 18, 4);
	test(*r->s3, 5, 2, *arr[3], 1, 19, 4);
	test(*r->s3, 5, 2, *arr[3], 1, 20, 4);
	test(*r->s3, 5, 2, *arr[3], 10, 0, 2);
	test(*r->s3, 5, 2, *arr[3], 10, 1, -5);
	test(*r->s3, 5, 2, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 2, *arr[3], 10, 9, -5);
	test(*r->s3, 5, 2, *arr[3], 10, 10, -5);
	test(*r->s3, 5, 2, *arr[3], 10, 11, -5);
	test(*r->s3, 5, 2, *arr[3], 19, 0, 2);
	test(*r->s3, 5, 2, *arr[3], 19, 1, -14);
	test(*r->s3, 5, 2, *arr[3], 19, 2, -14);
	test(*r->s3, 5, 2, *arr[3], 20, 0, 2);
	test(*r->s3, 5, 2, *arr[3], 20, 1, 2);
	test(*r->s3, 5, 2, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 4, *arr[0], 0, 0, 4);
	test(*r->s3, 5, 4, *arr[0], 0, 1, 4);
	test(*r->s3, 5, 4, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 4, *arr[1], 0, 0, 4);
	test(*r->s3, 5, 4, *arr[1], 0, 1, 5);
	test(*r->s3, 5, 4, *arr[1], 0, 2, 5);
	test(*r->s3, 5, 4, *arr[1], 0, 4, 5);
	test(*r->s3, 5, 4, *arr[1], 0, 5, 5);
	test(*r->s3, 5, 4, *arr[1], 0, 6, 5);
	test(*r->s3, 5, 4, *arr[1], 1, 0, 4);
	test(*r->s3, 5, 4, *arr[1], 1, 1, 4);
	test(*r->s3, 5, 4, *arr[1], 1, 2, 4);
	test(*r->s3, 5, 4, *arr[1], 1, 3, 4);
	test(*r->s3, 5, 4, *arr[1], 1, 4, 4);
	test(*r->s3, 5, 4, *arr[1], 1, 5, 4);
	test(*r->s3, 5, 4, *arr[1], 2, 0, 4);
	test(*r->s3, 5, 4, *arr[1], 2, 1, 3);
	test(*r->s3, 5, 4, *arr[1], 2, 2, 3);
	test(*r->s3, 5, 4, *arr[1], 2, 3, 3);
	test(*r->s3, 5, 4, *arr[1], 2, 4, 3);
	test(*r->s3, 5, 4, *arr[1], 4, 0, 4);
	test(*r->s3, 5, 4, *arr[1], 4, 1, 1);
	test(*r->s3, 5, 4, *arr[1], 4, 2, 1);
	test(*r->s3, 5, 4, *arr[1], 5, 0, 4);
	test(*r->s3, 5, 4, *arr[1], 5, 1, 4);
	test(*r->s3, 5, 4, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 4, *arr[2], 0, 0, 4);
	test(*r->s3, 5, 4, *arr[2], 0, 1, 5);
	test(*r->s3, 5, 4, *arr[2], 0, 5, 5);
	test(*r->s3, 5, 4, *arr[2], 0, 9, 5);
	test(*r->s3, 5, 4, *arr[2], 0, 10, 5);
	test(*r->s3, 5, 4, *arr[2], 0, 11, 5);
	test(*r->s3, 5, 4, *arr[2], 1, 0, 4);
	test(*r->s3, 5, 4, *arr[2], 1, 1, 4);
	test(*r->s3, 5, 4, *arr[2], 1, 4, 4);
	test(*r->s3, 5, 4, *arr[2], 1, 8, 4);
	test(*r->s3, 5, 4, *arr[2], 1, 9, 4);
	test(*r->s3, 5, 4, *arr[2], 1, 10, 4);
	test(*r->s3, 5, 4, *arr[2], 5, 0, 4);
	test(*r->s3, 5, 4, *arr[2], 5, 1, 3);
	test(*r->s3, 5, 4, *arr[2], 5, 2, 2);
	test(*r->s3, 5, 4, *arr[2], 5, 4, 0);
	test(*r->s3, 5, 4, *arr[2], 5, 5, -1);
	test(*r->s3, 5, 4, *arr[2], 5, 6, -1);
	test(*r->s3, 5, 4, *arr[2], 9, 0, 4);
	test(*r->s3, 5, 4, *arr[2], 9, 1, -4);
	test(*r->s3, 5, 4, *arr[2], 9, 2, -4);
	test(*r->s3, 5, 4, *arr[2], 10, 0, 4);
	test(*r->s3, 5, 4, *arr[2], 10, 1, 4);
	test(*r->s3, 5, 4, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 4, *arr[3], 0, 0, 4);
	test(*r->s3, 5, 4, *arr[3], 0, 1, 5);
	test(*r->s3, 5, 4, *arr[3], 0, 10, 5);
	test(*r->s3, 5, 4, *arr[3], 0, 19, 5);
	test(*r->s3, 5, 4, *arr[3], 0, 20, 5);
	test(*r->s3, 5, 4, *arr[3], 0, 21, 5);
	test(*r->s3, 5, 4, *arr[3], 1, 0, 4);
	test(*r->s3, 5, 4, *arr[3], 1, 1, 4);
	test(*r->s3, 5, 4, *arr[3], 1, 9, 4);
	test(*r->s3, 5, 4, *arr[3], 1, 18, 4);
	test(*r->s3, 5, 4, *arr[3], 1, 19, 4);
	test(*r->s3, 5, 4, *arr[3], 1, 20, 4);
	test(*r->s3, 5, 4, *arr[3], 10, 0, 4);
	test(*r->s3, 5, 4, *arr[3], 10, 1, -5);
	test(*r->s3, 5, 4, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 4, *arr[3], 10, 9, -5);
}

template <class S, class U>
void
test31(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 5, 4, *arr[3], 10, 10, -5);
	test(*r->s3, 5, 4, *arr[3], 10, 11, -5);
	test(*r->s3, 5, 4, *arr[3], 19, 0, 4);
	test(*r->s3, 5, 4, *arr[3], 19, 1, -14);
	test(*r->s3, 5, 4, *arr[3], 19, 2, -14);
	test(*r->s3, 5, 4, *arr[3], 20, 0, 4);
	test(*r->s3, 5, 4, *arr[3], 20, 1, 4);
	test(*r->s3, 5, 4, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 5, *arr[0], 0, 0, 5);
	test(*r->s3, 5, 5, *arr[0], 0, 1, 5);
	test(*r->s3, 5, 5, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 5, *arr[1], 0, 0, 5);
	test(*r->s3, 5, 5, *arr[1], 0, 1, 5);
	test(*r->s3, 5, 5, *arr[1], 0, 2, 5);
	test(*r->s3, 5, 5, *arr[1], 0, 4, 5);
	test(*r->s3, 5, 5, *arr[1], 0, 5, 5);
	test(*r->s3, 5, 5, *arr[1], 0, 6, 5);
	test(*r->s3, 5, 5, *arr[1], 1, 0, 5);
	test(*r->s3, 5, 5, *arr[1], 1, 1, 4);
	test(*r->s3, 5, 5, *arr[1], 1, 2, 4);
	test(*r->s3, 5, 5, *arr[1], 1, 3, 4);
	test(*r->s3, 5, 5, *arr[1], 1, 4, 4);
	test(*r->s3, 5, 5, *arr[1], 1, 5, 4);
	test(*r->s3, 5, 5, *arr[1], 2, 0, 5);
	test(*r->s3, 5, 5, *arr[1], 2, 1, 3);
	test(*r->s3, 5, 5, *arr[1], 2, 2, 3);
	test(*r->s3, 5, 5, *arr[1], 2, 3, 3);
	test(*r->s3, 5, 5, *arr[1], 2, 4, 3);
	test(*r->s3, 5, 5, *arr[1], 4, 0, 5);
	test(*r->s3, 5, 5, *arr[1], 4, 1, 1);
	test(*r->s3, 5, 5, *arr[1], 4, 2, 1);
	test(*r->s3, 5, 5, *arr[1], 5, 0, 5);
	test(*r->s3, 5, 5, *arr[1], 5, 1, 5);
	test(*r->s3, 5, 5, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 5, *arr[2], 0, 0, 5);
	test(*r->s3, 5, 5, *arr[2], 0, 1, 5);
	test(*r->s3, 5, 5, *arr[2], 0, 5, 5);
	test(*r->s3, 5, 5, *arr[2], 0, 9, 5);
	test(*r->s3, 5, 5, *arr[2], 0, 10, 5);
	test(*r->s3, 5, 5, *arr[2], 0, 11, 5);
	test(*r->s3, 5, 5, *arr[2], 1, 0, 5);
	test(*r->s3, 5, 5, *arr[2], 1, 1, 4);
	test(*r->s3, 5, 5, *arr[2], 1, 4, 4);
	test(*r->s3, 5, 5, *arr[2], 1, 8, 4);
	test(*r->s3, 5, 5, *arr[2], 1, 9, 4);
	test(*r->s3, 5, 5, *arr[2], 1, 10, 4);
	test(*r->s3, 5, 5, *arr[2], 5, 0, 5);
	test(*r->s3, 5, 5, *arr[2], 5, 1, 4);
	test(*r->s3, 5, 5, *arr[2], 5, 2, 3);
	test(*r->s3, 5, 5, *arr[2], 5, 4, 1);
	test(*r->s3, 5, 5, *arr[2], 5, 5, 0);
	test(*r->s3, 5, 5, *arr[2], 5, 6, 0);
	test(*r->s3, 5, 5, *arr[2], 9, 0, 5);
	test(*r->s3, 5, 5, *arr[2], 9, 1, -4);
	test(*r->s3, 5, 5, *arr[2], 9, 2, -4);
	test(*r->s3, 5, 5, *arr[2], 10, 0, 5);
	test(*r->s3, 5, 5, *arr[2], 10, 1, 5);
	test(*r->s3, 5, 5, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 5, *arr[3], 0, 0, 5);
	test(*r->s3, 5, 5, *arr[3], 0, 1, 5);
	test(*r->s3, 5, 5, *arr[3], 0, 10, 5);
	test(*r->s3, 5, 5, *arr[3], 0, 19, 5);
	test(*r->s3, 5, 5, *arr[3], 0, 20, 5);
	test(*r->s3, 5, 5, *arr[3], 0, 21, 5);
	test(*r->s3, 5, 5, *arr[3], 1, 0, 5);
	test(*r->s3, 5, 5, *arr[3], 1, 1, 4);
	test(*r->s3, 5, 5, *arr[3], 1, 9, 4);
	test(*r->s3, 5, 5, *arr[3], 1, 18, 4);
	test(*r->s3, 5, 5, *arr[3], 1, 19, 4);
	test(*r->s3, 5, 5, *arr[3], 1, 20, 4);
	test(*r->s3, 5, 5, *arr[3], 10, 0, 5);
	test(*r->s3, 5, 5, *arr[3], 10, 1, -5);
	test(*r->s3, 5, 5, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 5, *arr[3], 10, 9, -5);
	test(*r->s3, 5, 5, *arr[3], 10, 10, -5);
	test(*r->s3, 5, 5, *arr[3], 10, 11, -5);
	test(*r->s3, 5, 5, *arr[3], 19, 0, 5);
	test(*r->s3, 5, 5, *arr[3], 19, 1, -14);
	test(*r->s3, 5, 5, *arr[3], 19, 2, -14);
	test(*r->s3, 5, 5, *arr[3], 20, 0, 5);
	test(*r->s3, 5, 5, *arr[3], 20, 1, 5);
	test(*r->s3, 5, 5, *arr[3], 21, 0, 0);
	test(*r->s3, 5, 6, *arr[0], 0, 0, 5);
	test(*r->s3, 5, 6, *arr[0], 0, 1, 5);
	test(*r->s3, 5, 6, *arr[0], 1, 0, 0);
	test(*r->s3, 5, 6, *arr[1], 0, 0, 5);
	test(*r->s3, 5, 6, *arr[1], 0, 1, 5);
	test(*r->s3, 5, 6, *arr[1], 0, 2, 5);
	test(*r->s3, 5, 6, *arr[1], 0, 4, 5);
	test(*r->s3, 5, 6, *arr[1], 0, 5, 5);
	test(*r->s3, 5, 6, *arr[1], 0, 6, 5);
	test(*r->s3, 5, 6, *arr[1], 1, 0, 5);
	test(*r->s3, 5, 6, *arr[1], 1, 1, 4);
	test(*r->s3, 5, 6, *arr[1], 1, 2, 4);
	test(*r->s3, 5, 6, *arr[1], 1, 3, 4);
	test(*r->s3, 5, 6, *arr[1], 1, 4, 4);
	test(*r->s3, 5, 6, *arr[1], 1, 5, 4);
	test(*r->s3, 5, 6, *arr[1], 2, 0, 5);
	test(*r->s3, 5, 6, *arr[1], 2, 1, 3);
	test(*r->s3, 5, 6, *arr[1], 2, 2, 3);
}

template <class S, class U>
void
test32(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 5, 6, *arr[1], 2, 3, 3);
	test(*r->s3, 5, 6, *arr[1], 2, 4, 3);
	test(*r->s3, 5, 6, *arr[1], 4, 0, 5);
	test(*r->s3, 5, 6, *arr[1], 4, 1, 1);
	test(*r->s3, 5, 6, *arr[1], 4, 2, 1);
	test(*r->s3, 5, 6, *arr[1], 5, 0, 5);
	test(*r->s3, 5, 6, *arr[1], 5, 1, 5);
	test(*r->s3, 5, 6, *arr[1], 6, 0, 0);
	test(*r->s3, 5, 6, *arr[2], 0, 0, 5);
	test(*r->s3, 5, 6, *arr[2], 0, 1, 5);
	test(*r->s3, 5, 6, *arr[2], 0, 5, 5);
	test(*r->s3, 5, 6, *arr[2], 0, 9, 5);
	test(*r->s3, 5, 6, *arr[2], 0, 10, 5);
	test(*r->s3, 5, 6, *arr[2], 0, 11, 5);
	test(*r->s3, 5, 6, *arr[2], 1, 0, 5);
	test(*r->s3, 5, 6, *arr[2], 1, 1, 4);
	test(*r->s3, 5, 6, *arr[2], 1, 4, 4);
	test(*r->s3, 5, 6, *arr[2], 1, 8, 4);
	test(*r->s3, 5, 6, *arr[2], 1, 9, 4);
	test(*r->s3, 5, 6, *arr[2], 1, 10, 4);
	test(*r->s3, 5, 6, *arr[2], 5, 0, 5);
	test(*r->s3, 5, 6, *arr[2], 5, 1, 4);
	test(*r->s3, 5, 6, *arr[2], 5, 2, 3);
	test(*r->s3, 5, 6, *arr[2], 5, 4, 1);
	test(*r->s3, 5, 6, *arr[2], 5, 5, 0);
	test(*r->s3, 5, 6, *arr[2], 5, 6, 0);
	test(*r->s3, 5, 6, *arr[2], 9, 0, 5);
	test(*r->s3, 5, 6, *arr[2], 9, 1, -4);
	test(*r->s3, 5, 6, *arr[2], 9, 2, -4);
	test(*r->s3, 5, 6, *arr[2], 10, 0, 5);
	test(*r->s3, 5, 6, *arr[2], 10, 1, 5);
	test(*r->s3, 5, 6, *arr[2], 11, 0, 0);
	test(*r->s3, 5, 6, *arr[3], 0, 0, 5);
	test(*r->s3, 5, 6, *arr[3], 0, 1, 5);
	test(*r->s3, 5, 6, *arr[3], 0, 10, 5);
	test(*r->s3, 5, 6, *arr[3], 0, 19, 5);
	test(*r->s3, 5, 6, *arr[3], 0, 20, 5);
	test(*r->s3, 5, 6, *arr[3], 0, 21, 5);
	test(*r->s3, 5, 6, *arr[3], 1, 0, 5);
	test(*r->s3, 5, 6, *arr[3], 1, 1, 4);
	test(*r->s3, 5, 6, *arr[3], 1, 9, 4);
	test(*r->s3, 5, 6, *arr[3], 1, 18, 4);
	test(*r->s3, 5, 6, *arr[3], 1, 19, 4);
	test(*r->s3, 5, 6, *arr[3], 1, 20, 4);
	test(*r->s3, 5, 6, *arr[3], 10, 0, 5);
	test(*r->s3, 5, 6, *arr[3], 10, 1, -5);
	test(*r->s3, 5, 6, *arr[3], 10, 5, -5);
	test(*r->s3, 5, 6, *arr[3], 10, 9, -5);
	test(*r->s3, 5, 6, *arr[3], 10, 10, -5);
	test(*r->s3, 5, 6, *arr[3], 10, 11, -5);
	test(*r->s3, 5, 6, *arr[3], 19, 0, 5);
	test(*r->s3, 5, 6, *arr[3], 19, 1, -14);
	test(*r->s3, 5, 6, *arr[3], 19, 2, -14);
	test(*r->s3, 5, 6, *arr[3], 20, 0, 5);
	test(*r->s3, 5, 6, *arr[3], 20, 1, 5);
	test(*r->s3, 5, 6, *arr[3], 21, 0, 0);
	test(*r->s3, 9, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 9, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 9, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 0, 1, -1);
	test(*r->s3, 9, 0, *arr[1], 0, 2, -2);
	test(*r->s3, 9, 0, *arr[1], 0, 4, -4);
	test(*r->s3, 9, 0, *arr[1], 0, 5, -5);
	test(*r->s3, 9, 0, *arr[1], 0, 6, -5);
	test(*r->s3, 9, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 1, 1, -1);
	test(*r->s3, 9, 0, *arr[1], 1, 2, -2);
	test(*r->s3, 9, 0, *arr[1], 1, 3, -3);
	test(*r->s3, 9, 0, *arr[1], 1, 4, -4);
	test(*r->s3, 9, 0, *arr[1], 1, 5, -4);
	test(*r->s3, 9, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 2, 1, -1);
	test(*r->s3, 9, 0, *arr[1], 2, 2, -2);
	test(*r->s3, 9, 0, *arr[1], 2, 3, -3);
	test(*r->s3, 9, 0, *arr[1], 2, 4, -3);
	test(*r->s3, 9, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 4, 1, -1);
	test(*r->s3, 9, 0, *arr[1], 4, 2, -1);
	test(*r->s3, 9, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 9, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 9, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 0, 1, -1);
	test(*r->s3, 9, 0, *arr[2], 0, 5, -5);
	test(*r->s3, 9, 0, *arr[2], 0, 9, -9);
	test(*r->s3, 9, 0, *arr[2], 0, 10, -10);
	test(*r->s3, 9, 0, *arr[2], 0, 11, -10);
	test(*r->s3, 9, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 1, 1, -1);
	test(*r->s3, 9, 0, *arr[2], 1, 4, -4);
	test(*r->s3, 9, 0, *arr[2], 1, 8, -8);
	test(*r->s3, 9, 0, *arr[2], 1, 9, -9);
	test(*r->s3, 9, 0, *arr[2], 1, 10, -9);
	test(*r->s3, 9, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 5, 1, -1);
	test(*r->s3, 9, 0, *arr[2], 5, 2, -2);
	test(*r->s3, 9, 0, *arr[2], 5, 4, -4);
	test(*r->s3, 9, 0, *arr[2], 5, 5, -5);
	test(*r->s3, 9, 0, *arr[2], 5, 6, -5);
}

template <class S, class U>
void
test33(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 9, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 9, 1, -1);
	test(*r->s3, 9, 0, *arr[2], 9, 2, -1);
	test(*r->s3, 9, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 9, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 9, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 0, 1, -1);
	test(*r->s3, 9, 0, *arr[3], 0, 10, -10);
	test(*r->s3, 9, 0, *arr[3], 0, 19, -19);
	test(*r->s3, 9, 0, *arr[3], 0, 20, -20);
	test(*r->s3, 9, 0, *arr[3], 0, 21, -20);
	test(*r->s3, 9, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 1, 1, -1);
	test(*r->s3, 9, 0, *arr[3], 1, 9, -9);
	test(*r->s3, 9, 0, *arr[3], 1, 18, -18);
	test(*r->s3, 9, 0, *arr[3], 1, 19, -19);
	test(*r->s3, 9, 0, *arr[3], 1, 20, -19);
	test(*r->s3, 9, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 10, 1, -1);
	test(*r->s3, 9, 0, *arr[3], 10, 5, -5);
	test(*r->s3, 9, 0, *arr[3], 10, 9, -9);
	test(*r->s3, 9, 0, *arr[3], 10, 10, -10);
	test(*r->s3, 9, 0, *arr[3], 10, 11, -10);
	test(*r->s3, 9, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 19, 1, -1);
	test(*r->s3, 9, 0, *arr[3], 19, 2, -1);
	test(*r->s3, 9, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 9, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 9, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 9, 1, *arr[0], 0, 0, 1);
	test(*r->s3, 9, 1, *arr[0], 0, 1, 1);
	test(*r->s3, 9, 1, *arr[0], 1, 0, 0);
	test(*r->s3, 9, 1, *arr[1], 0, 0, 1);
	test(*r->s3, 9, 1, *arr[1], 0, 1, 9);
	test(*r->s3, 9, 1, *arr[1], 0, 2, 9);
	test(*r->s3, 9, 1, *arr[1], 0, 4, 9);
	test(*r->s3, 9, 1, *arr[1], 0, 5, 9);
	test(*r->s3, 9, 1, *arr[1], 0, 6, 9);
	test(*r->s3, 9, 1, *arr[1], 1, 0, 1);
	test(*r->s3, 9, 1, *arr[1], 1, 1, 8);
	test(*r->s3, 9, 1, *arr[1], 1, 2, 8);
	test(*r->s3, 9, 1, *arr[1], 1, 3, 8);
	test(*r->s3, 9, 1, *arr[1], 1, 4, 8);
	test(*r->s3, 9, 1, *arr[1], 1, 5, 8);
	test(*r->s3, 9, 1, *arr[1], 2, 0, 1);
	test(*r->s3, 9, 1, *arr[1], 2, 1, 7);
	test(*r->s3, 9, 1, *arr[1], 2, 2, 7);
	test(*r->s3, 9, 1, *arr[1], 2, 3, 7);
	test(*r->s3, 9, 1, *arr[1], 2, 4, 7);
	test(*r->s3, 9, 1, *arr[1], 4, 0, 1);
	test(*r->s3, 9, 1, *arr[1], 4, 1, 5);
	test(*r->s3, 9, 1, *arr[1], 4, 2, 5);
	test(*r->s3, 9, 1, *arr[1], 5, 0, 1);
	test(*r->s3, 9, 1, *arr[1], 5, 1, 1);
	test(*r->s3, 9, 1, *arr[1], 6, 0, 0);
	test(*r->s3, 9, 1, *arr[2], 0, 0, 1);
	test(*r->s3, 9, 1, *arr[2], 0, 1, 9);
	test(*r->s3, 9, 1, *arr[2], 0, 5, 9);
	test(*r->s3, 9, 1, *arr[2], 0, 9, 9);
	test(*r->s3, 9, 1, *arr[2], 0, 10, 9);
	test(*r->s3, 9, 1, *arr[2], 0, 11, 9);
	test(*r->s3, 9, 1, *arr[2], 1, 0, 1);
	test(*r->s3, 9, 1, *arr[2], 1, 1, 8);
	test(*r->s3, 9, 1, *arr[2], 1, 4, 8);
	test(*r->s3, 9, 1, *arr[2], 1, 8, 8);
	test(*r->s3, 9, 1, *arr[2], 1, 9, 8);
	test(*r->s3, 9, 1, *arr[2], 1, 10, 8);
	test(*r->s3, 9, 1, *arr[2], 5, 0, 1);
	test(*r->s3, 9, 1, *arr[2], 5, 1, 4);
	test(*r->s3, 9, 1, *arr[2], 5, 2, 4);
	test(*r->s3, 9, 1, *arr[2], 5, 4, 4);
	test(*r->s3, 9, 1, *arr[2], 5, 5, 4);
	test(*r->s3, 9, 1, *arr[2], 5, 6, 4);
	test(*r->s3, 9, 1, *arr[2], 9, 0, 1);
	test(*r->s3, 9, 1, *arr[2], 9, 1, 0);
	test(*r->s3, 9, 1, *arr[2], 9, 2, 0);
	test(*r->s3, 9, 1, *arr[2], 10, 0, 1);
	test(*r->s3, 9, 1, *arr[2], 10, 1, 1);
	test(*r->s3, 9, 1, *arr[2], 11, 0, 0);
	test(*r->s3, 9, 1, *arr[3], 0, 0, 1);
	test(*r->s3, 9, 1, *arr[3], 0, 1, 9);
	test(*r->s3, 9, 1, *arr[3], 0, 10, 9);
	test(*r->s3, 9, 1, *arr[3], 0, 19, 9);
	test(*r->s3, 9, 1, *arr[3], 0, 20, 9);
	test(*r->s3, 9, 1, *arr[3], 0, 21, 9);
	test(*r->s3, 9, 1, *arr[3], 1, 0, 1);
	test(*r->s3, 9, 1, *arr[3], 1, 1, 8);
	test(*r->s3, 9, 1, *arr[3], 1, 9, 8);
	test(*r->s3, 9, 1, *arr[3], 1, 18, 8);
	test(*r->s3, 9, 1, *arr[3], 1, 19, 8);
	test(*r->s3, 9, 1, *arr[3], 1, 20, 8);
	test(*r->s3, 9, 1, *arr[3], 10, 0, 1);
	test(*r->s3, 9, 1, *arr[3], 10, 1, -1);
	test(*r->s3, 9, 1, *arr[3], 10, 5, -1);
	test(*r->s3, 9, 1, *arr[3], 10, 9, -1);
	test(*r->s3, 9, 1, *arr[3], 10, 10, -1);
	test(*r->s3, 9, 1, *arr[3], 10, 11, -1);
	test(*r->s3, 9, 1, *arr[3], 19, 0, 1);
	test(*r->s3, 9, 1, *arr[3], 19, 1, -10);
}

template <class S, class U>
void
test34(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 9, 1, *arr[3], 19, 2, -10);
	test(*r->s3, 9, 1, *arr[3], 20, 0, 1);
	test(*r->s3, 9, 1, *arr[3], 20, 1, 1);
	test(*r->s3, 9, 1, *arr[3], 21, 0, 0);
	test(*r->s3, 9, 2, *arr[0], 0, 0, 1);
	test(*r->s3, 9, 2, *arr[0], 0, 1, 1);
	test(*r->s3, 9, 2, *arr[0], 1, 0, 0);
	test(*r->s3, 9, 2, *arr[1], 0, 0, 1);
	test(*r->s3, 9, 2, *arr[1], 0, 1, 9);
	test(*r->s3, 9, 2, *arr[1], 0, 2, 9);
	test(*r->s3, 9, 2, *arr[1], 0, 4, 9);
	test(*r->s3, 9, 2, *arr[1], 0, 5, 9);
	test(*r->s3, 9, 2, *arr[1], 0, 6, 9);
	test(*r->s3, 9, 2, *arr[1], 1, 0, 1);
	test(*r->s3, 9, 2, *arr[1], 1, 1, 8);
	test(*r->s3, 9, 2, *arr[1], 1, 2, 8);
	test(*r->s3, 9, 2, *arr[1], 1, 3, 8);
	test(*r->s3, 9, 2, *arr[1], 1, 4, 8);
	test(*r->s3, 9, 2, *arr[1], 1, 5, 8);
	test(*r->s3, 9, 2, *arr[1], 2, 0, 1);
	test(*r->s3, 9, 2, *arr[1], 2, 1, 7);
	test(*r->s3, 9, 2, *arr[1], 2, 2, 7);
	test(*r->s3, 9, 2, *arr[1], 2, 3, 7);
	test(*r->s3, 9, 2, *arr[1], 2, 4, 7);
	test(*r->s3, 9, 2, *arr[1], 4, 0, 1);
	test(*r->s3, 9, 2, *arr[1], 4, 1, 5);
	test(*r->s3, 9, 2, *arr[1], 4, 2, 5);
	test(*r->s3, 9, 2, *arr[1], 5, 0, 1);
	test(*r->s3, 9, 2, *arr[1], 5, 1, 1);
	test(*r->s3, 9, 2, *arr[1], 6, 0, 0);
	test(*r->s3, 9, 2, *arr[2], 0, 0, 1);
	test(*r->s3, 9, 2, *arr[2], 0, 1, 9);
	test(*r->s3, 9, 2, *arr[2], 0, 5, 9);
	test(*r->s3, 9, 2, *arr[2], 0, 9, 9);
	test(*r->s3, 9, 2, *arr[2], 0, 10, 9);
	test(*r->s3, 9, 2, *arr[2], 0, 11, 9);
	test(*r->s3, 9, 2, *arr[2], 1, 0, 1);
	test(*r->s3, 9, 2, *arr[2], 1, 1, 8);
	test(*r->s3, 9, 2, *arr[2], 1, 4, 8);
	test(*r->s3, 9, 2, *arr[2], 1, 8, 8);
	test(*r->s3, 9, 2, *arr[2], 1, 9, 8);
	test(*r->s3, 9, 2, *arr[2], 1, 10, 8);
	test(*r->s3, 9, 2, *arr[2], 5, 0, 1);
	test(*r->s3, 9, 2, *arr[2], 5, 1, 4);
	test(*r->s3, 9, 2, *arr[2], 5, 2, 4);
	test(*r->s3, 9, 2, *arr[2], 5, 4, 4);
	test(*r->s3, 9, 2, *arr[2], 5, 5, 4);
	test(*r->s3, 9, 2, *arr[2], 5, 6, 4);
	test(*r->s3, 9, 2, *arr[2], 9, 0, 1);
	test(*r->s3, 9, 2, *arr[2], 9, 1, 0);
	test(*r->s3, 9, 2, *arr[2], 9, 2, 0);
	test(*r->s3, 9, 2, *arr[2], 10, 0, 1);
	test(*r->s3, 9, 2, *arr[2], 10, 1, 1);
	test(*r->s3, 9, 2, *arr[2], 11, 0, 0);
	test(*r->s3, 9, 2, *arr[3], 0, 0, 1);
	test(*r->s3, 9, 2, *arr[3], 0, 1, 9);
	test(*r->s3, 9, 2, *arr[3], 0, 10, 9);
	test(*r->s3, 9, 2, *arr[3], 0, 19, 9);
	test(*r->s3, 9, 2, *arr[3], 0, 20, 9);
	test(*r->s3, 9, 2, *arr[3], 0, 21, 9);
	test(*r->s3, 9, 2, *arr[3], 1, 0, 1);
	test(*r->s3, 9, 2, *arr[3], 1, 1, 8);
	test(*r->s3, 9, 2, *arr[3], 1, 9, 8);
	test(*r->s3, 9, 2, *arr[3], 1, 18, 8);
	test(*r->s3, 9, 2, *arr[3], 1, 19, 8);
	test(*r->s3, 9, 2, *arr[3], 1, 20, 8);
	test(*r->s3, 9, 2, *arr[3], 10, 0, 1);
	test(*r->s3, 9, 2, *arr[3], 10, 1, -1);
	test(*r->s3, 9, 2, *arr[3], 10, 5, -1);
	test(*r->s3, 9, 2, *arr[3], 10, 9, -1);
	test(*r->s3, 9, 2, *arr[3], 10, 10, -1);
	test(*r->s3, 9, 2, *arr[3], 10, 11, -1);
	test(*r->s3, 9, 2, *arr[3], 19, 0, 1);
	test(*r->s3, 9, 2, *arr[3], 19, 1, -10);
	test(*r->s3, 9, 2, *arr[3], 19, 2, -10);
	test(*r->s3, 9, 2, *arr[3], 20, 0, 1);
	test(*r->s3, 9, 2, *arr[3], 20, 1, 1);
	test(*r->s3, 9, 2, *arr[3], 21, 0, 0);
	test(*r->s3, 10, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 10, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 10, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 0, 1, -1);
	test(*r->s3, 10, 0, *arr[1], 0, 2, -2);
	test(*r->s3, 10, 0, *arr[1], 0, 4, -4);
	test(*r->s3, 10, 0, *arr[1], 0, 5, -5);
	test(*r->s3, 10, 0, *arr[1], 0, 6, -5);
	test(*r->s3, 10, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 1, 1, -1);
	test(*r->s3, 10, 0, *arr[1], 1, 2, -2);
	test(*r->s3, 10, 0, *arr[1], 1, 3, -3);
	test(*r->s3, 10, 0, *arr[1], 1, 4, -4);
	test(*r->s3, 10, 0, *arr[1], 1, 5, -4);
	test(*r->s3, 10, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 2, 1, -1);
	test(*r->s3, 10, 0, *arr[1], 2, 2, -2);
	test(*r->s3, 10, 0, *arr[1], 2, 3, -3);
	test(*r->s3, 10, 0, *arr[1], 2, 4, -3);
	test(*r->s3, 10, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 4, 1, -1);
}

template <class S, class U>
void
test35(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 10, 0, *arr[1], 4, 2, -1);
	test(*r->s3, 10, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 10, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 10, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 0, 1, -1);
	test(*r->s3, 10, 0, *arr[2], 0, 5, -5);
	test(*r->s3, 10, 0, *arr[2], 0, 9, -9);
	test(*r->s3, 10, 0, *arr[2], 0, 10, -10);
	test(*r->s3, 10, 0, *arr[2], 0, 11, -10);
	test(*r->s3, 10, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 1, 1, -1);
	test(*r->s3, 10, 0, *arr[2], 1, 4, -4);
	test(*r->s3, 10, 0, *arr[2], 1, 8, -8);
	test(*r->s3, 10, 0, *arr[2], 1, 9, -9);
	test(*r->s3, 10, 0, *arr[2], 1, 10, -9);
	test(*r->s3, 10, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 5, 1, -1);
	test(*r->s3, 10, 0, *arr[2], 5, 2, -2);
	test(*r->s3, 10, 0, *arr[2], 5, 4, -4);
	test(*r->s3, 10, 0, *arr[2], 5, 5, -5);
	test(*r->s3, 10, 0, *arr[2], 5, 6, -5);
	test(*r->s3, 10, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 9, 1, -1);
	test(*r->s3, 10, 0, *arr[2], 9, 2, -1);
	test(*r->s3, 10, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 10, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 10, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 0, 1, -1);
	test(*r->s3, 10, 0, *arr[3], 0, 10, -10);
	test(*r->s3, 10, 0, *arr[3], 0, 19, -19);
	test(*r->s3, 10, 0, *arr[3], 0, 20, -20);
	test(*r->s3, 10, 0, *arr[3], 0, 21, -20);
	test(*r->s3, 10, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 1, 1, -1);
	test(*r->s3, 10, 0, *arr[3], 1, 9, -9);
	test(*r->s3, 10, 0, *arr[3], 1, 18, -18);
	test(*r->s3, 10, 0, *arr[3], 1, 19, -19);
	test(*r->s3, 10, 0, *arr[3], 1, 20, -19);
	test(*r->s3, 10, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 10, 1, -1);
	test(*r->s3, 10, 0, *arr[3], 10, 5, -5);
	test(*r->s3, 10, 0, *arr[3], 10, 9, -9);
	test(*r->s3, 10, 0, *arr[3], 10, 10, -10);
	test(*r->s3, 10, 0, *arr[3], 10, 11, -10);
	test(*r->s3, 10, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 19, 1, -1);
	test(*r->s3, 10, 0, *arr[3], 19, 2, -1);
	test(*r->s3, 10, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 10, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 10, 0, *arr[3], 21, 0, 0);
	test(*r->s3, 10, 1, *arr[0], 0, 0, 0);
	test(*r->s3, 10, 1, *arr[0], 0, 1, 0);
	test(*r->s3, 10, 1, *arr[0], 1, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 0, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 0, 1, -1);
	test(*r->s3, 10, 1, *arr[1], 0, 2, -2);
	test(*r->s3, 10, 1, *arr[1], 0, 4, -4);
	test(*r->s3, 10, 1, *arr[1], 0, 5, -5);
	test(*r->s3, 10, 1, *arr[1], 0, 6, -5);
	test(*r->s3, 10, 1, *arr[1], 1, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 1, 1, -1);
	test(*r->s3, 10, 1, *arr[1], 1, 2, -2);
	test(*r->s3, 10, 1, *arr[1], 1, 3, -3);
	test(*r->s3, 10, 1, *arr[1], 1, 4, -4);
	test(*r->s3, 10, 1, *arr[1], 1, 5, -4);
	test(*r->s3, 10, 1, *arr[1], 2, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 2, 1, -1);
	test(*r->s3, 10, 1, *arr[1], 2, 2, -2);
	test(*r->s3, 10, 1, *arr[1], 2, 3, -3);
	test(*r->s3, 10, 1, *arr[1], 2, 4, -3);
	test(*r->s3, 10, 1, *arr[1], 4, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 4, 1, -1);
	test(*r->s3, 10, 1, *arr[1], 4, 2, -1);
	test(*r->s3, 10, 1, *arr[1], 5, 0, 0);
	test(*r->s3, 10, 1, *arr[1], 5, 1, 0);
	test(*r->s3, 10, 1, *arr[1], 6, 0, 0);
	test(*r->s3, 10, 1, *arr[2], 0, 0, 0);
	test(*r->s3, 10, 1, *arr[2], 0, 1, -1);
	test(*r->s3, 10, 1, *arr[2], 0, 5, -5);
	test(*r->s3, 10, 1, *arr[2], 0, 9, -9);
	test(*r->s3, 10, 1, *arr[2], 0, 10, -10);
	test(*r->s3, 10, 1, *arr[2], 0, 11, -10);
	test(*r->s3, 10, 1, *arr[2], 1, 0, 0);
	test(*r->s3, 10, 1, *arr[2], 1, 1, -1);
	test(*r->s3, 10, 1, *arr[2], 1, 4, -4);
	test(*r->s3, 10, 1, *arr[2], 1, 8, -8);
	test(*r->s3, 10, 1, *arr[2], 1, 9, -9);
	test(*r->s3, 10, 1, *arr[2], 1, 10, -9);
	test(*r->s3, 10, 1, *arr[2], 5, 0, 0);
	test(*r->s3, 10, 1, *arr[2], 5, 1, -1);
	test(*r->s3, 10, 1, *arr[2], 5, 2, -2);
	test(*r->s3, 10, 1, *arr[2], 5, 4, -4);
	test(*r->s3, 10, 1, *arr[2], 5, 5, -5);
	test(*r->s3, 10, 1, *arr[2], 5, 6, -5);
	test(*r->s3, 10, 1, *arr[2], 9, 0, 0);
	test(*r->s3, 10, 1, *arr[2], 9, 1, -1);
	test(*r->s3, 10, 1, *arr[2], 9, 2, -1);
	test(*r->s3, 10, 1, *arr[2], 10, 0, 0);
}

template <class S, class U>
void
test36(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s3, 10, 1, *arr[2], 10, 1, 0);
	test(*r->s3, 10, 1, *arr[2], 11, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 0, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 0, 1, -1);
	test(*r->s3, 10, 1, *arr[3], 0, 10, -10);
	test(*r->s3, 10, 1, *arr[3], 0, 19, -19);
	test(*r->s3, 10, 1, *arr[3], 0, 20, -20);
	test(*r->s3, 10, 1, *arr[3], 0, 21, -20);
	test(*r->s3, 10, 1, *arr[3], 1, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 1, 1, -1);
	test(*r->s3, 10, 1, *arr[3], 1, 9, -9);
	test(*r->s3, 10, 1, *arr[3], 1, 18, -18);
	test(*r->s3, 10, 1, *arr[3], 1, 19, -19);
	test(*r->s3, 10, 1, *arr[3], 1, 20, -19);
	test(*r->s3, 10, 1, *arr[3], 10, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 10, 1, -1);
	test(*r->s3, 10, 1, *arr[3], 10, 5, -5);
	test(*r->s3, 10, 1, *arr[3], 10, 9, -9);
	test(*r->s3, 10, 1, *arr[3], 10, 10, -10);
	test(*r->s3, 10, 1, *arr[3], 10, 11, -10);
	test(*r->s3, 10, 1, *arr[3], 19, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 19, 1, -1);
	test(*r->s3, 10, 1, *arr[3], 19, 2, -1);
	test(*r->s3, 10, 1, *arr[3], 20, 0, 0);
	test(*r->s3, 10, 1, *arr[3], 20, 1, 0);
	test(*r->s3, 10, 1, *arr[3], 21, 0, 0);
	test(*r->s3, 11, 0, *arr[0], 0, 0, 0);
	test(*r->s3, 11, 0, *arr[0], 0, 1, 0);
	test(*r->s3, 11, 0, *arr[0], 1, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 1, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 2, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 4, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 5, 0);
	test(*r->s3, 11, 0, *arr[1], 0, 6, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 1, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 2, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 3, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 4, 0);
	test(*r->s3, 11, 0, *arr[1], 1, 5, 0);
	test(*r->s3, 11, 0, *arr[1], 2, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 2, 1, 0);
	test(*r->s3, 11, 0, *arr[1], 2, 2, 0);
	test(*r->s3, 11, 0, *arr[1], 2, 3, 0);
	test(*r->s3, 11, 0, *arr[1], 2, 4, 0);
	test(*r->s3, 11, 0, *arr[1], 4, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 4, 1, 0);
	test(*r->s3, 11, 0, *arr[1], 4, 2, 0);
	test(*r->s3, 11, 0, *arr[1], 5, 0, 0);
	test(*r->s3, 11, 0, *arr[1], 5, 1, 0);
	test(*r->s3, 11, 0, *arr[1], 6, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 1, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 5, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 9, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 10, 0);
	test(*r->s3, 11, 0, *arr[2], 0, 11, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 1, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 4, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 8, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 9, 0);
	test(*r->s3, 11, 0, *arr[2], 1, 10, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 1, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 2, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 4, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 5, 0);
	test(*r->s3, 11, 0, *arr[2], 5, 6, 0);
	test(*r->s3, 11, 0, *arr[2], 9, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 9, 1, 0);
	test(*r->s3, 11, 0, *arr[2], 9, 2, 0);
	test(*r->s3, 11, 0, *arr[2], 10, 0, 0);
	test(*r->s3, 11, 0, *arr[2], 10, 1, 0);
	test(*r->s3, 11, 0, *arr[2], 11, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 1, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 10, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 19, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 20, 0);
	test(*r->s3, 11, 0, *arr[3], 0, 21, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 1, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 9, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 18, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 19, 0);
	test(*r->s3, 11, 0, *arr[3], 1, 20, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 1, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 5, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 9, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 10, 0);
	test(*r->s3, 11, 0, *arr[3], 10, 11, 0);
	test(*r->s3, 11, 0, *arr[3], 19, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 19, 1, 0);
	test(*r->s3, 11, 0, *arr[3], 19, 2, 0);
	test(*r->s3, 11, 0, *arr[3], 20, 0, 0);
	test(*r->s3, 11, 0, *arr[3], 20, 1, 0);
	test(*r->s3, 11, 0, *arr[3], 21, 0, 0);
}

template <class S, class U>
void
test37(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 0, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 0, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 0, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 0, 1, -1);
	test(*r->s4, 0, 0, *arr[1], 0, 2, -2);
	test(*r->s4, 0, 0, *arr[1], 0, 4, -4);
	test(*r->s4, 0, 0, *arr[1], 0, 5, -5);
	test(*r->s4, 0, 0, *arr[1], 0, 6, -5);
	test(*r->s4, 0, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 0, *arr[1], 1, 2, -2);
	test(*r->s4, 0, 0, *arr[1], 1, 3, -3);
	test(*r->s4, 0, 0, *arr[1], 1, 4, -4);
	test(*r->s4, 0, 0, *arr[1], 1, 5, -4);
	test(*r->s4, 0, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 2, 1, -1);
	test(*r->s4, 0, 0, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 0, *arr[1], 2, 3, -3);
	test(*r->s4, 0, 0, *arr[1], 2, 4, -3);
	test(*r->s4, 0, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 4, 1, -1);
	test(*r->s4, 0, 0, *arr[1], 4, 2, -1);
	test(*r->s4, 0, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 0, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 0, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 0, 1, -1);
	test(*r->s4, 0, 0, *arr[2], 0, 5, -5);
	test(*r->s4, 0, 0, *arr[2], 0, 9, -9);
	test(*r->s4, 0, 0, *arr[2], 0, 10, -10);
	test(*r->s4, 0, 0, *arr[2], 0, 11, -10);
	test(*r->s4, 0, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 0, *arr[2], 1, 4, -4);
	test(*r->s4, 0, 0, *arr[2], 1, 8, -8);
	test(*r->s4, 0, 0, *arr[2], 1, 9, -9);
	test(*r->s4, 0, 0, *arr[2], 1, 10, -9);
	test(*r->s4, 0, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 5, 1, -1);
	test(*r->s4, 0, 0, *arr[2], 5, 2, -2);
	test(*r->s4, 0, 0, *arr[2], 5, 4, -4);
	test(*r->s4, 0, 0, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 0, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 9, 1, -1);
	test(*r->s4, 0, 0, *arr[2], 9, 2, -1);
	test(*r->s4, 0, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 0, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 0, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 0, 1, -1);
	test(*r->s4, 0, 0, *arr[3], 0, 10, -10);
	test(*r->s4, 0, 0, *arr[3], 0, 19, -19);
	test(*r->s4, 0, 0, *arr[3], 0, 20, -20);
	test(*r->s4, 0, 0, *arr[3], 0, 21, -20);
	test(*r->s4, 0, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 0, *arr[3], 1, 9, -9);
	test(*r->s4, 0, 0, *arr[3], 1, 18, -18);
	test(*r->s4, 0, 0, *arr[3], 1, 19, -19);
	test(*r->s4, 0, 0, *arr[3], 1, 20, -19);
	test(*r->s4, 0, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 10, 1, -1);
	test(*r->s4, 0, 0, *arr[3], 10, 5, -5);
	test(*r->s4, 0, 0, *arr[3], 10, 9, -9);
	test(*r->s4, 0, 0, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 0, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 19, 1, -1);
	test(*r->s4, 0, 0, *arr[3], 19, 2, -1);
	test(*r->s4, 0, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 0, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 0, 0, *arr[3], 21, 0, 0);
	test(*r->s4, 0, 1, *arr[0], 0, 0, 1);
	test(*r->s4, 0, 1, *arr[0], 0, 1, 1);
	test(*r->s4, 0, 1, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 1, *arr[1], 0, 0, 1);
	test(*r->s4, 0, 1, *arr[1], 0, 1, 0);
	test(*r->s4, 0, 1, *arr[1], 0, 2, -1);
	test(*r->s4, 0, 1, *arr[1], 0, 4, -3);
	test(*r->s4, 0, 1, *arr[1], 0, 5, -4);
	test(*r->s4, 0, 1, *arr[1], 0, 6, -4);
	test(*r->s4, 0, 1, *arr[1], 1, 0, 1);
	test(*r->s4, 0, 1, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 1, *arr[1], 1, 2, -1);
	test(*r->s4, 0, 1, *arr[1], 1, 3, -1);
	test(*r->s4, 0, 1, *arr[1], 1, 4, -1);
	test(*r->s4, 0, 1, *arr[1], 1, 5, -1);
	test(*r->s4, 0, 1, *arr[1], 2, 0, 1);
	test(*r->s4, 0, 1, *arr[1], 2, 1, -2);
	test(*r->s4, 0, 1, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 1, *arr[1], 2, 3, -2);
	test(*r->s4, 0, 1, *arr[1], 2, 4, -2);
	test(*r->s4, 0, 1, *arr[1], 4, 0, 1);
	test(*r->s4, 0, 1, *arr[1], 4, 1, -4);
	test(*r->s4, 0, 1, *arr[1], 4, 2, -4);
	test(*r->s4, 0, 1, *arr[1], 5, 0, 1);
	test(*r->s4, 0, 1, *arr[1], 5, 1, 1);
	test(*r->s4, 0, 1, *arr[1], 6, 0, 0);
}

template <class S, class U>
void
test38(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 0, 1, *arr[2], 0, 0, 1);
	test(*r->s4, 0, 1, *arr[2], 0, 1, 0);
	test(*r->s4, 0, 1, *arr[2], 0, 5, -4);
	test(*r->s4, 0, 1, *arr[2], 0, 9, -8);
	test(*r->s4, 0, 1, *arr[2], 0, 10, -9);
	test(*r->s4, 0, 1, *arr[2], 0, 11, -9);
	test(*r->s4, 0, 1, *arr[2], 1, 0, 1);
	test(*r->s4, 0, 1, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 1, *arr[2], 1, 4, -1);
	test(*r->s4, 0, 1, *arr[2], 1, 8, -1);
	test(*r->s4, 0, 1, *arr[2], 1, 9, -1);
	test(*r->s4, 0, 1, *arr[2], 1, 10, -1);
	test(*r->s4, 0, 1, *arr[2], 5, 0, 1);
	test(*r->s4, 0, 1, *arr[2], 5, 1, -5);
	test(*r->s4, 0, 1, *arr[2], 5, 2, -5);
	test(*r->s4, 0, 1, *arr[2], 5, 4, -5);
	test(*r->s4, 0, 1, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 1, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 1, *arr[2], 9, 0, 1);
	test(*r->s4, 0, 1, *arr[2], 9, 1, -9);
	test(*r->s4, 0, 1, *arr[2], 9, 2, -9);
	test(*r->s4, 0, 1, *arr[2], 10, 0, 1);
	test(*r->s4, 0, 1, *arr[2], 10, 1, 1);
	test(*r->s4, 0, 1, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 1, *arr[3], 0, 0, 1);
	test(*r->s4, 0, 1, *arr[3], 0, 1, 0);
	test(*r->s4, 0, 1, *arr[3], 0, 10, -9);
	test(*r->s4, 0, 1, *arr[3], 0, 19, -18);
	test(*r->s4, 0, 1, *arr[3], 0, 20, -19);
	test(*r->s4, 0, 1, *arr[3], 0, 21, -19);
	test(*r->s4, 0, 1, *arr[3], 1, 0, 1);
	test(*r->s4, 0, 1, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 1, *arr[3], 1, 9, -1);
	test(*r->s4, 0, 1, *arr[3], 1, 18, -1);
	test(*r->s4, 0, 1, *arr[3], 1, 19, -1);
	test(*r->s4, 0, 1, *arr[3], 1, 20, -1);
	test(*r->s4, 0, 1, *arr[3], 10, 0, 1);
	test(*r->s4, 0, 1, *arr[3], 10, 1, -10);
	test(*r->s4, 0, 1, *arr[3], 10, 5, -10);
	test(*r->s4, 0, 1, *arr[3], 10, 9, -10);
	test(*r->s4, 0, 1, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 1, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 1, *arr[3], 19, 0, 1);
	test(*r->s4, 0, 1, *arr[3], 19, 1, -19);
	test(*r->s4, 0, 1, *arr[3], 19, 2, -19);
	test(*r->s4, 0, 1, *arr[3], 20, 0, 1);
	test(*r->s4, 0, 1, *arr[3], 20, 1, 1);
	test(*r->s4, 0, 1, *arr[3], 21, 0, 0);
	test(*r->s4, 0, 10, *arr[0], 0, 0, 10);
	test(*r->s4, 0, 10, *arr[0], 0, 1, 10);
	test(*r->s4, 0, 10, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 10, *arr[1], 0, 0, 10);
	test(*r->s4, 0, 10, *arr[1], 0, 1, 9);
	test(*r->s4, 0, 10, *arr[1], 0, 2, 8);
	test(*r->s4, 0, 10, *arr[1], 0, 4, 6);
	test(*r->s4, 0, 10, *arr[1], 0, 5, 5);
	test(*r->s4, 0, 10, *arr[1], 0, 6, 5);
	test(*r->s4, 0, 10, *arr[1], 1, 0, 10);
	test(*r->s4, 0, 10, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 10, *arr[1], 1, 2, -1);
	test(*r->s4, 0, 10, *arr[1], 1, 3, -1);
	test(*r->s4, 0, 10, *arr[1], 1, 4, -1);
	test(*r->s4, 0, 10, *arr[1], 1, 5, -1);
	test(*r->s4, 0, 10, *arr[1], 2, 0, 10);
	test(*r->s4, 0, 10, *arr[1], 2, 1, -2);
	test(*r->s4, 0, 10, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 10, *arr[1], 2, 3, -2);
	test(*r->s4, 0, 10, *arr[1], 2, 4, -2);
	test(*r->s4, 0, 10, *arr[1], 4, 0, 10);
	test(*r->s4, 0, 10, *arr[1], 4, 1, -4);
	test(*r->s4, 0, 10, *arr[1], 4, 2, -4);
	test(*r->s4, 0, 10, *arr[1], 5, 0, 10);
	test(*r->s4, 0, 10, *arr[1], 5, 1, 10);
	test(*r->s4, 0, 10, *arr[1], 6, 0, 0);
	test(*r->s4, 0, 10, *arr[2], 0, 0, 10);
	test(*r->s4, 0, 10, *arr[2], 0, 1, 9);
	test(*r->s4, 0, 10, *arr[2], 0, 5, 5);
	test(*r->s4, 0, 10, *arr[2], 0, 9, 1);
	test(*r->s4, 0, 10, *arr[2], 0, 10, 0);
	test(*r->s4, 0, 10, *arr[2], 0, 11, 0);
	test(*r->s4, 0, 10, *arr[2], 1, 0, 10);
	test(*r->s4, 0, 10, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 10, *arr[2], 1, 4, -1);
	test(*r->s4, 0, 10, *arr[2], 1, 8, -1);
	test(*r->s4, 0, 10, *arr[2], 1, 9, -1);
	test(*r->s4, 0, 10, *arr[2], 1, 10, -1);
	test(*r->s4, 0, 10, *arr[2], 5, 0, 10);
	test(*r->s4, 0, 10, *arr[2], 5, 1, -5);
	test(*r->s4, 0, 10, *arr[2], 5, 2, -5);
	test(*r->s4, 0, 10, *arr[2], 5, 4, -5);
	test(*r->s4, 0, 10, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 10, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 10, *arr[2], 9, 0, 10);
	test(*r->s4, 0, 10, *arr[2], 9, 1, -9);
	test(*r->s4, 0, 10, *arr[2], 9, 2, -9);
	test(*r->s4, 0, 10, *arr[2], 10, 0, 10);
	test(*r->s4, 0, 10, *arr[2], 10, 1, 10);
	test(*r->s4, 0, 10, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 10, *arr[3], 0, 0, 10);
	test(*r->s4, 0, 10, *arr[3], 0, 1, 9);
}

template <class S, class U>
void
test39(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 0, 10, *arr[3], 0, 10, 0);
	test(*r->s4, 0, 10, *arr[3], 0, 19, -9);
	test(*r->s4, 0, 10, *arr[3], 0, 20, -10);
	test(*r->s4, 0, 10, *arr[3], 0, 21, -10);
	test(*r->s4, 0, 10, *arr[3], 1, 0, 10);
	test(*r->s4, 0, 10, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 10, *arr[3], 1, 9, -1);
	test(*r->s4, 0, 10, *arr[3], 1, 18, -1);
	test(*r->s4, 0, 10, *arr[3], 1, 19, -1);
	test(*r->s4, 0, 10, *arr[3], 1, 20, -1);
	test(*r->s4, 0, 10, *arr[3], 10, 0, 10);
	test(*r->s4, 0, 10, *arr[3], 10, 1, -10);
	test(*r->s4, 0, 10, *arr[3], 10, 5, -10);
	test(*r->s4, 0, 10, *arr[3], 10, 9, -10);
	test(*r->s4, 0, 10, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 10, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 10, *arr[3], 19, 0, 10);
	test(*r->s4, 0, 10, *arr[3], 19, 1, -19);
	test(*r->s4, 0, 10, *arr[3], 19, 2, -19);
	test(*r->s4, 0, 10, *arr[3], 20, 0, 10);
	test(*r->s4, 0, 10, *arr[3], 20, 1, 10);
	test(*r->s4, 0, 10, *arr[3], 21, 0, 0);
	test(*r->s4, 0, 19, *arr[0], 0, 0, 19);
	test(*r->s4, 0, 19, *arr[0], 0, 1, 19);
	test(*r->s4, 0, 19, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 19, *arr[1], 0, 0, 19);
	test(*r->s4, 0, 19, *arr[1], 0, 1, 18);
	test(*r->s4, 0, 19, *arr[1], 0, 2, 17);
	test(*r->s4, 0, 19, *arr[1], 0, 4, 15);
	test(*r->s4, 0, 19, *arr[1], 0, 5, 14);
	test(*r->s4, 0, 19, *arr[1], 0, 6, 14);
	test(*r->s4, 0, 19, *arr[1], 1, 0, 19);
	test(*r->s4, 0, 19, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 19, *arr[1], 1, 2, -1);
	test(*r->s4, 0, 19, *arr[1], 1, 3, -1);
	test(*r->s4, 0, 19, *arr[1], 1, 4, -1);
	test(*r->s4, 0, 19, *arr[1], 1, 5, -1);
	test(*r->s4, 0, 19, *arr[1], 2, 0, 19);
	test(*r->s4, 0, 19, *arr[1], 2, 1, -2);
	test(*r->s4, 0, 19, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 19, *arr[1], 2, 3, -2);
	test(*r->s4, 0, 19, *arr[1], 2, 4, -2);
	test(*r->s4, 0, 19, *arr[1], 4, 0, 19);
	test(*r->s4, 0, 19, *arr[1], 4, 1, -4);
	test(*r->s4, 0, 19, *arr[1], 4, 2, -4);
	test(*r->s4, 0, 19, *arr[1], 5, 0, 19);
	test(*r->s4, 0, 19, *arr[1], 5, 1, 19);
	test(*r->s4, 0, 19, *arr[1], 6, 0, 0);
	test(*r->s4, 0, 19, *arr[2], 0, 0, 19);
	test(*r->s4, 0, 19, *arr[2], 0, 1, 18);
	test(*r->s4, 0, 19, *arr[2], 0, 5, 14);
	test(*r->s4, 0, 19, *arr[2], 0, 9, 10);
	test(*r->s4, 0, 19, *arr[2], 0, 10, 9);
	test(*r->s4, 0, 19, *arr[2], 0, 11, 9);
	test(*r->s4, 0, 19, *arr[2], 1, 0, 19);
	test(*r->s4, 0, 19, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 19, *arr[2], 1, 4, -1);
	test(*r->s4, 0, 19, *arr[2], 1, 8, -1);
	test(*r->s4, 0, 19, *arr[2], 1, 9, -1);
	test(*r->s4, 0, 19, *arr[2], 1, 10, -1);
	test(*r->s4, 0, 19, *arr[2], 5, 0, 19);
	test(*r->s4, 0, 19, *arr[2], 5, 1, -5);
	test(*r->s4, 0, 19, *arr[2], 5, 2, -5);
	test(*r->s4, 0, 19, *arr[2], 5, 4, -5);
	test(*r->s4, 0, 19, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 19, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 19, *arr[2], 9, 0, 19);
	test(*r->s4, 0, 19, *arr[2], 9, 1, -9);
	test(*r->s4, 0, 19, *arr[2], 9, 2, -9);
	test(*r->s4, 0, 19, *arr[2], 10, 0, 19);
	test(*r->s4, 0, 19, *arr[2], 10, 1, 19);
	test(*r->s4, 0, 19, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 19, *arr[3], 0, 0, 19);
	test(*r->s4, 0, 19, *arr[3], 0, 1, 18);
	test(*r->s4, 0, 19, *arr[3], 0, 10, 9);
	test(*r->s4, 0, 19, *arr[3], 0, 19, 0);
	test(*r->s4, 0, 19, *arr[3], 0, 20, -1);
	test(*r->s4, 0, 19, *arr[3], 0, 21, -1);
	test(*r->s4, 0, 19, *arr[3], 1, 0, 19);
	test(*r->s4, 0, 19, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 19, *arr[3], 1, 9, -1);
	test(*r->s4, 0, 19, *arr[3], 1, 18, -1);
	test(*r->s4, 0, 19, *arr[3], 1, 19, -1);
	test(*r->s4, 0, 19, *arr[3], 1, 20, -1);
	test(*r->s4, 0, 19, *arr[3], 10, 0, 19);
	test(*r->s4, 0, 19, *arr[3], 10, 1, -10);
	test(*r->s4, 0, 19, *arr[3], 10, 5, -10);
	test(*r->s4, 0, 19, *arr[3], 10, 9, -10);
	test(*r->s4, 0, 19, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 19, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 19, *arr[3], 19, 0, 19);
	test(*r->s4, 0, 19, *arr[3], 19, 1, -19);
	test(*r->s4, 0, 19, *arr[3], 19, 2, -19);
	test(*r->s4, 0, 19, *arr[3], 20, 0, 19);
	test(*r->s4, 0, 19, *arr[3], 20, 1, 19);
	test(*r->s4, 0, 19, *arr[3], 21, 0, 0);
	test(*r->s4, 0, 20, *arr[0], 0, 0, 20);
	test(*r->s4, 0, 20, *arr[0], 0, 1, 20);
	test(*r->s4, 0, 20, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 20, *arr[1], 0, 0, 20);
}

template <class S, class U>
void
test40(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 0, 20, *arr[1], 0, 1, 19);
	test(*r->s4, 0, 20, *arr[1], 0, 2, 18);
	test(*r->s4, 0, 20, *arr[1], 0, 4, 16);
	test(*r->s4, 0, 20, *arr[1], 0, 5, 15);
	test(*r->s4, 0, 20, *arr[1], 0, 6, 15);
	test(*r->s4, 0, 20, *arr[1], 1, 0, 20);
	test(*r->s4, 0, 20, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 20, *arr[1], 1, 2, -1);
	test(*r->s4, 0, 20, *arr[1], 1, 3, -1);
	test(*r->s4, 0, 20, *arr[1], 1, 4, -1);
	test(*r->s4, 0, 20, *arr[1], 1, 5, -1);
	test(*r->s4, 0, 20, *arr[1], 2, 0, 20);
	test(*r->s4, 0, 20, *arr[1], 2, 1, -2);
	test(*r->s4, 0, 20, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 20, *arr[1], 2, 3, -2);
	test(*r->s4, 0, 20, *arr[1], 2, 4, -2);
	test(*r->s4, 0, 20, *arr[1], 4, 0, 20);
	test(*r->s4, 0, 20, *arr[1], 4, 1, -4);
	test(*r->s4, 0, 20, *arr[1], 4, 2, -4);
	test(*r->s4, 0, 20, *arr[1], 5, 0, 20);
	test(*r->s4, 0, 20, *arr[1], 5, 1, 20);
	test(*r->s4, 0, 20, *arr[1], 6, 0, 0);
	test(*r->s4, 0, 20, *arr[2], 0, 0, 20);
	test(*r->s4, 0, 20, *arr[2], 0, 1, 19);
	test(*r->s4, 0, 20, *arr[2], 0, 5, 15);
	test(*r->s4, 0, 20, *arr[2], 0, 9, 11);
	test(*r->s4, 0, 20, *arr[2], 0, 10, 10);
	test(*r->s4, 0, 20, *arr[2], 0, 11, 10);
	test(*r->s4, 0, 20, *arr[2], 1, 0, 20);
	test(*r->s4, 0, 20, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 20, *arr[2], 1, 4, -1);
	test(*r->s4, 0, 20, *arr[2], 1, 8, -1);
	test(*r->s4, 0, 20, *arr[2], 1, 9, -1);
	test(*r->s4, 0, 20, *arr[2], 1, 10, -1);
	test(*r->s4, 0, 20, *arr[2], 5, 0, 20);
	test(*r->s4, 0, 20, *arr[2], 5, 1, -5);
	test(*r->s4, 0, 20, *arr[2], 5, 2, -5);
	test(*r->s4, 0, 20, *arr[2], 5, 4, -5);
	test(*r->s4, 0, 20, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 20, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 20, *arr[2], 9, 0, 20);
	test(*r->s4, 0, 20, *arr[2], 9, 1, -9);
	test(*r->s4, 0, 20, *arr[2], 9, 2, -9);
	test(*r->s4, 0, 20, *arr[2], 10, 0, 20);
	test(*r->s4, 0, 20, *arr[2], 10, 1, 20);
	test(*r->s4, 0, 20, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 20, *arr[3], 0, 0, 20);
	test(*r->s4, 0, 20, *arr[3], 0, 1, 19);
	test(*r->s4, 0, 20, *arr[3], 0, 10, 10);
	test(*r->s4, 0, 20, *arr[3], 0, 19, 1);
	test(*r->s4, 0, 20, *arr[3], 0, 20, 0);
	test(*r->s4, 0, 20, *arr[3], 0, 21, 0);
	test(*r->s4, 0, 20, *arr[3], 1, 0, 20);
	test(*r->s4, 0, 20, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 20, *arr[3], 1, 9, -1);
	test(*r->s4, 0, 20, *arr[3], 1, 18, -1);
	test(*r->s4, 0, 20, *arr[3], 1, 19, -1);
	test(*r->s4, 0, 20, *arr[3], 1, 20, -1);
	test(*r->s4, 0, 20, *arr[3], 10, 0, 20);
	test(*r->s4, 0, 20, *arr[3], 10, 1, -10);
	test(*r->s4, 0, 20, *arr[3], 10, 5, -10);
	test(*r->s4, 0, 20, *arr[3], 10, 9, -10);
	test(*r->s4, 0, 20, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 20, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 20, *arr[3], 19, 0, 20);
	test(*r->s4, 0, 20, *arr[3], 19, 1, -19);
	test(*r->s4, 0, 20, *arr[3], 19, 2, -19);
	test(*r->s4, 0, 20, *arr[3], 20, 0, 20);
	test(*r->s4, 0, 20, *arr[3], 20, 1, 20);
	test(*r->s4, 0, 20, *arr[3], 21, 0, 0);
	test(*r->s4, 0, 21, *arr[0], 0, 0, 20);
	test(*r->s4, 0, 21, *arr[0], 0, 1, 20);
	test(*r->s4, 0, 21, *arr[0], 1, 0, 0);
	test(*r->s4, 0, 21, *arr[1], 0, 0, 20);
	test(*r->s4, 0, 21, *arr[1], 0, 1, 19);
	test(*r->s4, 0, 21, *arr[1], 0, 2, 18);
	test(*r->s4, 0, 21, *arr[1], 0, 4, 16);
	test(*r->s4, 0, 21, *arr[1], 0, 5, 15);
	test(*r->s4, 0, 21, *arr[1], 0, 6, 15);
	test(*r->s4, 0, 21, *arr[1], 1, 0, 20);
	test(*r->s4, 0, 21, *arr[1], 1, 1, -1);
	test(*r->s4, 0, 21, *arr[1], 1, 2, -1);
	test(*r->s4, 0, 21, *arr[1], 1, 3, -1);
	test(*r->s4, 0, 21, *arr[1], 1, 4, -1);
	test(*r->s4, 0, 21, *arr[1], 1, 5, -1);
	test(*r->s4, 0, 21, *arr[1], 2, 0, 20);
	test(*r->s4, 0, 21, *arr[1], 2, 1, -2);
	test(*r->s4, 0, 21, *arr[1], 2, 2, -2);
	test(*r->s4, 0, 21, *arr[1], 2, 3, -2);
	test(*r->s4, 0, 21, *arr[1], 2, 4, -2);
	test(*r->s4, 0, 21, *arr[1], 4, 0, 20);
	test(*r->s4, 0, 21, *arr[1], 4, 1, -4);
	test(*r->s4, 0, 21, *arr[1], 4, 2, -4);
	test(*r->s4, 0, 21, *arr[1], 5, 0, 20);
	test(*r->s4, 0, 21, *arr[1], 5, 1, 20);
	test(*r->s4, 0, 21, *arr[1], 6, 0, 0);
	test(*r->s4, 0, 21, *arr[2], 0, 0, 20);
	test(*r->s4, 0, 21, *arr[2], 0, 1, 19);
	test(*r->s4, 0, 21, *arr[2], 0, 5, 15);
	test(*r->s4, 0, 21, *arr[2], 0, 9, 11);
}

template <class S, class U>
void
test41(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 0, 21, *arr[2], 0, 10, 10);
	test(*r->s4, 0, 21, *arr[2], 0, 11, 10);
	test(*r->s4, 0, 21, *arr[2], 1, 0, 20);
	test(*r->s4, 0, 21, *arr[2], 1, 1, -1);
	test(*r->s4, 0, 21, *arr[2], 1, 4, -1);
	test(*r->s4, 0, 21, *arr[2], 1, 8, -1);
	test(*r->s4, 0, 21, *arr[2], 1, 9, -1);
	test(*r->s4, 0, 21, *arr[2], 1, 10, -1);
	test(*r->s4, 0, 21, *arr[2], 5, 0, 20);
	test(*r->s4, 0, 21, *arr[2], 5, 1, -5);
	test(*r->s4, 0, 21, *arr[2], 5, 2, -5);
	test(*r->s4, 0, 21, *arr[2], 5, 4, -5);
	test(*r->s4, 0, 21, *arr[2], 5, 5, -5);
	test(*r->s4, 0, 21, *arr[2], 5, 6, -5);
	test(*r->s4, 0, 21, *arr[2], 9, 0, 20);
	test(*r->s4, 0, 21, *arr[2], 9, 1, -9);
	test(*r->s4, 0, 21, *arr[2], 9, 2, -9);
	test(*r->s4, 0, 21, *arr[2], 10, 0, 20);
	test(*r->s4, 0, 21, *arr[2], 10, 1, 20);
	test(*r->s4, 0, 21, *arr[2], 11, 0, 0);
	test(*r->s4, 0, 21, *arr[3], 0, 0, 20);
	test(*r->s4, 0, 21, *arr[3], 0, 1, 19);
	test(*r->s4, 0, 21, *arr[3], 0, 10, 10);
	test(*r->s4, 0, 21, *arr[3], 0, 19, 1);
	test(*r->s4, 0, 21, *arr[3], 0, 20, 0);
	test(*r->s4, 0, 21, *arr[3], 0, 21, 0);
	test(*r->s4, 0, 21, *arr[3], 1, 0, 20);
	test(*r->s4, 0, 21, *arr[3], 1, 1, -1);
	test(*r->s4, 0, 21, *arr[3], 1, 9, -1);
	test(*r->s4, 0, 21, *arr[3], 1, 18, -1);
	test(*r->s4, 0, 21, *arr[3], 1, 19, -1);
	test(*r->s4, 0, 21, *arr[3], 1, 20, -1);
	test(*r->s4, 0, 21, *arr[3], 10, 0, 20);
	test(*r->s4, 0, 21, *arr[3], 10, 1, -10);
	test(*r->s4, 0, 21, *arr[3], 10, 5, -10);
	test(*r->s4, 0, 21, *arr[3], 10, 9, -10);
	test(*r->s4, 0, 21, *arr[3], 10, 10, -10);
	test(*r->s4, 0, 21, *arr[3], 10, 11, -10);
	test(*r->s4, 0, 21, *arr[3], 19, 0, 20);
	test(*r->s4, 0, 21, *arr[3], 19, 1, -19);
	test(*r->s4, 0, 21, *arr[3], 19, 2, -19);
	test(*r->s4, 0, 21, *arr[3], 20, 0, 20);
	test(*r->s4, 0, 21, *arr[3], 20, 1, 20);
	test(*r->s4, 0, 21, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 1, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 1, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 0, 1, -1);
	test(*r->s4, 1, 0, *arr[1], 0, 2, -2);
	test(*r->s4, 1, 0, *arr[1], 0, 4, -4);
	test(*r->s4, 1, 0, *arr[1], 0, 5, -5);
	test(*r->s4, 1, 0, *arr[1], 0, 6, -5);
	test(*r->s4, 1, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 1, 1, -1);
	test(*r->s4, 1, 0, *arr[1], 1, 2, -2);
	test(*r->s4, 1, 0, *arr[1], 1, 3, -3);
	test(*r->s4, 1, 0, *arr[1], 1, 4, -4);
	test(*r->s4, 1, 0, *arr[1], 1, 5, -4);
	test(*r->s4, 1, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 0, *arr[1], 2, 2, -2);
	test(*r->s4, 1, 0, *arr[1], 2, 3, -3);
	test(*r->s4, 1, 0, *arr[1], 2, 4, -3);
	test(*r->s4, 1, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 4, 1, -1);
	test(*r->s4, 1, 0, *arr[1], 4, 2, -1);
	test(*r->s4, 1, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 1, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 1, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 0, 1, -1);
	test(*r->s4, 1, 0, *arr[2], 0, 5, -5);
	test(*r->s4, 1, 0, *arr[2], 0, 9, -9);
	test(*r->s4, 1, 0, *arr[2], 0, 10, -10);
	test(*r->s4, 1, 0, *arr[2], 0, 11, -10);
	test(*r->s4, 1, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 1, 1, -1);
	test(*r->s4, 1, 0, *arr[2], 1, 4, -4);
	test(*r->s4, 1, 0, *arr[2], 1, 8, -8);
	test(*r->s4, 1, 0, *arr[2], 1, 9, -9);
	test(*r->s4, 1, 0, *arr[2], 1, 10, -9);
	test(*r->s4, 1, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 5, 1, -1);
	test(*r->s4, 1, 0, *arr[2], 5, 2, -2);
	test(*r->s4, 1, 0, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 0, *arr[2], 5, 5, -5);
	test(*r->s4, 1, 0, *arr[2], 5, 6, -5);
	test(*r->s4, 1, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 9, 1, -1);
	test(*r->s4, 1, 0, *arr[2], 9, 2, -1);
	test(*r->s4, 1, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 1, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 1, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 0, 1, -1);
	test(*r->s4, 1, 0, *arr[3], 0, 10, -10);
	test(*r->s4, 1, 0, *arr[3], 0, 19, -19);
	test(*r->s4, 1, 0, *arr[3], 0, 20, -20);
	test(*r->s4, 1, 0, *arr[3], 0, 21, -20);
}

template <class S, class U>
void
test42(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 1, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 1, 1, -1);
	test(*r->s4, 1, 0, *arr[3], 1, 9, -9);
	test(*r->s4, 1, 0, *arr[3], 1, 18, -18);
	test(*r->s4, 1, 0, *arr[3], 1, 19, -19);
	test(*r->s4, 1, 0, *arr[3], 1, 20, -19);
	test(*r->s4, 1, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 10, 1, -1);
	test(*r->s4, 1, 0, *arr[3], 10, 5, -5);
	test(*r->s4, 1, 0, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 0, *arr[3], 10, 10, -10);
	test(*r->s4, 1, 0, *arr[3], 10, 11, -10);
	test(*r->s4, 1, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 19, 1, -1);
	test(*r->s4, 1, 0, *arr[3], 19, 2, -1);
	test(*r->s4, 1, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 1, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 1, 0, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 1, *arr[0], 0, 0, 1);
	test(*r->s4, 1, 1, *arr[0], 0, 1, 1);
	test(*r->s4, 1, 1, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 1, *arr[1], 0, 0, 1);
	test(*r->s4, 1, 1, *arr[1], 0, 1, 1);
	test(*r->s4, 1, 1, *arr[1], 0, 2, 1);
	test(*r->s4, 1, 1, *arr[1], 0, 4, 1);
	test(*r->s4, 1, 1, *arr[1], 0, 5, 1);
	test(*r->s4, 1, 1, *arr[1], 0, 6, 1);
	test(*r->s4, 1, 1, *arr[1], 1, 0, 1);
	test(*r->s4, 1, 1, *arr[1], 1, 1, 0);
	test(*r->s4, 1, 1, *arr[1], 1, 2, -1);
	test(*r->s4, 1, 1, *arr[1], 1, 3, -2);
	test(*r->s4, 1, 1, *arr[1], 1, 4, -3);
	test(*r->s4, 1, 1, *arr[1], 1, 5, -3);
	test(*r->s4, 1, 1, *arr[1], 2, 0, 1);
	test(*r->s4, 1, 1, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 1, *arr[1], 2, 2, -1);
	test(*r->s4, 1, 1, *arr[1], 2, 3, -1);
	test(*r->s4, 1, 1, *arr[1], 2, 4, -1);
	test(*r->s4, 1, 1, *arr[1], 4, 0, 1);
	test(*r->s4, 1, 1, *arr[1], 4, 1, -3);
	test(*r->s4, 1, 1, *arr[1], 4, 2, -3);
	test(*r->s4, 1, 1, *arr[1], 5, 0, 1);
	test(*r->s4, 1, 1, *arr[1], 5, 1, 1);
	test(*r->s4, 1, 1, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 1, *arr[2], 0, 0, 1);
	test(*r->s4, 1, 1, *arr[2], 0, 1, 1);
	test(*r->s4, 1, 1, *arr[2], 0, 5, 1);
	test(*r->s4, 1, 1, *arr[2], 0, 9, 1);
	test(*r->s4, 1, 1, *arr[2], 0, 10, 1);
	test(*r->s4, 1, 1, *arr[2], 0, 11, 1);
	test(*r->s4, 1, 1, *arr[2], 1, 0, 1);
	test(*r->s4, 1, 1, *arr[2], 1, 1, 0);
	test(*r->s4, 1, 1, *arr[2], 1, 4, -3);
	test(*r->s4, 1, 1, *arr[2], 1, 8, -7);
	test(*r->s4, 1, 1, *arr[2], 1, 9, -8);
	test(*r->s4, 1, 1, *arr[2], 1, 10, -8);
	test(*r->s4, 1, 1, *arr[2], 5, 0, 1);
	test(*r->s4, 1, 1, *arr[2], 5, 1, -4);
	test(*r->s4, 1, 1, *arr[2], 5, 2, -4);
	test(*r->s4, 1, 1, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 1, *arr[2], 5, 5, -4);
	test(*r->s4, 1, 1, *arr[2], 5, 6, -4);
	test(*r->s4, 1, 1, *arr[2], 9, 0, 1);
	test(*r->s4, 1, 1, *arr[2], 9, 1, -8);
	test(*r->s4, 1, 1, *arr[2], 9, 2, -8);
	test(*r->s4, 1, 1, *arr[2], 10, 0, 1);
	test(*r->s4, 1, 1, *arr[2], 10, 1, 1);
	test(*r->s4, 1, 1, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 1, *arr[3], 0, 0, 1);
	test(*r->s4, 1, 1, *arr[3], 0, 1, 1);
	test(*r->s4, 1, 1, *arr[3], 0, 10, 1);
	test(*r->s4, 1, 1, *arr[3], 0, 19, 1);
	test(*r->s4, 1, 1, *arr[3], 0, 20, 1);
	test(*r->s4, 1, 1, *arr[3], 0, 21, 1);
	test(*r->s4, 1, 1, *arr[3], 1, 0, 1);
	test(*r->s4, 1, 1, *arr[3], 1, 1, 0);
	test(*r->s4, 1, 1, *arr[3], 1, 9, -8);
	test(*r->s4, 1, 1, *arr[3], 1, 18, -17);
	test(*r->s4, 1, 1, *arr[3], 1, 19, -18);
	test(*r->s4, 1, 1, *arr[3], 1, 20, -18);
	test(*r->s4, 1, 1, *arr[3], 10, 0, 1);
	test(*r->s4, 1, 1, *arr[3], 10, 1, -9);
	test(*r->s4, 1, 1, *arr[3], 10, 5, -9);
	test(*r->s4, 1, 1, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 1, *arr[3], 10, 10, -9);
	test(*r->s4, 1, 1, *arr[3], 10, 11, -9);
	test(*r->s4, 1, 1, *arr[3], 19, 0, 1);
	test(*r->s4, 1, 1, *arr[3], 19, 1, -18);
	test(*r->s4, 1, 1, *arr[3], 19, 2, -18);
	test(*r->s4, 1, 1, *arr[3], 20, 0, 1);
	test(*r->s4, 1, 1, *arr[3], 20, 1, 1);
	test(*r->s4, 1, 1, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 9, *arr[0], 0, 0, 9);
	test(*r->s4, 1, 9, *arr[0], 0, 1, 9);
	test(*r->s4, 1, 9, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 9, *arr[1], 0, 0, 9);
	test(*r->s4, 1, 9, *arr[1], 0, 1, 1);
	test(*r->s4, 1, 9, *arr[1], 0, 2, 1);
	test(*r->s4, 1, 9, *arr[1], 0, 4, 1);
	test(*r->s4, 1, 9, *arr[1], 0, 5, 1);
}

template <class S, class U>
void
test43(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 1, 9, *arr[1], 0, 6, 1);
	test(*r->s4, 1, 9, *arr[1], 1, 0, 9);
	test(*r->s4, 1, 9, *arr[1], 1, 1, 8);
	test(*r->s4, 1, 9, *arr[1], 1, 2, 7);
	test(*r->s4, 1, 9, *arr[1], 1, 3, 6);
	test(*r->s4, 1, 9, *arr[1], 1, 4, 5);
	test(*r->s4, 1, 9, *arr[1], 1, 5, 5);
	test(*r->s4, 1, 9, *arr[1], 2, 0, 9);
	test(*r->s4, 1, 9, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 9, *arr[1], 2, 2, -1);
	test(*r->s4, 1, 9, *arr[1], 2, 3, -1);
	test(*r->s4, 1, 9, *arr[1], 2, 4, -1);
	test(*r->s4, 1, 9, *arr[1], 4, 0, 9);
	test(*r->s4, 1, 9, *arr[1], 4, 1, -3);
	test(*r->s4, 1, 9, *arr[1], 4, 2, -3);
	test(*r->s4, 1, 9, *arr[1], 5, 0, 9);
	test(*r->s4, 1, 9, *arr[1], 5, 1, 9);
	test(*r->s4, 1, 9, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 9, *arr[2], 0, 0, 9);
	test(*r->s4, 1, 9, *arr[2], 0, 1, 1);
	test(*r->s4, 1, 9, *arr[2], 0, 5, 1);
	test(*r->s4, 1, 9, *arr[2], 0, 9, 1);
	test(*r->s4, 1, 9, *arr[2], 0, 10, 1);
	test(*r->s4, 1, 9, *arr[2], 0, 11, 1);
	test(*r->s4, 1, 9, *arr[2], 1, 0, 9);
	test(*r->s4, 1, 9, *arr[2], 1, 1, 8);
	test(*r->s4, 1, 9, *arr[2], 1, 4, 5);
	test(*r->s4, 1, 9, *arr[2], 1, 8, 1);
	test(*r->s4, 1, 9, *arr[2], 1, 9, 0);
	test(*r->s4, 1, 9, *arr[2], 1, 10, 0);
	test(*r->s4, 1, 9, *arr[2], 5, 0, 9);
	test(*r->s4, 1, 9, *arr[2], 5, 1, -4);
	test(*r->s4, 1, 9, *arr[2], 5, 2, -4);
	test(*r->s4, 1, 9, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 9, *arr[2], 5, 5, -4);
	test(*r->s4, 1, 9, *arr[2], 5, 6, -4);
	test(*r->s4, 1, 9, *arr[2], 9, 0, 9);
	test(*r->s4, 1, 9, *arr[2], 9, 1, -8);
	test(*r->s4, 1, 9, *arr[2], 9, 2, -8);
	test(*r->s4, 1, 9, *arr[2], 10, 0, 9);
	test(*r->s4, 1, 9, *arr[2], 10, 1, 9);
	test(*r->s4, 1, 9, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 9, *arr[3], 0, 0, 9);
	test(*r->s4, 1, 9, *arr[3], 0, 1, 1);
	test(*r->s4, 1, 9, *arr[3], 0, 10, 1);
	test(*r->s4, 1, 9, *arr[3], 0, 19, 1);
	test(*r->s4, 1, 9, *arr[3], 0, 20, 1);
	test(*r->s4, 1, 9, *arr[3], 0, 21, 1);
	test(*r->s4, 1, 9, *arr[3], 1, 0, 9);
	test(*r->s4, 1, 9, *arr[3], 1, 1, 8);
	test(*r->s4, 1, 9, *arr[3], 1, 9, 0);
	test(*r->s4, 1, 9, *arr[3], 1, 18, -9);
	test(*r->s4, 1, 9, *arr[3], 1, 19, -10);
	test(*r->s4, 1, 9, *arr[3], 1, 20, -10);
	test(*r->s4, 1, 9, *arr[3], 10, 0, 9);
	test(*r->s4, 1, 9, *arr[3], 10, 1, -9);
	test(*r->s4, 1, 9, *arr[3], 10, 5, -9);
	test(*r->s4, 1, 9, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 9, *arr[3], 10, 10, -9);
	test(*r->s4, 1, 9, *arr[3], 10, 11, -9);
	test(*r->s4, 1, 9, *arr[3], 19, 0, 9);
	test(*r->s4, 1, 9, *arr[3], 19, 1, -18);
	test(*r->s4, 1, 9, *arr[3], 19, 2, -18);
	test(*r->s4, 1, 9, *arr[3], 20, 0, 9);
	test(*r->s4, 1, 9, *arr[3], 20, 1, 9);
	test(*r->s4, 1, 9, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 18, *arr[0], 0, 0, 18);
	test(*r->s4, 1, 18, *arr[0], 0, 1, 18);
	test(*r->s4, 1, 18, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 18, *arr[1], 0, 0, 18);
	test(*r->s4, 1, 18, *arr[1], 0, 1, 1);
	test(*r->s4, 1, 18, *arr[1], 0, 2, 1);
	test(*r->s4, 1, 18, *arr[1], 0, 4, 1);
	test(*r->s4, 1, 18, *arr[1], 0, 5, 1);
	test(*r->s4, 1, 18, *arr[1], 0, 6, 1);
	test(*r->s4, 1, 18, *arr[1], 1, 0, 18);
	test(*r->s4, 1, 18, *arr[1], 1, 1, 17);
	test(*r->s4, 1, 18, *arr[1], 1, 2, 16);
	test(*r->s4, 1, 18, *arr[1], 1, 3, 15);
	test(*r->s4, 1, 18, *arr[1], 1, 4, 14);
	test(*r->s4, 1, 18, *arr[1], 1, 5, 14);
	test(*r->s4, 1, 18, *arr[1], 2, 0, 18);
	test(*r->s4, 1, 18, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 18, *arr[1], 2, 2, -1);
	test(*r->s4, 1, 18, *arr[1], 2, 3, -1);
	test(*r->s4, 1, 18, *arr[1], 2, 4, -1);
	test(*r->s4, 1, 18, *arr[1], 4, 0, 18);
	test(*r->s4, 1, 18, *arr[1], 4, 1, -3);
	test(*r->s4, 1, 18, *arr[1], 4, 2, -3);
	test(*r->s4, 1, 18, *arr[1], 5, 0, 18);
	test(*r->s4, 1, 18, *arr[1], 5, 1, 18);
	test(*r->s4, 1, 18, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 18, *arr[2], 0, 0, 18);
	test(*r->s4, 1, 18, *arr[2], 0, 1, 1);
	test(*r->s4, 1, 18, *arr[2], 0, 5, 1);
	test(*r->s4, 1, 18, *arr[2], 0, 9, 1);
	test(*r->s4, 1, 18, *arr[2], 0, 10, 1);
	test(*r->s4, 1, 18, *arr[2], 0, 11, 1);
	test(*r->s4, 1, 18, *arr[2], 1, 0, 18);
	test(*r->s4, 1, 18, *arr[2], 1, 1, 17);
}

template <class S, class U>
void
test44(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 1, 18, *arr[2], 1, 4, 14);
	test(*r->s4, 1, 18, *arr[2], 1, 8, 10);
	test(*r->s4, 1, 18, *arr[2], 1, 9, 9);
	test(*r->s4, 1, 18, *arr[2], 1, 10, 9);
	test(*r->s4, 1, 18, *arr[2], 5, 0, 18);
	test(*r->s4, 1, 18, *arr[2], 5, 1, -4);
	test(*r->s4, 1, 18, *arr[2], 5, 2, -4);
	test(*r->s4, 1, 18, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 18, *arr[2], 5, 5, -4);
	test(*r->s4, 1, 18, *arr[2], 5, 6, -4);
	test(*r->s4, 1, 18, *arr[2], 9, 0, 18);
	test(*r->s4, 1, 18, *arr[2], 9, 1, -8);
	test(*r->s4, 1, 18, *arr[2], 9, 2, -8);
	test(*r->s4, 1, 18, *arr[2], 10, 0, 18);
	test(*r->s4, 1, 18, *arr[2], 10, 1, 18);
	test(*r->s4, 1, 18, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 18, *arr[3], 0, 0, 18);
	test(*r->s4, 1, 18, *arr[3], 0, 1, 1);
	test(*r->s4, 1, 18, *arr[3], 0, 10, 1);
	test(*r->s4, 1, 18, *arr[3], 0, 19, 1);
	test(*r->s4, 1, 18, *arr[3], 0, 20, 1);
	test(*r->s4, 1, 18, *arr[3], 0, 21, 1);
	test(*r->s4, 1, 18, *arr[3], 1, 0, 18);
	test(*r->s4, 1, 18, *arr[3], 1, 1, 17);
	test(*r->s4, 1, 18, *arr[3], 1, 9, 9);
	test(*r->s4, 1, 18, *arr[3], 1, 18, 0);
	test(*r->s4, 1, 18, *arr[3], 1, 19, -1);
	test(*r->s4, 1, 18, *arr[3], 1, 20, -1);
	test(*r->s4, 1, 18, *arr[3], 10, 0, 18);
	test(*r->s4, 1, 18, *arr[3], 10, 1, -9);
	test(*r->s4, 1, 18, *arr[3], 10, 5, -9);
	test(*r->s4, 1, 18, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 18, *arr[3], 10, 10, -9);
	test(*r->s4, 1, 18, *arr[3], 10, 11, -9);
	test(*r->s4, 1, 18, *arr[3], 19, 0, 18);
	test(*r->s4, 1, 18, *arr[3], 19, 1, -18);
	test(*r->s4, 1, 18, *arr[3], 19, 2, -18);
	test(*r->s4, 1, 18, *arr[3], 20, 0, 18);
	test(*r->s4, 1, 18, *arr[3], 20, 1, 18);
	test(*r->s4, 1, 18, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 19, *arr[0], 0, 0, 19);
	test(*r->s4, 1, 19, *arr[0], 0, 1, 19);
	test(*r->s4, 1, 19, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 19, *arr[1], 0, 0, 19);
	test(*r->s4, 1, 19, *arr[1], 0, 1, 1);
	test(*r->s4, 1, 19, *arr[1], 0, 2, 1);
	test(*r->s4, 1, 19, *arr[1], 0, 4, 1);
	test(*r->s4, 1, 19, *arr[1], 0, 5, 1);
	test(*r->s4, 1, 19, *arr[1], 0, 6, 1);
	test(*r->s4, 1, 19, *arr[1], 1, 0, 19);
	test(*r->s4, 1, 19, *arr[1], 1, 1, 18);
	test(*r->s4, 1, 19, *arr[1], 1, 2, 17);
	test(*r->s4, 1, 19, *arr[1], 1, 3, 16);
	test(*r->s4, 1, 19, *arr[1], 1, 4, 15);
	test(*r->s4, 1, 19, *arr[1], 1, 5, 15);
	test(*r->s4, 1, 19, *arr[1], 2, 0, 19);
	test(*r->s4, 1, 19, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 19, *arr[1], 2, 2, -1);
	test(*r->s4, 1, 19, *arr[1], 2, 3, -1);
	test(*r->s4, 1, 19, *arr[1], 2, 4, -1);
	test(*r->s4, 1, 19, *arr[1], 4, 0, 19);
	test(*r->s4, 1, 19, *arr[1], 4, 1, -3);
	test(*r->s4, 1, 19, *arr[1], 4, 2, -3);
	test(*r->s4, 1, 19, *arr[1], 5, 0, 19);
	test(*r->s4, 1, 19, *arr[1], 5, 1, 19);
	test(*r->s4, 1, 19, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 19, *arr[2], 0, 0, 19);
	test(*r->s4, 1, 19, *arr[2], 0, 1, 1);
	test(*r->s4, 1, 19, *arr[2], 0, 5, 1);
	test(*r->s4, 1, 19, *arr[2], 0, 9, 1);
	test(*r->s4, 1, 19, *arr[2], 0, 10, 1);
	test(*r->s4, 1, 19, *arr[2], 0, 11, 1);
	test(*r->s4, 1, 19, *arr[2], 1, 0, 19);
	test(*r->s4, 1, 19, *arr[2], 1, 1, 18);
	test(*r->s4, 1, 19, *arr[2], 1, 4, 15);
	test(*r->s4, 1, 19, *arr[2], 1, 8, 11);
	test(*r->s4, 1, 19, *arr[2], 1, 9, 10);
	test(*r->s4, 1, 19, *arr[2], 1, 10, 10);
	test(*r->s4, 1, 19, *arr[2], 5, 0, 19);
	test(*r->s4, 1, 19, *arr[2], 5, 1, -4);
	test(*r->s4, 1, 19, *arr[2], 5, 2, -4);
	test(*r->s4, 1, 19, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 19, *arr[2], 5, 5, -4);
	test(*r->s4, 1, 19, *arr[2], 5, 6, -4);
	test(*r->s4, 1, 19, *arr[2], 9, 0, 19);
	test(*r->s4, 1, 19, *arr[2], 9, 1, -8);
	test(*r->s4, 1, 19, *arr[2], 9, 2, -8);
	test(*r->s4, 1, 19, *arr[2], 10, 0, 19);
	test(*r->s4, 1, 19, *arr[2], 10, 1, 19);
	test(*r->s4, 1, 19, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 19, *arr[3], 0, 0, 19);
	test(*r->s4, 1, 19, *arr[3], 0, 1, 1);
	test(*r->s4, 1, 19, *arr[3], 0, 10, 1);
	test(*r->s4, 1, 19, *arr[3], 0, 19, 1);
	test(*r->s4, 1, 19, *arr[3], 0, 20, 1);
	test(*r->s4, 1, 19, *arr[3], 0, 21, 1);
	test(*r->s4, 1, 19, *arr[3], 1, 0, 19);
	test(*r->s4, 1, 19, *arr[3], 1, 1, 18);
	test(*r->s4, 1, 19, *arr[3], 1, 9, 10);
	test(*r->s4, 1, 19, *arr[3], 1, 18, 1);
}

template <class S, class U>
void
test45(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 1, 19, *arr[3], 1, 19, 0);
	test(*r->s4, 1, 19, *arr[3], 1, 20, 0);
	test(*r->s4, 1, 19, *arr[3], 10, 0, 19);
	test(*r->s4, 1, 19, *arr[3], 10, 1, -9);
	test(*r->s4, 1, 19, *arr[3], 10, 5, -9);
	test(*r->s4, 1, 19, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 19, *arr[3], 10, 10, -9);
	test(*r->s4, 1, 19, *arr[3], 10, 11, -9);
	test(*r->s4, 1, 19, *arr[3], 19, 0, 19);
	test(*r->s4, 1, 19, *arr[3], 19, 1, -18);
	test(*r->s4, 1, 19, *arr[3], 19, 2, -18);
	test(*r->s4, 1, 19, *arr[3], 20, 0, 19);
	test(*r->s4, 1, 19, *arr[3], 20, 1, 19);
	test(*r->s4, 1, 19, *arr[3], 21, 0, 0);
	test(*r->s4, 1, 20, *arr[0], 0, 0, 19);
	test(*r->s4, 1, 20, *arr[0], 0, 1, 19);
	test(*r->s4, 1, 20, *arr[0], 1, 0, 0);
	test(*r->s4, 1, 20, *arr[1], 0, 0, 19);
	test(*r->s4, 1, 20, *arr[1], 0, 1, 1);
	test(*r->s4, 1, 20, *arr[1], 0, 2, 1);
	test(*r->s4, 1, 20, *arr[1], 0, 4, 1);
	test(*r->s4, 1, 20, *arr[1], 0, 5, 1);
	test(*r->s4, 1, 20, *arr[1], 0, 6, 1);
	test(*r->s4, 1, 20, *arr[1], 1, 0, 19);
	test(*r->s4, 1, 20, *arr[1], 1, 1, 18);
	test(*r->s4, 1, 20, *arr[1], 1, 2, 17);
	test(*r->s4, 1, 20, *arr[1], 1, 3, 16);
	test(*r->s4, 1, 20, *arr[1], 1, 4, 15);
	test(*r->s4, 1, 20, *arr[1], 1, 5, 15);
	test(*r->s4, 1, 20, *arr[1], 2, 0, 19);
	test(*r->s4, 1, 20, *arr[1], 2, 1, -1);
	test(*r->s4, 1, 20, *arr[1], 2, 2, -1);
	test(*r->s4, 1, 20, *arr[1], 2, 3, -1);
	test(*r->s4, 1, 20, *arr[1], 2, 4, -1);
	test(*r->s4, 1, 20, *arr[1], 4, 0, 19);
	test(*r->s4, 1, 20, *arr[1], 4, 1, -3);
	test(*r->s4, 1, 20, *arr[1], 4, 2, -3);
	test(*r->s4, 1, 20, *arr[1], 5, 0, 19);
	test(*r->s4, 1, 20, *arr[1], 5, 1, 19);
	test(*r->s4, 1, 20, *arr[1], 6, 0, 0);
	test(*r->s4, 1, 20, *arr[2], 0, 0, 19);
	test(*r->s4, 1, 20, *arr[2], 0, 1, 1);
	test(*r->s4, 1, 20, *arr[2], 0, 5, 1);
	test(*r->s4, 1, 20, *arr[2], 0, 9, 1);
	test(*r->s4, 1, 20, *arr[2], 0, 10, 1);
	test(*r->s4, 1, 20, *arr[2], 0, 11, 1);
	test(*r->s4, 1, 20, *arr[2], 1, 0, 19);
	test(*r->s4, 1, 20, *arr[2], 1, 1, 18);
	test(*r->s4, 1, 20, *arr[2], 1, 4, 15);
	test(*r->s4, 1, 20, *arr[2], 1, 8, 11);
	test(*r->s4, 1, 20, *arr[2], 1, 9, 10);
	test(*r->s4, 1, 20, *arr[2], 1, 10, 10);
	test(*r->s4, 1, 20, *arr[2], 5, 0, 19);
	test(*r->s4, 1, 20, *arr[2], 5, 1, -4);
	test(*r->s4, 1, 20, *arr[2], 5, 2, -4);
	test(*r->s4, 1, 20, *arr[2], 5, 4, -4);
	test(*r->s4, 1, 20, *arr[2], 5, 5, -4);
	test(*r->s4, 1, 20, *arr[2], 5, 6, -4);
	test(*r->s4, 1, 20, *arr[2], 9, 0, 19);
	test(*r->s4, 1, 20, *arr[2], 9, 1, -8);
	test(*r->s4, 1, 20, *arr[2], 9, 2, -8);
	test(*r->s4, 1, 20, *arr[2], 10, 0, 19);
	test(*r->s4, 1, 20, *arr[2], 10, 1, 19);
	test(*r->s4, 1, 20, *arr[2], 11, 0, 0);
	test(*r->s4, 1, 20, *arr[3], 0, 0, 19);
	test(*r->s4, 1, 20, *arr[3], 0, 1, 1);
	test(*r->s4, 1, 20, *arr[3], 0, 10, 1);
	test(*r->s4, 1, 20, *arr[3], 0, 19, 1);
	test(*r->s4, 1, 20, *arr[3], 0, 20, 1);
	test(*r->s4, 1, 20, *arr[3], 0, 21, 1);
	test(*r->s4, 1, 20, *arr[3], 1, 0, 19);
	test(*r->s4, 1, 20, *arr[3], 1, 1, 18);
	test(*r->s4, 1, 20, *arr[3], 1, 9, 10);
	test(*r->s4, 1, 20, *arr[3], 1, 18, 1);
	test(*r->s4, 1, 20, *arr[3], 1, 19, 0);
	test(*r->s4, 1, 20, *arr[3], 1, 20, 0);
	test(*r->s4, 1, 20, *arr[3], 10, 0, 19);
	test(*r->s4, 1, 20, *arr[3], 10, 1, -9);
	test(*r->s4, 1, 20, *arr[3], 10, 5, -9);
	test(*r->s4, 1, 20, *arr[3], 10, 9, -9);
	test(*r->s4, 1, 20, *arr[3], 10, 10, -9);
	test(*r->s4, 1, 20, *arr[3], 10, 11, -9);
	test(*r->s4, 1, 20, *arr[3], 19, 0, 19);
	test(*r->s4, 1, 20, *arr[3], 19, 1, -18);
	test(*r->s4, 1, 20, *arr[3], 19, 2, -18);
	test(*r->s4, 1, 20, *arr[3], 20, 0, 19);
	test(*r->s4, 1, 20, *arr[3], 20, 1, 19);
	test(*r->s4, 1, 20, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 10, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 10, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 0, 1, -1);
	test(*r->s4, 10, 0, *arr[1], 0, 2, -2);
	test(*r->s4, 10, 0, *arr[1], 0, 4, -4);
	test(*r->s4, 10, 0, *arr[1], 0, 5, -5);
	test(*r->s4, 10, 0, *arr[1], 0, 6, -5);
	test(*r->s4, 10, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 1, 1, -1);
	test(*r->s4, 10, 0, *arr[1], 1, 2, -2);
}

template <class S, class U>
void
test46(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 10, 0, *arr[1], 1, 3, -3);
	test(*r->s4, 10, 0, *arr[1], 1, 4, -4);
	test(*r->s4, 10, 0, *arr[1], 1, 5, -4);
	test(*r->s4, 10, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 2, 1, -1);
	test(*r->s4, 10, 0, *arr[1], 2, 2, -2);
	test(*r->s4, 10, 0, *arr[1], 2, 3, -3);
	test(*r->s4, 10, 0, *arr[1], 2, 4, -3);
	test(*r->s4, 10, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 4, 1, -1);
	test(*r->s4, 10, 0, *arr[1], 4, 2, -1);
	test(*r->s4, 10, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 10, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 10, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 0, 1, -1);
	test(*r->s4, 10, 0, *arr[2], 0, 5, -5);
	test(*r->s4, 10, 0, *arr[2], 0, 9, -9);
	test(*r->s4, 10, 0, *arr[2], 0, 10, -10);
	test(*r->s4, 10, 0, *arr[2], 0, 11, -10);
	test(*r->s4, 10, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 1, 1, -1);
	test(*r->s4, 10, 0, *arr[2], 1, 4, -4);
	test(*r->s4, 10, 0, *arr[2], 1, 8, -8);
	test(*r->s4, 10, 0, *arr[2], 1, 9, -9);
	test(*r->s4, 10, 0, *arr[2], 1, 10, -9);
	test(*r->s4, 10, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 5, 1, -1);
	test(*r->s4, 10, 0, *arr[2], 5, 2, -2);
	test(*r->s4, 10, 0, *arr[2], 5, 4, -4);
	test(*r->s4, 10, 0, *arr[2], 5, 5, -5);
	test(*r->s4, 10, 0, *arr[2], 5, 6, -5);
	test(*r->s4, 10, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 9, 1, -1);
	test(*r->s4, 10, 0, *arr[2], 9, 2, -1);
	test(*r->s4, 10, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 10, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 10, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 0, 1, -1);
	test(*r->s4, 10, 0, *arr[3], 0, 10, -10);
	test(*r->s4, 10, 0, *arr[3], 0, 19, -19);
	test(*r->s4, 10, 0, *arr[3], 0, 20, -20);
	test(*r->s4, 10, 0, *arr[3], 0, 21, -20);
	test(*r->s4, 10, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 1, 1, -1);
	test(*r->s4, 10, 0, *arr[3], 1, 9, -9);
	test(*r->s4, 10, 0, *arr[3], 1, 18, -18);
	test(*r->s4, 10, 0, *arr[3], 1, 19, -19);
	test(*r->s4, 10, 0, *arr[3], 1, 20, -19);
	test(*r->s4, 10, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 10, 1, -1);
	test(*r->s4, 10, 0, *arr[3], 10, 5, -5);
	test(*r->s4, 10, 0, *arr[3], 10, 9, -9);
	test(*r->s4, 10, 0, *arr[3], 10, 10, -10);
	test(*r->s4, 10, 0, *arr[3], 10, 11, -10);
	test(*r->s4, 10, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 19, 1, -1);
	test(*r->s4, 10, 0, *arr[3], 19, 2, -1);
	test(*r->s4, 10, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 10, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 10, 0, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 1, *arr[0], 0, 0, 1);
	test(*r->s4, 10, 1, *arr[0], 0, 1, 1);
	test(*r->s4, 10, 1, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 1, *arr[1], 0, 0, 1);
	test(*r->s4, 10, 1, *arr[1], 0, 1, 10);
	test(*r->s4, 10, 1, *arr[1], 0, 2, 10);
	test(*r->s4, 10, 1, *arr[1], 0, 4, 10);
	test(*r->s4, 10, 1, *arr[1], 0, 5, 10);
	test(*r->s4, 10, 1, *arr[1], 0, 6, 10);
	test(*r->s4, 10, 1, *arr[1], 1, 0, 1);
	test(*r->s4, 10, 1, *arr[1], 1, 1, 9);
	test(*r->s4, 10, 1, *arr[1], 1, 2, 9);
	test(*r->s4, 10, 1, *arr[1], 1, 3, 9);
	test(*r->s4, 10, 1, *arr[1], 1, 4, 9);
	test(*r->s4, 10, 1, *arr[1], 1, 5, 9);
	test(*r->s4, 10, 1, *arr[1], 2, 0, 1);
	test(*r->s4, 10, 1, *arr[1], 2, 1, 8);
	test(*r->s4, 10, 1, *arr[1], 2, 2, 8);
	test(*r->s4, 10, 1, *arr[1], 2, 3, 8);
	test(*r->s4, 10, 1, *arr[1], 2, 4, 8);
	test(*r->s4, 10, 1, *arr[1], 4, 0, 1);
	test(*r->s4, 10, 1, *arr[1], 4, 1, 6);
	test(*r->s4, 10, 1, *arr[1], 4, 2, 6);
	test(*r->s4, 10, 1, *arr[1], 5, 0, 1);
	test(*r->s4, 10, 1, *arr[1], 5, 1, 1);
	test(*r->s4, 10, 1, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 1, *arr[2], 0, 0, 1);
	test(*r->s4, 10, 1, *arr[2], 0, 1, 10);
	test(*r->s4, 10, 1, *arr[2], 0, 5, 10);
	test(*r->s4, 10, 1, *arr[2], 0, 9, 10);
	test(*r->s4, 10, 1, *arr[2], 0, 10, 10);
	test(*r->s4, 10, 1, *arr[2], 0, 11, 10);
	test(*r->s4, 10, 1, *arr[2], 1, 0, 1);
	test(*r->s4, 10, 1, *arr[2], 1, 1, 9);
	test(*r->s4, 10, 1, *arr[2], 1, 4, 9);
	test(*r->s4, 10, 1, *arr[2], 1, 8, 9);
	test(*r->s4, 10, 1, *arr[2], 1, 9, 9);
	test(*r->s4, 10, 1, *arr[2], 1, 10, 9);
}

template <class S, class U>
void
test47(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 10, 1, *arr[2], 5, 0, 1);
	test(*r->s4, 10, 1, *arr[2], 5, 1, 5);
	test(*r->s4, 10, 1, *arr[2], 5, 2, 5);
	test(*r->s4, 10, 1, *arr[2], 5, 4, 5);
	test(*r->s4, 10, 1, *arr[2], 5, 5, 5);
	test(*r->s4, 10, 1, *arr[2], 5, 6, 5);
	test(*r->s4, 10, 1, *arr[2], 9, 0, 1);
	test(*r->s4, 10, 1, *arr[2], 9, 1, 1);
	test(*r->s4, 10, 1, *arr[2], 9, 2, 1);
	test(*r->s4, 10, 1, *arr[2], 10, 0, 1);
	test(*r->s4, 10, 1, *arr[2], 10, 1, 1);
	test(*r->s4, 10, 1, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 1, *arr[3], 0, 0, 1);
	test(*r->s4, 10, 1, *arr[3], 0, 1, 10);
	test(*r->s4, 10, 1, *arr[3], 0, 10, 10);
	test(*r->s4, 10, 1, *arr[3], 0, 19, 10);
	test(*r->s4, 10, 1, *arr[3], 0, 20, 10);
	test(*r->s4, 10, 1, *arr[3], 0, 21, 10);
	test(*r->s4, 10, 1, *arr[3], 1, 0, 1);
	test(*r->s4, 10, 1, *arr[3], 1, 1, 9);
	test(*r->s4, 10, 1, *arr[3], 1, 9, 9);
	test(*r->s4, 10, 1, *arr[3], 1, 18, 9);
	test(*r->s4, 10, 1, *arr[3], 1, 19, 9);
	test(*r->s4, 10, 1, *arr[3], 1, 20, 9);
	test(*r->s4, 10, 1, *arr[3], 10, 0, 1);
	test(*r->s4, 10, 1, *arr[3], 10, 1, 0);
	test(*r->s4, 10, 1, *arr[3], 10, 5, -4);
	test(*r->s4, 10, 1, *arr[3], 10, 9, -8);
	test(*r->s4, 10, 1, *arr[3], 10, 10, -9);
	test(*r->s4, 10, 1, *arr[3], 10, 11, -9);
	test(*r->s4, 10, 1, *arr[3], 19, 0, 1);
	test(*r->s4, 10, 1, *arr[3], 19, 1, -9);
	test(*r->s4, 10, 1, *arr[3], 19, 2, -9);
	test(*r->s4, 10, 1, *arr[3], 20, 0, 1);
	test(*r->s4, 10, 1, *arr[3], 20, 1, 1);
	test(*r->s4, 10, 1, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 5, *arr[0], 0, 0, 5);
	test(*r->s4, 10, 5, *arr[0], 0, 1, 5);
	test(*r->s4, 10, 5, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 5, *arr[1], 0, 0, 5);
	test(*r->s4, 10, 5, *arr[1], 0, 1, 10);
	test(*r->s4, 10, 5, *arr[1], 0, 2, 10);
	test(*r->s4, 10, 5, *arr[1], 0, 4, 10);
	test(*r->s4, 10, 5, *arr[1], 0, 5, 10);
	test(*r->s4, 10, 5, *arr[1], 0, 6, 10);
	test(*r->s4, 10, 5, *arr[1], 1, 0, 5);
	test(*r->s4, 10, 5, *arr[1], 1, 1, 9);
	test(*r->s4, 10, 5, *arr[1], 1, 2, 9);
	test(*r->s4, 10, 5, *arr[1], 1, 3, 9);
	test(*r->s4, 10, 5, *arr[1], 1, 4, 9);
	test(*r->s4, 10, 5, *arr[1], 1, 5, 9);
	test(*r->s4, 10, 5, *arr[1], 2, 0, 5);
	test(*r->s4, 10, 5, *arr[1], 2, 1, 8);
	test(*r->s4, 10, 5, *arr[1], 2, 2, 8);
	test(*r->s4, 10, 5, *arr[1], 2, 3, 8);
	test(*r->s4, 10, 5, *arr[1], 2, 4, 8);
	test(*r->s4, 10, 5, *arr[1], 4, 0, 5);
	test(*r->s4, 10, 5, *arr[1], 4, 1, 6);
	test(*r->s4, 10, 5, *arr[1], 4, 2, 6);
	test(*r->s4, 10, 5, *arr[1], 5, 0, 5);
	test(*r->s4, 10, 5, *arr[1], 5, 1, 5);
	test(*r->s4, 10, 5, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 5, *arr[2], 0, 0, 5);
	test(*r->s4, 10, 5, *arr[2], 0, 1, 10);
	test(*r->s4, 10, 5, *arr[2], 0, 5, 10);
	test(*r->s4, 10, 5, *arr[2], 0, 9, 10);
	test(*r->s4, 10, 5, *arr[2], 0, 10, 10);
	test(*r->s4, 10, 5, *arr[2], 0, 11, 10);
	test(*r->s4, 10, 5, *arr[2], 1, 0, 5);
	test(*r->s4, 10, 5, *arr[2], 1, 1, 9);
	test(*r->s4, 10, 5, *arr[2], 1, 4, 9);
	test(*r->s4, 10, 5, *arr[2], 1, 8, 9);
	test(*r->s4, 10, 5, *arr[2], 1, 9, 9);
	test(*r->s4, 10, 5, *arr[2], 1, 10, 9);
	test(*r->s4, 10, 5, *arr[2], 5, 0, 5);
	test(*r->s4, 10, 5, *arr[2], 5, 1, 5);
	test(*r->s4, 10, 5, *arr[2], 5, 2, 5);
	test(*r->s4, 10, 5, *arr[2], 5, 4, 5);
	test(*r->s4, 10, 5, *arr[2], 5, 5, 5);
	test(*r->s4, 10, 5, *arr[2], 5, 6, 5);
	test(*r->s4, 10, 5, *arr[2], 9, 0, 5);
	test(*r->s4, 10, 5, *arr[2], 9, 1, 1);
	test(*r->s4, 10, 5, *arr[2], 9, 2, 1);
	test(*r->s4, 10, 5, *arr[2], 10, 0, 5);
	test(*r->s4, 10, 5, *arr[2], 10, 1, 5);
	test(*r->s4, 10, 5, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 5, *arr[3], 0, 0, 5);
	test(*r->s4, 10, 5, *arr[3], 0, 1, 10);
	test(*r->s4, 10, 5, *arr[3], 0, 10, 10);
	test(*r->s4, 10, 5, *arr[3], 0, 19, 10);
	test(*r->s4, 10, 5, *arr[3], 0, 20, 10);
	test(*r->s4, 10, 5, *arr[3], 0, 21, 10);
	test(*r->s4, 10, 5, *arr[3], 1, 0, 5);
	test(*r->s4, 10, 5, *arr[3], 1, 1, 9);
	test(*r->s4, 10, 5, *arr[3], 1, 9, 9);
	test(*r->s4, 10, 5, *arr[3], 1, 18, 9);
	test(*r->s4, 10, 5, *arr[3], 1, 19, 9);
	test(*r->s4, 10, 5, *arr[3], 1, 20, 9);
	test(*r->s4, 10, 5, *arr[3], 10, 0, 5);
	test(*r->s4, 10, 5, *arr[3], 10, 1, 4);
}

template <class S, class U>
void
test48(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 10, 5, *arr[3], 10, 5, 0);
	test(*r->s4, 10, 5, *arr[3], 10, 9, -4);
	test(*r->s4, 10, 5, *arr[3], 10, 10, -5);
	test(*r->s4, 10, 5, *arr[3], 10, 11, -5);
	test(*r->s4, 10, 5, *arr[3], 19, 0, 5);
	test(*r->s4, 10, 5, *arr[3], 19, 1, -9);
	test(*r->s4, 10, 5, *arr[3], 19, 2, -9);
	test(*r->s4, 10, 5, *arr[3], 20, 0, 5);
	test(*r->s4, 10, 5, *arr[3], 20, 1, 5);
	test(*r->s4, 10, 5, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 9, *arr[0], 0, 0, 9);
	test(*r->s4, 10, 9, *arr[0], 0, 1, 9);
	test(*r->s4, 10, 9, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 9, *arr[1], 0, 0, 9);
	test(*r->s4, 10, 9, *arr[1], 0, 1, 10);
	test(*r->s4, 10, 9, *arr[1], 0, 2, 10);
	test(*r->s4, 10, 9, *arr[1], 0, 4, 10);
	test(*r->s4, 10, 9, *arr[1], 0, 5, 10);
	test(*r->s4, 10, 9, *arr[1], 0, 6, 10);
	test(*r->s4, 10, 9, *arr[1], 1, 0, 9);
	test(*r->s4, 10, 9, *arr[1], 1, 1, 9);
	test(*r->s4, 10, 9, *arr[1], 1, 2, 9);
	test(*r->s4, 10, 9, *arr[1], 1, 3, 9);
	test(*r->s4, 10, 9, *arr[1], 1, 4, 9);
	test(*r->s4, 10, 9, *arr[1], 1, 5, 9);
	test(*r->s4, 10, 9, *arr[1], 2, 0, 9);
	test(*r->s4, 10, 9, *arr[1], 2, 1, 8);
	test(*r->s4, 10, 9, *arr[1], 2, 2, 8);
	test(*r->s4, 10, 9, *arr[1], 2, 3, 8);
	test(*r->s4, 10, 9, *arr[1], 2, 4, 8);
	test(*r->s4, 10, 9, *arr[1], 4, 0, 9);
	test(*r->s4, 10, 9, *arr[1], 4, 1, 6);
	test(*r->s4, 10, 9, *arr[1], 4, 2, 6);
	test(*r->s4, 10, 9, *arr[1], 5, 0, 9);
	test(*r->s4, 10, 9, *arr[1], 5, 1, 9);
	test(*r->s4, 10, 9, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 9, *arr[2], 0, 0, 9);
	test(*r->s4, 10, 9, *arr[2], 0, 1, 10);
	test(*r->s4, 10, 9, *arr[2], 0, 5, 10);
	test(*r->s4, 10, 9, *arr[2], 0, 9, 10);
	test(*r->s4, 10, 9, *arr[2], 0, 10, 10);
	test(*r->s4, 10, 9, *arr[2], 0, 11, 10);
	test(*r->s4, 10, 9, *arr[2], 1, 0, 9);
	test(*r->s4, 10, 9, *arr[2], 1, 1, 9);
	test(*r->s4, 10, 9, *arr[2], 1, 4, 9);
	test(*r->s4, 10, 9, *arr[2], 1, 8, 9);
	test(*r->s4, 10, 9, *arr[2], 1, 9, 9);
	test(*r->s4, 10, 9, *arr[2], 1, 10, 9);
	test(*r->s4, 10, 9, *arr[2], 5, 0, 9);
	test(*r->s4, 10, 9, *arr[2], 5, 1, 5);
	test(*r->s4, 10, 9, *arr[2], 5, 2, 5);
	test(*r->s4, 10, 9, *arr[2], 5, 4, 5);
	test(*r->s4, 10, 9, *arr[2], 5, 5, 5);
	test(*r->s4, 10, 9, *arr[2], 5, 6, 5);
	test(*r->s4, 10, 9, *arr[2], 9, 0, 9);
	test(*r->s4, 10, 9, *arr[2], 9, 1, 1);
	test(*r->s4, 10, 9, *arr[2], 9, 2, 1);
	test(*r->s4, 10, 9, *arr[2], 10, 0, 9);
	test(*r->s4, 10, 9, *arr[2], 10, 1, 9);
	test(*r->s4, 10, 9, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 9, *arr[3], 0, 0, 9);
	test(*r->s4, 10, 9, *arr[3], 0, 1, 10);
	test(*r->s4, 10, 9, *arr[3], 0, 10, 10);
	test(*r->s4, 10, 9, *arr[3], 0, 19, 10);
	test(*r->s4, 10, 9, *arr[3], 0, 20, 10);
	test(*r->s4, 10, 9, *arr[3], 0, 21, 10);
	test(*r->s4, 10, 9, *arr[3], 1, 0, 9);
	test(*r->s4, 10, 9, *arr[3], 1, 1, 9);
	test(*r->s4, 10, 9, *arr[3], 1, 9, 9);
	test(*r->s4, 10, 9, *arr[3], 1, 18, 9);
	test(*r->s4, 10, 9, *arr[3], 1, 19, 9);
	test(*r->s4, 10, 9, *arr[3], 1, 20, 9);
	test(*r->s4, 10, 9, *arr[3], 10, 0, 9);
	test(*r->s4, 10, 9, *arr[3], 10, 1, 8);
	test(*r->s4, 10, 9, *arr[3], 10, 5, 4);
	test(*r->s4, 10, 9, *arr[3], 10, 9, 0);
	test(*r->s4, 10, 9, *arr[3], 10, 10, -1);
	test(*r->s4, 10, 9, *arr[3], 10, 11, -1);
	test(*r->s4, 10, 9, *arr[3], 19, 0, 9);
	test(*r->s4, 10, 9, *arr[3], 19, 1, -9);
	test(*r->s4, 10, 9, *arr[3], 19, 2, -9);
	test(*r->s4, 10, 9, *arr[3], 20, 0, 9);
	test(*r->s4, 10, 9, *arr[3], 20, 1, 9);
	test(*r->s4, 10, 9, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 10, *arr[0], 0, 0, 10);
	test(*r->s4, 10, 10, *arr[0], 0, 1, 10);
	test(*r->s4, 10, 10, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 10, *arr[1], 0, 0, 10);
	test(*r->s4, 10, 10, *arr[1], 0, 1, 10);
	test(*r->s4, 10, 10, *arr[1], 0, 2, 10);
	test(*r->s4, 10, 10, *arr[1], 0, 4, 10);
	test(*r->s4, 10, 10, *arr[1], 0, 5, 10);
	test(*r->s4, 10, 10, *arr[1], 0, 6, 10);
	test(*r->s4, 10, 10, *arr[1], 1, 0, 10);
	test(*r->s4, 10, 10, *arr[1], 1, 1, 9);
	test(*r->s4, 10, 10, *arr[1], 1, 2, 9);
	test(*r->s4, 10, 10, *arr[1], 1, 3, 9);
	test(*r->s4, 10, 10, *arr[1], 1, 4, 9);
	test(*r->s4, 10, 10, *arr[1], 1, 5, 9);
	test(*r->s4, 10, 10, *arr[1], 2, 0, 10);
}

template <class S, class U>
void
test49(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 10, 10, *arr[1], 2, 1, 8);
	test(*r->s4, 10, 10, *arr[1], 2, 2, 8);
	test(*r->s4, 10, 10, *arr[1], 2, 3, 8);
	test(*r->s4, 10, 10, *arr[1], 2, 4, 8);
	test(*r->s4, 10, 10, *arr[1], 4, 0, 10);
	test(*r->s4, 10, 10, *arr[1], 4, 1, 6);
	test(*r->s4, 10, 10, *arr[1], 4, 2, 6);
	test(*r->s4, 10, 10, *arr[1], 5, 0, 10);
	test(*r->s4, 10, 10, *arr[1], 5, 1, 10);
	test(*r->s4, 10, 10, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 10, *arr[2], 0, 0, 10);
	test(*r->s4, 10, 10, *arr[2], 0, 1, 10);
	test(*r->s4, 10, 10, *arr[2], 0, 5, 10);
	test(*r->s4, 10, 10, *arr[2], 0, 9, 10);
	test(*r->s4, 10, 10, *arr[2], 0, 10, 10);
	test(*r->s4, 10, 10, *arr[2], 0, 11, 10);
	test(*r->s4, 10, 10, *arr[2], 1, 0, 10);
	test(*r->s4, 10, 10, *arr[2], 1, 1, 9);
	test(*r->s4, 10, 10, *arr[2], 1, 4, 9);
	test(*r->s4, 10, 10, *arr[2], 1, 8, 9);
	test(*r->s4, 10, 10, *arr[2], 1, 9, 9);
	test(*r->s4, 10, 10, *arr[2], 1, 10, 9);
	test(*r->s4, 10, 10, *arr[2], 5, 0, 10);
	test(*r->s4, 10, 10, *arr[2], 5, 1, 5);
	test(*r->s4, 10, 10, *arr[2], 5, 2, 5);
	test(*r->s4, 10, 10, *arr[2], 5, 4, 5);
	test(*r->s4, 10, 10, *arr[2], 5, 5, 5);
	test(*r->s4, 10, 10, *arr[2], 5, 6, 5);
	test(*r->s4, 10, 10, *arr[2], 9, 0, 10);
	test(*r->s4, 10, 10, *arr[2], 9, 1, 1);
	test(*r->s4, 10, 10, *arr[2], 9, 2, 1);
	test(*r->s4, 10, 10, *arr[2], 10, 0, 10);
	test(*r->s4, 10, 10, *arr[2], 10, 1, 10);
	test(*r->s4, 10, 10, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 10, *arr[3], 0, 0, 10);
	test(*r->s4, 10, 10, *arr[3], 0, 1, 10);
	test(*r->s4, 10, 10, *arr[3], 0, 10, 10);
	test(*r->s4, 10, 10, *arr[3], 0, 19, 10);
	test(*r->s4, 10, 10, *arr[3], 0, 20, 10);
	test(*r->s4, 10, 10, *arr[3], 0, 21, 10);
	test(*r->s4, 10, 10, *arr[3], 1, 0, 10);
	test(*r->s4, 10, 10, *arr[3], 1, 1, 9);
	test(*r->s4, 10, 10, *arr[3], 1, 9, 9);
	test(*r->s4, 10, 10, *arr[3], 1, 18, 9);
	test(*r->s4, 10, 10, *arr[3], 1, 19, 9);
	test(*r->s4, 10, 10, *arr[3], 1, 20, 9);
	test(*r->s4, 10, 10, *arr[3], 10, 0, 10);
	test(*r->s4, 10, 10, *arr[3], 10, 1, 9);
	test(*r->s4, 10, 10, *arr[3], 10, 5, 5);
	test(*r->s4, 10, 10, *arr[3], 10, 9, 1);
	test(*r->s4, 10, 10, *arr[3], 10, 10, 0);
	test(*r->s4, 10, 10, *arr[3], 10, 11, 0);
	test(*r->s4, 10, 10, *arr[3], 19, 0, 10);
	test(*r->s4, 10, 10, *arr[3], 19, 1, -9);
	test(*r->s4, 10, 10, *arr[3], 19, 2, -9);
	test(*r->s4, 10, 10, *arr[3], 20, 0, 10);
	test(*r->s4, 10, 10, *arr[3], 20, 1, 10);
	test(*r->s4, 10, 10, *arr[3], 21, 0, 0);
	test(*r->s4, 10, 11, *arr[0], 0, 0, 10);
	test(*r->s4, 10, 11, *arr[0], 0, 1, 10);
	test(*r->s4, 10, 11, *arr[0], 1, 0, 0);
	test(*r->s4, 10, 11, *arr[1], 0, 0, 10);
	test(*r->s4, 10, 11, *arr[1], 0, 1, 10);
	test(*r->s4, 10, 11, *arr[1], 0, 2, 10);
	test(*r->s4, 10, 11, *arr[1], 0, 4, 10);
	test(*r->s4, 10, 11, *arr[1], 0, 5, 10);
	test(*r->s4, 10, 11, *arr[1], 0, 6, 10);
	test(*r->s4, 10, 11, *arr[1], 1, 0, 10);
	test(*r->s4, 10, 11, *arr[1], 1, 1, 9);
	test(*r->s4, 10, 11, *arr[1], 1, 2, 9);
	test(*r->s4, 10, 11, *arr[1], 1, 3, 9);
	test(*r->s4, 10, 11, *arr[1], 1, 4, 9);
	test(*r->s4, 10, 11, *arr[1], 1, 5, 9);
	test(*r->s4, 10, 11, *arr[1], 2, 0, 10);
	test(*r->s4, 10, 11, *arr[1], 2, 1, 8);
	test(*r->s4, 10, 11, *arr[1], 2, 2, 8);
	test(*r->s4, 10, 11, *arr[1], 2, 3, 8);
	test(*r->s4, 10, 11, *arr[1], 2, 4, 8);
	test(*r->s4, 10, 11, *arr[1], 4, 0, 10);
	test(*r->s4, 10, 11, *arr[1], 4, 1, 6);
	test(*r->s4, 10, 11, *arr[1], 4, 2, 6);
	test(*r->s4, 10, 11, *arr[1], 5, 0, 10);
	test(*r->s4, 10, 11, *arr[1], 5, 1, 10);
	test(*r->s4, 10, 11, *arr[1], 6, 0, 0);
	test(*r->s4, 10, 11, *arr[2], 0, 0, 10);
	test(*r->s4, 10, 11, *arr[2], 0, 1, 10);
	test(*r->s4, 10, 11, *arr[2], 0, 5, 10);
	test(*r->s4, 10, 11, *arr[2], 0, 9, 10);
	test(*r->s4, 10, 11, *arr[2], 0, 10, 10);
	test(*r->s4, 10, 11, *arr[2], 0, 11, 10);
	test(*r->s4, 10, 11, *arr[2], 1, 0, 10);
	test(*r->s4, 10, 11, *arr[2], 1, 1, 9);
	test(*r->s4, 10, 11, *arr[2], 1, 4, 9);
	test(*r->s4, 10, 11, *arr[2], 1, 8, 9);
	test(*r->s4, 10, 11, *arr[2], 1, 9, 9);
	test(*r->s4, 10, 11, *arr[2], 1, 10, 9);
	test(*r->s4, 10, 11, *arr[2], 5, 0, 10);
	test(*r->s4, 10, 11, *arr[2], 5, 1, 5);
	test(*r->s4, 10, 11, *arr[2], 5, 2, 5);
	test(*r->s4, 10, 11, *arr[2], 5, 4, 5);
}

template <class S, class U>
void
test50(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 10, 11, *arr[2], 5, 5, 5);
	test(*r->s4, 10, 11, *arr[2], 5, 6, 5);
	test(*r->s4, 10, 11, *arr[2], 9, 0, 10);
	test(*r->s4, 10, 11, *arr[2], 9, 1, 1);
	test(*r->s4, 10, 11, *arr[2], 9, 2, 1);
	test(*r->s4, 10, 11, *arr[2], 10, 0, 10);
	test(*r->s4, 10, 11, *arr[2], 10, 1, 10);
	test(*r->s4, 10, 11, *arr[2], 11, 0, 0);
	test(*r->s4, 10, 11, *arr[3], 0, 0, 10);
	test(*r->s4, 10, 11, *arr[3], 0, 1, 10);
	test(*r->s4, 10, 11, *arr[3], 0, 10, 10);
	test(*r->s4, 10, 11, *arr[3], 0, 19, 10);
	test(*r->s4, 10, 11, *arr[3], 0, 20, 10);
	test(*r->s4, 10, 11, *arr[3], 0, 21, 10);
	test(*r->s4, 10, 11, *arr[3], 1, 0, 10);
	test(*r->s4, 10, 11, *arr[3], 1, 1, 9);
	test(*r->s4, 10, 11, *arr[3], 1, 9, 9);
	test(*r->s4, 10, 11, *arr[3], 1, 18, 9);
	test(*r->s4, 10, 11, *arr[3], 1, 19, 9);
	test(*r->s4, 10, 11, *arr[3], 1, 20, 9);
	test(*r->s4, 10, 11, *arr[3], 10, 0, 10);
	test(*r->s4, 10, 11, *arr[3], 10, 1, 9);
	test(*r->s4, 10, 11, *arr[3], 10, 5, 5);
	test(*r->s4, 10, 11, *arr[3], 10, 9, 1);
	test(*r->s4, 10, 11, *arr[3], 10, 10, 0);
	test(*r->s4, 10, 11, *arr[3], 10, 11, 0);
	test(*r->s4, 10, 11, *arr[3], 19, 0, 10);
	test(*r->s4, 10, 11, *arr[3], 19, 1, -9);
	test(*r->s4, 10, 11, *arr[3], 19, 2, -9);
	test(*r->s4, 10, 11, *arr[3], 20, 0, 10);
	test(*r->s4, 10, 11, *arr[3], 20, 1, 10);
	test(*r->s4, 10, 11, *arr[3], 21, 0, 0);
	test(*r->s4, 19, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 19, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 19, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 0, 1, -1);
	test(*r->s4, 19, 0, *arr[1], 0, 2, -2);
	test(*r->s4, 19, 0, *arr[1], 0, 4, -4);
	test(*r->s4, 19, 0, *arr[1], 0, 5, -5);
	test(*r->s4, 19, 0, *arr[1], 0, 6, -5);
	test(*r->s4, 19, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 1, 1, -1);
	test(*r->s4, 19, 0, *arr[1], 1, 2, -2);
	test(*r->s4, 19, 0, *arr[1], 1, 3, -3);
	test(*r->s4, 19, 0, *arr[1], 1, 4, -4);
	test(*r->s4, 19, 0, *arr[1], 1, 5, -4);
	test(*r->s4, 19, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 2, 1, -1);
	test(*r->s4, 19, 0, *arr[1], 2, 2, -2);
	test(*r->s4, 19, 0, *arr[1], 2, 3, -3);
	test(*r->s4, 19, 0, *arr[1], 2, 4, -3);
	test(*r->s4, 19, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 4, 1, -1);
	test(*r->s4, 19, 0, *arr[1], 4, 2, -1);
	test(*r->s4, 19, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 19, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 19, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 0, 1, -1);
	test(*r->s4, 19, 0, *arr[2], 0, 5, -5);
	test(*r->s4, 19, 0, *arr[2], 0, 9, -9);
	test(*r->s4, 19, 0, *arr[2], 0, 10, -10);
	test(*r->s4, 19, 0, *arr[2], 0, 11, -10);
	test(*r->s4, 19, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 1, 1, -1);
	test(*r->s4, 19, 0, *arr[2], 1, 4, -4);
	test(*r->s4, 19, 0, *arr[2], 1, 8, -8);
	test(*r->s4, 19, 0, *arr[2], 1, 9, -9);
	test(*r->s4, 19, 0, *arr[2], 1, 10, -9);
	test(*r->s4, 19, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 5, 1, -1);
	test(*r->s4, 19, 0, *arr[2], 5, 2, -2);
	test(*r->s4, 19, 0, *arr[2], 5, 4, -4);
	test(*r->s4, 19, 0, *arr[2], 5, 5, -5);
	test(*r->s4, 19, 0, *arr[2], 5, 6, -5);
	test(*r->s4, 19, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 9, 1, -1);
	test(*r->s4, 19, 0, *arr[2], 9, 2, -1);
	test(*r->s4, 19, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 19, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 19, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 0, 1, -1);
	test(*r->s4, 19, 0, *arr[3], 0, 10, -10);
	test(*r->s4, 19, 0, *arr[3], 0, 19, -19);
	test(*r->s4, 19, 0, *arr[3], 0, 20, -20);
	test(*r->s4, 19, 0, *arr[3], 0, 21, -20);
	test(*r->s4, 19, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 1, 1, -1);
	test(*r->s4, 19, 0, *arr[3], 1, 9, -9);
	test(*r->s4, 19, 0, *arr[3], 1, 18, -18);
	test(*r->s4, 19, 0, *arr[3], 1, 19, -19);
	test(*r->s4, 19, 0, *arr[3], 1, 20, -19);
	test(*r->s4, 19, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 10, 1, -1);
	test(*r->s4, 19, 0, *arr[3], 10, 5, -5);
	test(*r->s4, 19, 0, *arr[3], 10, 9, -9);
	test(*r->s4, 19, 0, *arr[3], 10, 10, -10);
	test(*r->s4, 19, 0, *arr[3], 10, 11, -10);
}

template <class S, class U>
void
test51(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 19, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 19, 1, -1);
	test(*r->s4, 19, 0, *arr[3], 19, 2, -1);
	test(*r->s4, 19, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 19, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 19, 0, *arr[3], 21, 0, 0);
	test(*r->s4, 19, 1, *arr[0], 0, 0, 1);
	test(*r->s4, 19, 1, *arr[0], 0, 1, 1);
	test(*r->s4, 19, 1, *arr[0], 1, 0, 0);
	test(*r->s4, 19, 1, *arr[1], 0, 0, 1);
	test(*r->s4, 19, 1, *arr[1], 0, 1, 19);
	test(*r->s4, 19, 1, *arr[1], 0, 2, 19);
	test(*r->s4, 19, 1, *arr[1], 0, 4, 19);
	test(*r->s4, 19, 1, *arr[1], 0, 5, 19);
	test(*r->s4, 19, 1, *arr[1], 0, 6, 19);
	test(*r->s4, 19, 1, *arr[1], 1, 0, 1);
	test(*r->s4, 19, 1, *arr[1], 1, 1, 18);
	test(*r->s4, 19, 1, *arr[1], 1, 2, 18);
	test(*r->s4, 19, 1, *arr[1], 1, 3, 18);
	test(*r->s4, 19, 1, *arr[1], 1, 4, 18);
	test(*r->s4, 19, 1, *arr[1], 1, 5, 18);
	test(*r->s4, 19, 1, *arr[1], 2, 0, 1);
	test(*r->s4, 19, 1, *arr[1], 2, 1, 17);
	test(*r->s4, 19, 1, *arr[1], 2, 2, 17);
	test(*r->s4, 19, 1, *arr[1], 2, 3, 17);
	test(*r->s4, 19, 1, *arr[1], 2, 4, 17);
	test(*r->s4, 19, 1, *arr[1], 4, 0, 1);
	test(*r->s4, 19, 1, *arr[1], 4, 1, 15);
	test(*r->s4, 19, 1, *arr[1], 4, 2, 15);
	test(*r->s4, 19, 1, *arr[1], 5, 0, 1);
	test(*r->s4, 19, 1, *arr[1], 5, 1, 1);
	test(*r->s4, 19, 1, *arr[1], 6, 0, 0);
	test(*r->s4, 19, 1, *arr[2], 0, 0, 1);
	test(*r->s4, 19, 1, *arr[2], 0, 1, 19);
	test(*r->s4, 19, 1, *arr[2], 0, 5, 19);
	test(*r->s4, 19, 1, *arr[2], 0, 9, 19);
	test(*r->s4, 19, 1, *arr[2], 0, 10, 19);
	test(*r->s4, 19, 1, *arr[2], 0, 11, 19);
	test(*r->s4, 19, 1, *arr[2], 1, 0, 1);
	test(*r->s4, 19, 1, *arr[2], 1, 1, 18);
	test(*r->s4, 19, 1, *arr[2], 1, 4, 18);
	test(*r->s4, 19, 1, *arr[2], 1, 8, 18);
	test(*r->s4, 19, 1, *arr[2], 1, 9, 18);
	test(*r->s4, 19, 1, *arr[2], 1, 10, 18);
	test(*r->s4, 19, 1, *arr[2], 5, 0, 1);
	test(*r->s4, 19, 1, *arr[2], 5, 1, 14);
	test(*r->s4, 19, 1, *arr[2], 5, 2, 14);
	test(*r->s4, 19, 1, *arr[2], 5, 4, 14);
	test(*r->s4, 19, 1, *arr[2], 5, 5, 14);
	test(*r->s4, 19, 1, *arr[2], 5, 6, 14);
	test(*r->s4, 19, 1, *arr[2], 9, 0, 1);
	test(*r->s4, 19, 1, *arr[2], 9, 1, 10);
	test(*r->s4, 19, 1, *arr[2], 9, 2, 10);
	test(*r->s4, 19, 1, *arr[2], 10, 0, 1);
	test(*r->s4, 19, 1, *arr[2], 10, 1, 1);
	test(*r->s4, 19, 1, *arr[2], 11, 0, 0);
	test(*r->s4, 19, 1, *arr[3], 0, 0, 1);
	test(*r->s4, 19, 1, *arr[3], 0, 1, 19);
	test(*r->s4, 19, 1, *arr[3], 0, 10, 19);
	test(*r->s4, 19, 1, *arr[3], 0, 19, 19);
	test(*r->s4, 19, 1, *arr[3], 0, 20, 19);
	test(*r->s4, 19, 1, *arr[3], 0, 21, 19);
	test(*r->s4, 19, 1, *arr[3], 1, 0, 1);
	test(*r->s4, 19, 1, *arr[3], 1, 1, 18);
	test(*r->s4, 19, 1, *arr[3], 1, 9, 18);
	test(*r->s4, 19, 1, *arr[3], 1, 18, 18);
	test(*r->s4, 19, 1, *arr[3], 1, 19, 18);
	test(*r->s4, 19, 1, *arr[3], 1, 20, 18);
	test(*r->s4, 19, 1, *arr[3], 10, 0, 1);
	test(*r->s4, 19, 1, *arr[3], 10, 1, 9);
	test(*r->s4, 19, 1, *arr[3], 10, 5, 9);
	test(*r->s4, 19, 1, *arr[3], 10, 9, 9);
	test(*r->s4, 19, 1, *arr[3], 10, 10, 9);
	test(*r->s4, 19, 1, *arr[3], 10, 11, 9);
	test(*r->s4, 19, 1, *arr[3], 19, 0, 1);
	test(*r->s4, 19, 1, *arr[3], 19, 1, 0);
	test(*r->s4, 19, 1, *arr[3], 19, 2, 0);
	test(*r->s4, 19, 1, *arr[3], 20, 0, 1);
	test(*r->s4, 19, 1, *arr[3], 20, 1, 1);
	test(*r->s4, 19, 1, *arr[3], 21, 0, 0);
	test(*r->s4, 19, 2, *arr[0], 0, 0, 1);
	test(*r->s4, 19, 2, *arr[0], 0, 1, 1);
	test(*r->s4, 19, 2, *arr[0], 1, 0, 0);
	test(*r->s4, 19, 2, *arr[1], 0, 0, 1);
	test(*r->s4, 19, 2, *arr[1], 0, 1, 19);
	test(*r->s4, 19, 2, *arr[1], 0, 2, 19);
	test(*r->s4, 19, 2, *arr[1], 0, 4, 19);
	test(*r->s4, 19, 2, *arr[1], 0, 5, 19);
	test(*r->s4, 19, 2, *arr[1], 0, 6, 19);
	test(*r->s4, 19, 2, *arr[1], 1, 0, 1);
	test(*r->s4, 19, 2, *arr[1], 1, 1, 18);
	test(*r->s4, 19, 2, *arr[1], 1, 2, 18);
	test(*r->s4, 19, 2, *arr[1], 1, 3, 18);
	test(*r->s4, 19, 2, *arr[1], 1, 4, 18);
	test(*r->s4, 19, 2, *arr[1], 1, 5, 18);
	test(*r->s4, 19, 2, *arr[1], 2, 0, 1);
	test(*r->s4, 19, 2, *arr[1], 2, 1, 17);
	test(*r->s4, 19, 2, *arr[1], 2, 2, 17);
	test(*r->s4, 19, 2, *arr[1], 2, 3, 17);
	test(*r->s4, 19, 2, *arr[1], 2, 4, 17);
}

template <class S, class U>
void
test52(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 19, 2, *arr[1], 4, 0, 1);
	test(*r->s4, 19, 2, *arr[1], 4, 1, 15);
	test(*r->s4, 19, 2, *arr[1], 4, 2, 15);
	test(*r->s4, 19, 2, *arr[1], 5, 0, 1);
	test(*r->s4, 19, 2, *arr[1], 5, 1, 1);
	test(*r->s4, 19, 2, *arr[1], 6, 0, 0);
	test(*r->s4, 19, 2, *arr[2], 0, 0, 1);
	test(*r->s4, 19, 2, *arr[2], 0, 1, 19);
	test(*r->s4, 19, 2, *arr[2], 0, 5, 19);
	test(*r->s4, 19, 2, *arr[2], 0, 9, 19);
	test(*r->s4, 19, 2, *arr[2], 0, 10, 19);
	test(*r->s4, 19, 2, *arr[2], 0, 11, 19);
	test(*r->s4, 19, 2, *arr[2], 1, 0, 1);
	test(*r->s4, 19, 2, *arr[2], 1, 1, 18);
	test(*r->s4, 19, 2, *arr[2], 1, 4, 18);
	test(*r->s4, 19, 2, *arr[2], 1, 8, 18);
	test(*r->s4, 19, 2, *arr[2], 1, 9, 18);
	test(*r->s4, 19, 2, *arr[2], 1, 10, 18);
	test(*r->s4, 19, 2, *arr[2], 5, 0, 1);
	test(*r->s4, 19, 2, *arr[2], 5, 1, 14);
	test(*r->s4, 19, 2, *arr[2], 5, 2, 14);
	test(*r->s4, 19, 2, *arr[2], 5, 4, 14);
	test(*r->s4, 19, 2, *arr[2], 5, 5, 14);
	test(*r->s4, 19, 2, *arr[2], 5, 6, 14);
	test(*r->s4, 19, 2, *arr[2], 9, 0, 1);
	test(*r->s4, 19, 2, *arr[2], 9, 1, 10);
	test(*r->s4, 19, 2, *arr[2], 9, 2, 10);
	test(*r->s4, 19, 2, *arr[2], 10, 0, 1);
	test(*r->s4, 19, 2, *arr[2], 10, 1, 1);
	test(*r->s4, 19, 2, *arr[2], 11, 0, 0);
	test(*r->s4, 19, 2, *arr[3], 0, 0, 1);
	test(*r->s4, 19, 2, *arr[3], 0, 1, 19);
	test(*r->s4, 19, 2, *arr[3], 0, 10, 19);
	test(*r->s4, 19, 2, *arr[3], 0, 19, 19);
	test(*r->s4, 19, 2, *arr[3], 0, 20, 19);
	test(*r->s4, 19, 2, *arr[3], 0, 21, 19);
	test(*r->s4, 19, 2, *arr[3], 1, 0, 1);
	test(*r->s4, 19, 2, *arr[3], 1, 1, 18);
	test(*r->s4, 19, 2, *arr[3], 1, 9, 18);
	test(*r->s4, 19, 2, *arr[3], 1, 18, 18);
	test(*r->s4, 19, 2, *arr[3], 1, 19, 18);
	test(*r->s4, 19, 2, *arr[3], 1, 20, 18);
	test(*r->s4, 19, 2, *arr[3], 10, 0, 1);
	test(*r->s4, 19, 2, *arr[3], 10, 1, 9);
	test(*r->s4, 19, 2, *arr[3], 10, 5, 9);
	test(*r->s4, 19, 2, *arr[3], 10, 9, 9);
	test(*r->s4, 19, 2, *arr[3], 10, 10, 9);
	test(*r->s4, 19, 2, *arr[3], 10, 11, 9);
	test(*r->s4, 19, 2, *arr[3], 19, 0, 1);
	test(*r->s4, 19, 2, *arr[3], 19, 1, 0);
	test(*r->s4, 19, 2, *arr[3], 19, 2, 0);
	test(*r->s4, 19, 2, *arr[3], 20, 0, 1);
	test(*r->s4, 19, 2, *arr[3], 20, 1, 1);
	test(*r->s4, 19, 2, *arr[3], 21, 0, 0);
	test(*r->s4, 20, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 20, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 20, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 0, 1, -1);
	test(*r->s4, 20, 0, *arr[1], 0, 2, -2);
	test(*r->s4, 20, 0, *arr[1], 0, 4, -4);
	test(*r->s4, 20, 0, *arr[1], 0, 5, -5);
	test(*r->s4, 20, 0, *arr[1], 0, 6, -5);
	test(*r->s4, 20, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 1, 1, -1);
	test(*r->s4, 20, 0, *arr[1], 1, 2, -2);
	test(*r->s4, 20, 0, *arr[1], 1, 3, -3);
	test(*r->s4, 20, 0, *arr[1], 1, 4, -4);
	test(*r->s4, 20, 0, *arr[1], 1, 5, -4);
	test(*r->s4, 20, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 2, 1, -1);
	test(*r->s4, 20, 0, *arr[1], 2, 2, -2);
	test(*r->s4, 20, 0, *arr[1], 2, 3, -3);
	test(*r->s4, 20, 0, *arr[1], 2, 4, -3);
	test(*r->s4, 20, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 4, 1, -1);
	test(*r->s4, 20, 0, *arr[1], 4, 2, -1);
	test(*r->s4, 20, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 20, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 20, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 0, 1, -1);
	test(*r->s4, 20, 0, *arr[2], 0, 5, -5);
	test(*r->s4, 20, 0, *arr[2], 0, 9, -9);
	test(*r->s4, 20, 0, *arr[2], 0, 10, -10);
	test(*r->s4, 20, 0, *arr[2], 0, 11, -10);
	test(*r->s4, 20, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 1, 1, -1);
	test(*r->s4, 20, 0, *arr[2], 1, 4, -4);
	test(*r->s4, 20, 0, *arr[2], 1, 8, -8);
	test(*r->s4, 20, 0, *arr[2], 1, 9, -9);
	test(*r->s4, 20, 0, *arr[2], 1, 10, -9);
	test(*r->s4, 20, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 5, 1, -1);
	test(*r->s4, 20, 0, *arr[2], 5, 2, -2);
	test(*r->s4, 20, 0, *arr[2], 5, 4, -4);
	test(*r->s4, 20, 0, *arr[2], 5, 5, -5);
	test(*r->s4, 20, 0, *arr[2], 5, 6, -5);
	test(*r->s4, 20, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 9, 1, -1);
}

template <class S, class U>
void
test53(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 20, 0, *arr[2], 9, 2, -1);
	test(*r->s4, 20, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 20, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 20, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 0, 1, -1);
	test(*r->s4, 20, 0, *arr[3], 0, 10, -10);
	test(*r->s4, 20, 0, *arr[3], 0, 19, -19);
	test(*r->s4, 20, 0, *arr[3], 0, 20, -20);
	test(*r->s4, 20, 0, *arr[3], 0, 21, -20);
	test(*r->s4, 20, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 1, 1, -1);
	test(*r->s4, 20, 0, *arr[3], 1, 9, -9);
	test(*r->s4, 20, 0, *arr[3], 1, 18, -18);
	test(*r->s4, 20, 0, *arr[3], 1, 19, -19);
	test(*r->s4, 20, 0, *arr[3], 1, 20, -19);
	test(*r->s4, 20, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 10, 1, -1);
	test(*r->s4, 20, 0, *arr[3], 10, 5, -5);
	test(*r->s4, 20, 0, *arr[3], 10, 9, -9);
	test(*r->s4, 20, 0, *arr[3], 10, 10, -10);
	test(*r->s4, 20, 0, *arr[3], 10, 11, -10);
	test(*r->s4, 20, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 19, 1, -1);
	test(*r->s4, 20, 0, *arr[3], 19, 2, -1);
	test(*r->s4, 20, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 20, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 20, 0, *arr[3], 21, 0, 0);
	test(*r->s4, 20, 1, *arr[0], 0, 0, 0);
	test(*r->s4, 20, 1, *arr[0], 0, 1, 0);
	test(*r->s4, 20, 1, *arr[0], 1, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 0, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 0, 1, -1);
	test(*r->s4, 20, 1, *arr[1], 0, 2, -2);
	test(*r->s4, 20, 1, *arr[1], 0, 4, -4);
	test(*r->s4, 20, 1, *arr[1], 0, 5, -5);
	test(*r->s4, 20, 1, *arr[1], 0, 6, -5);
	test(*r->s4, 20, 1, *arr[1], 1, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 1, 1, -1);
	test(*r->s4, 20, 1, *arr[1], 1, 2, -2);
	test(*r->s4, 20, 1, *arr[1], 1, 3, -3);
	test(*r->s4, 20, 1, *arr[1], 1, 4, -4);
	test(*r->s4, 20, 1, *arr[1], 1, 5, -4);
	test(*r->s4, 20, 1, *arr[1], 2, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 2, 1, -1);
	test(*r->s4, 20, 1, *arr[1], 2, 2, -2);
	test(*r->s4, 20, 1, *arr[1], 2, 3, -3);
	test(*r->s4, 20, 1, *arr[1], 2, 4, -3);
	test(*r->s4, 20, 1, *arr[1], 4, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 4, 1, -1);
	test(*r->s4, 20, 1, *arr[1], 4, 2, -1);
	test(*r->s4, 20, 1, *arr[1], 5, 0, 0);
	test(*r->s4, 20, 1, *arr[1], 5, 1, 0);
	test(*r->s4, 20, 1, *arr[1], 6, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 0, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 0, 1, -1);
	test(*r->s4, 20, 1, *arr[2], 0, 5, -5);
	test(*r->s4, 20, 1, *arr[2], 0, 9, -9);
	test(*r->s4, 20, 1, *arr[2], 0, 10, -10);
	test(*r->s4, 20, 1, *arr[2], 0, 11, -10);
	test(*r->s4, 20, 1, *arr[2], 1, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 1, 1, -1);
	test(*r->s4, 20, 1, *arr[2], 1, 4, -4);
	test(*r->s4, 20, 1, *arr[2], 1, 8, -8);
	test(*r->s4, 20, 1, *arr[2], 1, 9, -9);
	test(*r->s4, 20, 1, *arr[2], 1, 10, -9);
	test(*r->s4, 20, 1, *arr[2], 5, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 5, 1, -1);
	test(*r->s4, 20, 1, *arr[2], 5, 2, -2);
	test(*r->s4, 20, 1, *arr[2], 5, 4, -4);
	test(*r->s4, 20, 1, *arr[2], 5, 5, -5);
	test(*r->s4, 20, 1, *arr[2], 5, 6, -5);
	test(*r->s4, 20, 1, *arr[2], 9, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 9, 1, -1);
	test(*r->s4, 20, 1, *arr[2], 9, 2, -1);
	test(*r->s4, 20, 1, *arr[2], 10, 0, 0);
	test(*r->s4, 20, 1, *arr[2], 10, 1, 0);
	test(*r->s4, 20, 1, *arr[2], 11, 0, 0);
	test(*r->s4, 20, 1, *arr[3], 0, 0, 0);
	test(*r->s4, 20, 1, *arr[3], 0, 1, -1);
	test(*r->s4, 20, 1, *arr[3], 0, 10, -10);
	test(*r->s4, 20, 1, *arr[3], 0, 19, -19);
	test(*r->s4, 20, 1, *arr[3], 0, 20, -20);
	test(*r->s4, 20, 1, *arr[3], 0, 21, -20);
	test(*r->s4, 20, 1, *arr[3], 1, 0, 0);
	test(*r->s4, 20, 1, *arr[3], 1, 1, -1);
	test(*r->s4, 20, 1, *arr[3], 1, 9, -9);
	test(*r->s4, 20, 1, *arr[3], 1, 18, -18);
	test(*r->s4, 20, 1, *arr[3], 1, 19, -19);
	test(*r->s4, 20, 1, *arr[3], 1, 20, -19);
	test(*r->s4, 20, 1, *arr[3], 10, 0, 0);
	test(*r->s4, 20, 1, *arr[3], 10, 1, -1);
	test(*r->s4, 20, 1, *arr[3], 10, 5, -5);
	test(*r->s4, 20, 1, *arr[3], 10, 9, -9);
	test(*r->s4, 20, 1, *arr[3], 10, 10, -10);
	test(*r->s4, 20, 1, *arr[3], 10, 11, -10);
	test(*r->s4, 20, 1, *arr[3], 19, 0, 0);
	test(*r->s4, 20, 1, *arr[3], 19, 1, -1);
	test(*r->s4, 20, 1, *arr[3], 19, 2, -1);
	test(*r->s4, 20, 1, *arr[3], 20, 0, 0);
}

template <class S, class U>
void
test54(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test(*r->s4, 20, 1, *arr[3], 20, 1, 0);
	test(*r->s4, 20, 1, *arr[3], 21, 0, 0);
	test(*r->s4, 21, 0, *arr[0], 0, 0, 0);
	test(*r->s4, 21, 0, *arr[0], 0, 1, 0);
	test(*r->s4, 21, 0, *arr[0], 1, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 1, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 2, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 4, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 5, 0);
	test(*r->s4, 21, 0, *arr[1], 0, 6, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 1, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 2, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 3, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 4, 0);
	test(*r->s4, 21, 0, *arr[1], 1, 5, 0);
	test(*r->s4, 21, 0, *arr[1], 2, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 2, 1, 0);
	test(*r->s4, 21, 0, *arr[1], 2, 2, 0);
	test(*r->s4, 21, 0, *arr[1], 2, 3, 0);
	test(*r->s4, 21, 0, *arr[1], 2, 4, 0);
	test(*r->s4, 21, 0, *arr[1], 4, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 4, 1, 0);
	test(*r->s4, 21, 0, *arr[1], 4, 2, 0);
	test(*r->s4, 21, 0, *arr[1], 5, 0, 0);
	test(*r->s4, 21, 0, *arr[1], 5, 1, 0);
	test(*r->s4, 21, 0, *arr[1], 6, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 1, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 5, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 9, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 10, 0);
	test(*r->s4, 21, 0, *arr[2], 0, 11, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 1, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 4, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 8, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 9, 0);
	test(*r->s4, 21, 0, *arr[2], 1, 10, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 1, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 2, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 4, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 5, 0);
	test(*r->s4, 21, 0, *arr[2], 5, 6, 0);
	test(*r->s4, 21, 0, *arr[2], 9, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 9, 1, 0);
	test(*r->s4, 21, 0, *arr[2], 9, 2, 0);
	test(*r->s4, 21, 0, *arr[2], 10, 0, 0);
	test(*r->s4, 21, 0, *arr[2], 10, 1, 0);
	test(*r->s4, 21, 0, *arr[2], 11, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 1, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 10, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 19, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 20, 0);
	test(*r->s4, 21, 0, *arr[3], 0, 21, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 1, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 9, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 18, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 19, 0);
	test(*r->s4, 21, 0, *arr[3], 1, 20, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 1, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 5, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 9, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 10, 0);
	test(*r->s4, 21, 0, *arr[3], 10, 11, 0);
	test(*r->s4, 21, 0, *arr[3], 19, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 19, 1, 0);
	test(*r->s4, 21, 0, *arr[3], 19, 2, 0);
	test(*r->s4, 21, 0, *arr[3], 20, 0, 0);
	test(*r->s4, 21, 0, *arr[3], 20, 1, 0);
	test(*r->s4, 21, 0, *arr[3], 21, 0, 0);
}

template <class S, class U>
void
test55(pmem::obj::persistent_ptr<root> &r, const std::array<U, 4> &arr)
{
	test_npos(*r->s1, 0, 0, *arr[0], 0, 0);
	test_npos(*r->s1, 0, 0, *arr[1], 0, -5);
	test_npos(*r->s2, 0, 0, *arr[2], 0, -10);
	test_npos(*r->s2, 0, 0, *arr[2], 1, -9);
	test_npos(*r->s2, 0, 0, *arr[2], 5, -5);
}

void
run(pmem::obj::pool<root> &pop)
{
	auto r = pop.root();

	try {
		nvobj::transaction::run(pop, [&] {
			r->s1 = nvobj::make_persistent<S>("");
			r->s2 = nvobj::make_persistent<S>("abcde");
			r->s3 = nvobj::make_persistent<S>("abcdefghij");
			r->s4 = nvobj::make_persistent<S>(
				"abcdefghijklmnopqrst");
		});

		std::array<S *, 4> arr{&*r->s1, &*r->s2, &*r->s3, &*r->s4};

		std::string s1(""), s2("abcde"), s3("abcdefghij"),
			s4("abcdefghijklmnopqrst");
		std::array<std::string *, 4> arr_std_str = {&s1, &s2, &s3, &s4};

		/* test pmem::string with pmem::string comparison */
		test0<S>(r, arr);
		test1<S>(r, arr);
		test2<S>(r, arr);
		test3<S>(r, arr);
		test4<S>(r, arr);
		test5<S>(r, arr);
		test6<S>(r, arr);
		test7<S>(r, arr);
		test8<S>(r, arr);
		test9<S>(r, arr);
		test10<S>(r, arr);
		test11<S>(r, arr);
		test12<S>(r, arr);
		test13<S>(r, arr);
		test14<S>(r, arr);
		test15<S>(r, arr);
		test16<S>(r, arr);
		test17<S>(r, arr);
		test18<S>(r, arr);
		test19<S>(r, arr);
		test20<S>(r, arr);
		test21<S>(r, arr);
		test22<S>(r, arr);
		test23<S>(r, arr);
		test24<S>(r, arr);
		test25<S>(r, arr);
		test26<S>(r, arr);
		test27<S>(r, arr);
		test28<S>(r, arr);
		test29<S>(r, arr);
		test30<S>(r, arr);
		test31<S>(r, arr);
		test32<S>(r, arr);
		test33<S>(r, arr);
		test34<S>(r, arr);
		test35<S>(r, arr);
		test36<S>(r, arr);
		test37<S>(r, arr);
		test38<S>(r, arr);
		test39<S>(r, arr);
		test40<S>(r, arr);
		test41<S>(r, arr);
		test42<S>(r, arr);
		test43<S>(r, arr);
		test44<S>(r, arr);
		test45<S>(r, arr);
		test46<S>(r, arr);
		test47<S>(r, arr);
		test48<S>(r, arr);
		test49<S>(r, arr);
		test50<S>(r, arr);
		test51<S>(r, arr);
		test52<S>(r, arr);
		test53<S>(r, arr);
		test54<S>(r, arr);
		test55<S>(r, arr);

		/* test pmem::string with std::string comparison */
		test0<S>(r, arr_std_str);
		test1<S>(r, arr_std_str);
		test2<S>(r, arr_std_str);
		test3<S>(r, arr_std_str);
		test4<S>(r, arr_std_str);
		test5<S>(r, arr_std_str);
		test6<S>(r, arr_std_str);
		test7<S>(r, arr_std_str);
		test8<S>(r, arr_std_str);
		test9<S>(r, arr_std_str);
		test10<S>(r, arr_std_str);
		test11<S>(r, arr_std_str);
		test12<S>(r, arr_std_str);
		test13<S>(r, arr_std_str);
		test14<S>(r, arr_std_str);
		test15<S>(r, arr_std_str);
		test16<S>(r, arr_std_str);
		test17<S>(r, arr_std_str);
		test18<S>(r, arr_std_str);
		test19<S>(r, arr_std_str);
		test20<S>(r, arr_std_str);
		test21<S>(r, arr_std_str);
		test22<S>(r, arr_std_str);
		test23<S>(r, arr_std_str);
		test24<S>(r, arr_std_str);
		test25<S>(r, arr_std_str);
		test26<S>(r, arr_std_str);
		test27<S>(r, arr_std_str);
		test28<S>(r, arr_std_str);
		test29<S>(r, arr_std_str);
		test30<S>(r, arr_std_str);
		test31<S>(r, arr_std_str);
		test32<S>(r, arr_std_str);
		test33<S>(r, arr_std_str);
		test34<S>(r, arr_std_str);
		test35<S>(r, arr_std_str);
		test36<S>(r, arr_std_str);
		test37<S>(r, arr_std_str);
		test38<S>(r, arr_std_str);
		test39<S>(r, arr_std_str);
		test40<S>(r, arr_std_str);
		test41<S>(r, arr_std_str);
		test42<S>(r, arr_std_str);
		test43<S>(r, arr_std_str);
		test44<S>(r, arr_std_str);
		test45<S>(r, arr_std_str);
		test46<S>(r, arr_std_str);
		test47<S>(r, arr_std_str);
		test48<S>(r, arr_std_str);
		test49<S>(r, arr_std_str);
		test50<S>(r, arr_std_str);
		test51<S>(r, arr_std_str);
		test52<S>(r, arr_std_str);
		test53<S>(r, arr_std_str);
		test54<S>(r, arr_std_str);
		test55<S>(r, arr_std_str);

		nvobj::transaction::run(pop, [&] {
			nvobj::delete_persistent<S>(r->s1);
			nvobj::delete_persistent<S>(r->s2);
			nvobj::delete_persistent<S>(r->s3);
			nvobj::delete_persistent<S>(r->s4);
		});
	} catch (std::exception &e) {
		UT_FATALexc(e);
	}
}

int
main(int argc, char *argv[])
{
	START();

	if (argc != 2)
		UT_FATAL("usage: %s file-name", argv[0]);

	const char *path = argv[1];

	pmem::obj::pool<root> pop;

	try {
		pop = pmem::obj::pool<root>::create(path, "basic_string",
						    PMEMOBJ_MIN_POOL,
						    S_IWUSR | S_IRUSR);
	} catch (...) {
		UT_FATAL("!pmemobj_create: %s", path);
	}

	run(pop);

	pop.close();

	return 0;
}
