/* ConfigurationLoader.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.menus;

import be.ugent.caagt.swirl.actions.Description;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Loads configurations from file.
 */
class ConfigurationLoader {

    //
    private final Map<String, Iterable<EntryConfiguration>> roots;

    //
    private final Map<String, ActionConfiguration> actions;
    //
    private final Map<String, ToggleConfiguration> toggles;
    //
    private final Map<String, GroupConfiguration> groups;

    //
    private String bundleName;

    public synchronized void setBundleName(String bundleName) {
        this.bundleName = bundleName;
    }

    public ConfigurationLoader() {
        this.roots = new HashMap<String, Iterable<EntryConfiguration>>();
        this.actions = new HashMap<String, ActionConfiguration>();
        this.toggles = new HashMap<String, ToggleConfiguration>();
        this.groups = new HashMap<String, GroupConfiguration>();
    }

    /**
     * Return the list of configuration elements that
     * belong to a root element with the given identifier.
     */
    public Iterable<EntryConfiguration> getRoot(String id) {
        return roots.get(id);
    }

    /**
     * Resolves dtds from the class path.
     */
    private static class SWIRLEntityResolver implements EntityResolver {

        SWIRLEntityResolver() {
        // avoids creating extra interfaces
        }

        public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
            if (publicId.startsWith("-//SWIRL//")) {
                int pos = systemId.lastIndexOf('/');
                String resourceName = "/be/ugent/caagt/swirl/dtds/" + systemId.substring(pos + 1);
                return new InputSource(MenuBuilder.class.getResourceAsStream(resourceName));
            } else {
                return null;
            }
        }
    }
    //
    private static final SAXBuilder BUILDER;

    static {
        BUILDER = new SAXBuilder(true);
        BUILDER.setEntityResolver(new SWIRLEntityResolver());
    }

    public void load(String resource) throws JDOMException, IOException  {
        InputStream inputStream = MenuBuilder.class.getResourceAsStream(resource);
        if (inputStream == null) {
            throw new MenuConfigurationException("Could not open menu configuration resource: " + resource);
        } else {
            load(inputStream);
        }
    }

    public synchronized void load(InputStream inputStream) throws JDOMException, IOException {
        Document document = BUILDER.build(inputStream);
        Element root = document.getRootElement();
        for (Object obj : root.getChildren()) {
            registerElement((Element) obj);
        }
    }

    /**
     * Process a single JDOM element and return the corresponding configuration.
     * @param element Element to be processed
     */
    private void registerElement(Element element) {
        String name = element.getName();
        if ("root".equals(name)) {
            readRootElement(element);
        } else if ("action".equals(name)) {
            readActionElement(element);
        } else if ("toggle".equals(name)) {
            readToggleElement(element);
        } else if ("group".equals(name)) {
            readGroupElement(element);
        } else {
            throw new MenuConfigurationException("Invalid top level element in configuration file: " + name);
        }
    }

    /**
     * Process a single JDOM element and return the corresponding configuration.
     * @param element Element to be processed
     */
    private EntryConfiguration readElement(Element element) {
        String name = element.getName();
        if ("menu".equals(name)) {
            return readMenuElement(element);
        } else if ("action".equals(name)) {
            return readActionElement(element);
        } else if ("toggle".equals(name)) {
            return readToggleElement(element);
        } else if ("group".equals(name)) {
            return readGroupElement(element);
        } else if ("separator".equals(name)) {
            return readSeparatorElement(element);
        } else if ("action-ref".equals(name)) {
            return readActionReferenceElement(element);
        } else if ("toggle-ref".equals(name)) {
            return readToggleReferenceElement(element);
        } else if ("group-ref".equals(name)) {
            return readGroupReferenceElement(element);
        } else {
            throw new MenuConfigurationException("Unknown element in configuration file: " + name);
        }
    }

    private void readRootElement(Element element) {
        String id = element.getAttributeValue("id");
        if (roots.containsKey(id)) {
            throw new MenuConfigurationException("Root id should be unique: " + id);
        }
        List<EntryConfiguration> result = new ArrayList<EntryConfiguration>();
        for (Object obj : element.getChildren()) {
            result.add(readElement((Element) obj));
        }
        roots.put(id, result);
    }

    /**
     * Process the predicate information in the given element and add it to
     * the result.
     */
    private void readPredicate(Element element, EntryConfiguration result) {
        String unless = element.getAttributeValue("unless");
        String when = element.getAttributeValue("when");
        if (when == null) {
            if (unless == null) {
                result.setPredicate(null, true);
            } else {
                result.setPredicate(unless, false);
            }
        } else {
            if (unless == null) {
                result.setPredicate(when, true);
            } else {
                throw new MenuConfigurationException("only one of 'when' and 'unless' is allowed in configuration");
            }
        }
    }

    private String readIdentifier(Element element, IdentifiedConfiguration result) {
        String identifier = element.getAttributeValue("id");
        result.setId(identifier);
        return identifier;
    }

    private ActionReferenceConfiguration readActionReferenceElement(Element element) {
        ActionReferenceConfiguration result = new ActionReferenceConfiguration();
        String id = readIdentifier(element, result);
        readPredicate(element, result);
        if (actions.containsKey(id)) {
            result.setActionConfiguration(actions.get(id));
        } else {
            throw new MenuConfigurationException("Reference to action with unknown identifier: " + id);
        }
        return result;
    }

    private ToggleReferenceConfiguration readToggleReferenceElement(Element element) {
        ToggleReferenceConfiguration result = new ToggleReferenceConfiguration();
        String id = readIdentifier(element, result);
        readPredicate(element, result);
        if (toggles.containsKey(id)) {
            result.setToggleConfiguration(toggles.get(id));
        } else {
            throw new MenuConfigurationException("Reference to toggle with unknown identifier: " + id);
        }
        return result;
    }

    private GroupReferenceConfiguration readGroupReferenceElement(Element element) {
        GroupReferenceConfiguration result = new GroupReferenceConfiguration();
        String id = readIdentifier(element, result);
        readPredicate(element, result);
        if (groups.containsKey(id)) {
            result.setGroupConfiguration(groups.get(id));
        } else {
            throw new MenuConfigurationException("Reference to group with unknown identifier: " + id);
        }
        return result;
    }

    private SeparatorConfiguration readSeparatorElement(Element element) {
        SeparatorConfiguration result = new SeparatorConfiguration();
        readPredicate(element, result);
        if ("glue".equals(element.getAttributeValue("type"))) {
            result.setGlue(true);
        }
        return result;
    }

    private String readI18nElement(Element parent, String name) {
        Element el = parent.getChild(name);
        if (el == null) {
            return null;
        } else {
            String key = el.getAttributeValue("i18n");
            if (key == null) {
                return null;
            } else {
                return ResourceBundle.getBundle(bundleName).getString(key);
            }
        }
    }

    private Icon readIconElement(Element parent, String name) {
        Element el = parent.getChild(name);
        if (el == null) {
            return null;
        } else {
            try {
                URL url = new URL(el.getAttributeValue("url"));
                Icon result = new ImageIcon(url);
                if (result == null) {
                    throw new MenuConfigurationException("Could not find icon: " + url);
                } else {
                    return result;
                }
            } catch (MalformedURLException ex) {
                throw new MenuConfigurationException("Invalid icon URL ", ex);
            }
        }
    }

    private FullDescription readFullDescription(Element element) {
        String description = readI18nElement(element, "caption");
        String tooltip = readI18nElement(element, "tooltip");

        return new FullDescription(description,
                readIconElement(element, "menu-icon"),
                readIconElement(element, "menu-icon-disabled"),
                readIconElement(element, "icon"),
                readIconElement(element, "icon-disabled"),
                tooltip);
    }

    private ActionConfiguration readActionElement(Element element) {
        ActionConfiguration result = new ActionConfiguration();
        readIdentifier(element, result);
        readPredicate(element, result);
        result.setDescription(readFullDescription(element));
        actions.put(result.getId(), result);
        return result;
    }

    private ToggleConfiguration readToggleElement(Element element) {
        ToggleConfiguration result = new ToggleConfiguration();
        readIdentifier(element, result);
        readPredicate(element, result);
        result.setDescription(readFullDescription(element));
        toggles.put(result.getId(), result);
        return result;
    }

    private ButtonConfiguration readButtonElement(Element element) {
        ButtonConfiguration result = new ButtonConfiguration();
        readIdentifier(element, result);
        readPredicate(element, result);
        result.setDescription(readFullDescription(element));
        return result;
    }

    private GroupConfiguration readGroupElement(Element element) {
        GroupConfiguration result = new GroupConfiguration();
        readIdentifier(element, result);
        readPredicate(element, result);
        String clearable = element.getAttributeValue("clearable");
        result.setClearable ("true".equals(clearable));
        for (Object object : element.getChildren()) {
            Element child = (Element) object;
            if ("button".equals(child.getName())) {
                result.add(readButtonElement(child));
            }
        }
        groups.put(result.getId(), result);
        return result;
    }

    private MenuConfiguration readMenuElement(Element element) {
        MenuConfiguration result = new MenuConfiguration();
        readPredicate(element, result);
        result.setName(element.getAttributeValue("name"));
        String key = element.getAttributeValue("i18n");
        if (key != null) {
            String description = ResourceBundle.getBundle(bundleName).getString(key);
            result.setDescription(new Description(description));
        }
        for (Object obj : element.getChildren()) {
            result.add(readElement((Element) obj));
        }
        return result;
    }
}
