package org.debian.maven.packager;

/*
 * Copyright 2009 Ludovic Claude.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.model.Developer;
import org.apache.maven.model.License;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import java.io.*;
import java.util.*;

import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.project.MavenProject;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.debian.maven.repo.ListOfPOMs;

/**
 * Generate the Debian files for packaging the current Maven project.
 *
 * @goal generate
 * @aggregator
 * @requiresDependencyResolution
 * @phase process-sources
 * 
 * @author Ludovic Claude
 */
public class GenerateDebianFilesMojo
        extends AbstractMojo {

    /**
     * The Maven Project Object
     *
     * @parameter expression="${project}"
     * @readonly
     * @required
     */
    protected MavenProject project;
    /**
     * A list of every project in this reactor; provided by Maven
     * @parameter expression="${project.collectedProjects}"
     */
    protected List collectedProjects;
    /**
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepository;
    /**
     * Location of the file.
     * @parameter expression="${debian.directory}"
     *   default-value="debian"
     */
    protected File outputDirectory;
    /**
     * Name of the packager (e.g. 'Ludovic Claude')
     * @parameter expression="${packager}"
     * @required
     */
    protected String packager;
    /**
     * Email of the packager (e.g. 'ludovic.claude@laposte.net')
     * @parameter expression="${email}"
     * @required
     */
    protected String email;
    /**
     * License used by the packager (e.g. 'GPL-3' or 'Apache-2.0')
     * See http://dep.debian.net/deps/dep5/ for the list of licenses.
     * @parameter expression="${packagerLicense}" default-value="GPL-3"
     * @required
     */
    protected String packagerLicense;
    /**
     * Name of the source package (e.g. 'commons-lang')
     * @parameter expression="${package}"
     * @required
     */
    protected String packageName;
    /**
     * Name of the binary package (e.g. 'libcommons-lang-java')
     * @parameter expression="${bin.package}"
     * @required
     */
    protected String binPackageName;
    /**
     * Type of the package (e.g. 'maven' or 'ant')
     * @parameter expression="${packageType}" default-value="maven"
     */
    protected String packageType;
    /**
     * URL for downloading the source code, in the form scm:[svn|cvs]:http://xxx/
     * for downloads using a source code repository,
     * or http://xxx.[tar|zip|gz|tgz] for downloads using source tarballs.
     * @parameter expression="${downloadUrl}"
     */
    protected String downloadUrl;
    /**
     * If true, include running the tests during the build.
     * @parameter expression="${runTests}" default-value="false"
     */
    protected boolean runTests;
    /**
     * If true, generate the Javadoc packaged in a separate package.
     * @parameter expression="${generateJavadoc}" default-value="false"
     */
    protected boolean generateJavadoc;

    public void execute()
            throws MojoExecutionException {
        File f = outputDirectory;
        if (!f.exists()) {
            f.mkdirs();
        }

        String controlTemplate = "control.vm";
        String rulesTemplate = "rules.vm";
        if ("ant".equals(packageType)) {
            controlTemplate = "control.ant.vm";
            rulesTemplate = "rules.ant.vm";
        }
        // #638788: clean up email
        if (email != null && email.indexOf('<') >= 0 && email.indexOf('>') >= 0) {
            email = email.substring(email.indexOf('<') + 1, email.indexOf('>') - 1);
        }

        try {
            Properties velocityProperties = new Properties();
            velocityProperties.put("resource.loader", "class");
            velocityProperties.put("class.resource.loader.class", "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
            Velocity.init(velocityProperties);
            VelocityContext context = new VelocityContext();
            context.put("package", packageName);
            context.put("packageType", packageType);
            context.put("binPackage", binPackageName);
            context.put("packager", packager);
            context.put("packagerEmail", email);
            context.put("project", project);
            context.put("collectedProjects", wrapMavenProjects(collectedProjects));
            context.put("runTests", Boolean.valueOf(runTests));
            context.put("generateJavadoc", Boolean.valueOf(generateJavadoc));

            if (project.getName() == null || project.getName().isEmpty()) {
                System.out.println("POM does not contain the project name. Please enter the name of the project:");
                project.setName(readLine());
            }
            if (project.getUrl() == null || project.getUrl().isEmpty()) {
                System.out.println("POM does not contain the project URL. Please enter the URL of the project:");
                project.setUrl(readLine());
            }

            Set licenses = discoverLicenses();
            context.put("licenses", licenses);

            if (licenses.size() == 1) {
                packagerLicense = (String) licenses.iterator().next();
            }
            if (packagerLicense == null) {
                System.out.println("Packager license for the debian/ filse was not found, please enter a license name preferably in one of:");
                System.out.println("Apache Artistic BSD FreeBSD ISC CC-BY CC-BY-SA CC-BY-ND CC-BY-NC CC-BY-NC-SA CC-BY-NC-ND CC0 CDDL CPL Eiffel");
                System.out.println("Expat GPL LGPL GFDL GFDL-NIV LPPL MPL Perl PSF QPL W3C-Software ZLIB Zope");
                String s = readLine();
                if (s.length() > 0) {
                    packagerLicense = s;
                }
            }
            context.put("packagerLicense", packagerLicense);

            String copyrightOwner = "";
            String projectTeam = "";
            if (project.getOrganization() != null) {
                copyrightOwner = project.getOrganization().getName();
                projectTeam = project.getOrganization().getName() + " developers";
            }
            if (copyrightOwner == null || copyrightOwner.isEmpty()) {
                Iterator devs = project.getDevelopers().iterator();
                if (devs.hasNext()) {
                    Developer dev = (Developer) devs.next();
                    copyrightOwner = dev.getName();
                    if (dev.getEmail() != null && !dev.getEmail().isEmpty()) {
                        copyrightOwner += " <" + dev.getEmail() + ">";
                    }
                }
            }
            if (copyrightOwner == null || copyrightOwner.isEmpty()) {
                System.out.println("Could not find who owns the copyright for the upstream sources, please enter his name:");
                copyrightOwner = readLine();
            }
            context.put("copyrightOwner", copyrightOwner);

            if (projectTeam == null || projectTeam.isEmpty()) {
                projectTeam = project.getName() + " developers";
            }
            context.put("projectTeam", projectTeam);

            String copyrightYear;
            int currentYear = new GregorianCalendar().get(Calendar.YEAR);
            if (project.getInceptionYear() != null) {
                copyrightYear = project.getInceptionYear();
                if (Integer.parseInt(copyrightYear) < currentYear) {
                    copyrightYear += "-" + currentYear;
                }
            } else {
                copyrightYear = String.valueOf(currentYear);
            }
            context.put("copyrightYear", copyrightYear);
            context.put("currentYear", new Integer(currentYear));

            List description = new ArrayList();
            if (project.getDescription() == null || project.getDescription().trim().isEmpty()) {
                System.out.println("Please enter a short description of the project, press Enter twice to stop.");
                StringBuffer sb = new StringBuffer();
                int emptyEnterCount = 0;
                while (emptyEnterCount < 2) {
                    String s = readLine();
                    if (s.isEmpty()) {
                        emptyEnterCount++;
                    } else {
                        if (emptyEnterCount > 0) {
                            emptyEnterCount = 0;
                            sb.append("\n");
                        }
                        sb.append(s);
                        sb.append("\n");
                    }
                }
                project.setDescription(sb.toString());
            }
            if (project.getDescription() != null) {
                StringTokenizer st = new StringTokenizer(project.getDescription().trim(), "\n\t ");
                StringBuffer descLine = new StringBuffer();
                while (st.hasMoreTokens()) {
                    descLine.append(st.nextToken());
                    descLine.append(" ");
                    if (descLine.length() > 70 || !st.hasMoreTokens()) {
                        String line = descLine.toString().trim();
                        if (line.isEmpty()) {
                            line = ".";
                        }
                        description.add(line);
                        descLine = new StringBuffer();
                    }
                }
            }
            context.put("description", description);

            File substvarsFile = new File(outputDirectory, binPackageName + ".substvars");
            if (substvarsFile.exists()) {
                Properties substvars = new Properties();
                substvars.load(new FileReader(substvarsFile));
                List compileDepends = new ArrayList();
                compileDepends.addAll(split(substvars.getProperty("maven.CompileDepends")));
                compileDepends.addAll(split(substvars.getProperty("maven.Depends")));
                List buildDepends = new ArrayList(compileDepends);
                List testDepends = new ArrayList(split(substvars.getProperty("maven.TestDepends")));
                if (runTests) {
                    buildDepends.addAll(testDepends);
                }
                if (generateJavadoc) {
                    buildDepends.addAll(split(substvars.getProperty("maven.DocDepends")));
                    buildDepends.addAll(split(substvars.getProperty("maven.DocOptionalDepends")));
                }
                if ("maven".equals(packageType)) {
                    boolean seenJavadocPlugin = false;
                    // Remove dependencies that are implied by maven-debian-helper
                    for (Iterator i = buildDepends.iterator(); i.hasNext();) {
                        String dependency = (String) i.next();
                        if (dependency.startsWith("libmaven-clean-plugin-java") ||
                                dependency.startsWith("libmaven-resources-plugin-java") ||
                                dependency.startsWith("libmaven-compiler-plugin-java") ||
                                dependency.startsWith("libmaven-jar-plugin-java") ||
                                dependency.startsWith("libmaven-site-plugin-java") ||
                                dependency.startsWith("libsurefire-java") ||
                                dependency.startsWith("velocity") ||
                                dependency.startsWith("libplexus-velocity-java")) {
                            i.remove();
                        } else if (dependency.startsWith("libmaven-javadoc-plugin-java")) {
                            seenJavadocPlugin = true;
                        }
                    }
                    if (generateJavadoc && !seenJavadocPlugin) {
                        buildDepends.add("libmaven-javadoc-plugin-java");
                    }
                } else if ("ant".equals(packageType)) {
                    // Remove dependencies that are implied by ant packaging
                    for (Iterator i = buildDepends.iterator(); i.hasNext(); ) {
                        String dependency = (String) i.next();
                        if (dependency.equals("ant") ||
                                dependency.startsWith("ant ") ||
                                dependency.startsWith("ant-optional")) {
                            i.remove();
                        }
                    }
                    buildDepends.remove("ant");
                    buildDepends.remove("ant-optional");
                }
                context.put("buildDependencies", buildDepends);
                context.put("runtimeDependencies", split(substvars.getProperty("maven.Depends")));
                context.put("testDependencies", split(substvars.getProperty("maven.TestDepends")));
                context.put("optionalDependencies", split(substvars.getProperty("maven.OptionalDepends")));
                context.put("javadocDependencies", split(substvars.getProperty("maven.DocDepends")));
                context.put("javadocOptionalDependencies", split(substvars.getProperty("maven.DocOptionalDepends")));

                if ("ant".equals(packageType)) {
                    Set compileJars = new TreeSet();
                    for (Iterator i = compileDepends.iterator(); i.hasNext();) {
                        String library = (String) i.next();
                        compileJars.addAll(listSharedJars(library));
                    }
                    context.put("compileJars", compileJars);
                    Set testJars = new TreeSet();
                    for (Iterator i = testDepends.iterator(); i.hasNext();) {
                        String library = (String) i.next();
                        testJars.addAll(listSharedJars(library));
                    }
                    context.put("testJars", testJars);
                }
            } else {
                System.err.println("Cannot find file " + substvarsFile);
            }

            if ("ant".equals(packageType)) {
                ListOfPOMs listOfPOMs = new ListOfPOMs(new File(outputDirectory, binPackageName + ".poms"));
                ListOfPOMs listOfJavadocPOMs = null;
                if (generateJavadoc && "ant".equals(packageType)) {
                    listOfJavadocPOMs =  new ListOfPOMs(new File(outputDirectory, binPackageName + "-doc.poms"));
                }
                setupArtifactLocation(listOfPOMs, listOfJavadocPOMs, project);
                for (Iterator i = collectedProjects.iterator(); i.hasNext();) {
                    MavenProject mavenProject = (MavenProject) i.next();
                    setupArtifactLocation(listOfPOMs, listOfJavadocPOMs, mavenProject);
                }
                listOfPOMs.save();
                if (listOfJavadocPOMs != null) {
                    listOfJavadocPOMs.save();
                }
            }

            String projectVersion = project.getVersion();
            int downloadType = DownloadType.UNKNOWN;

            if (downloadUrl == null) {
                if (project.getScm() != null) {
                    downloadUrl = project.getScm().getConnection();
                }
            }
            if (downloadUrl != null && downloadUrl.startsWith("scm:svn:")) {
                downloadType = DownloadType.SVN;
                downloadUrl = downloadUrl.substring("scm:svn:".length());
                String tag = projectVersion;
                int tagPos = downloadUrl.indexOf(tag);
                String baseUrl = null;
                String suffixUrl = null;
                String tagMarker = null;
                if (tagPos >= 0) {
                    baseUrl = downloadUrl.substring(0, tagPos);
                    suffixUrl = downloadUrl.substring(tagPos + tag.length());
                    if (!suffixUrl.endsWith("/")) {
                        suffixUrl += "/";
                    }
                    int slashPos = baseUrl.lastIndexOf("/");
                    tagMarker = baseUrl.substring(slashPos + 1);
                    baseUrl = baseUrl.substring(0, slashPos);
                }
                if (tagPos < 0 && downloadUrl.indexOf("/trunk") >= 0) {
                    System.out.println("Download URL does not include a tagged revision but /trunk found,");
                    System.out.println("Trying to guess the address of the tagged revision.");
                    tag = "trunk";
                    tagPos = downloadUrl.indexOf(tag);
                    baseUrl = downloadUrl.substring(0, tagPos);
                    baseUrl += "tags";
                    tagMarker = packageName  + "-";
                    suffixUrl = "";
                }
                if (tagPos >= 0) {
                    context.put("baseUrl", baseUrl);
                    context.put("tagMarker", tagMarker);
                    context.put("suffixUrl", suffixUrl);

                    generateFile(context, "watch.svn.vm", outputDirectory, "watch");
                    generateFile(context, "orig-tar.svn.vm", outputDirectory, "orig-tar.sh");

                    makeExecutable("debian/orig-tar.sh");

                } else {
                    System.err.println("Cannot locate the version in the download url (" +
                            downloadUrl + ").");
                    System.err.println("Please run again and provide the download location with an explicit version tag, e.g.");
                    System.err.println("-DdownloadUrl=scm:svn:http://svn.codehaus.org/modello/tags/modello-1.0-alpha-21/");
                }
            }

            if (downloadType == DownloadType.UNKNOWN) {
                System.err.println("Cannot recognize the download url (" +
                        downloadUrl + ").");
            }

            generateFile(context, "README.source.vm", outputDirectory, "README.source");
            generateFile(context, "copyright.vm", outputDirectory, "copyright");
            generateFile(context, "compat.vm", outputDirectory, "compat");
            generateFile(context, rulesTemplate, outputDirectory, "rules");

            makeExecutable("debian/rules");

            String debianVersion = projectVersion.replace("-alpha-", "~alpha");
            debianVersion = debianVersion.replace("-beta-", "~beta");
            debianVersion += "-1";
            context.put("version.vm", debianVersion);

            generateFile(context, rulesTemplate, new File("."), ".debianVersion");

            if (generateJavadoc) {
                if (project.getPackaging().equals("pom") && collectedProjects.size() > 1) {
                    generateFile(context, "java-doc.doc-base.api.multi.vm", outputDirectory, binPackageName + "-doc.doc-base.api");
                    generateFile(context, "java-doc.install.multi.vm", outputDirectory, binPackageName + "-doc.install");
                } else {
                  generateFile(context, "java-doc.doc-base.api.vm", outputDirectory, binPackageName + "-doc.doc-base.api");
                  generateFile(context, "java-doc.install.vm", outputDirectory, binPackageName + "-doc.install");
                }
            }

            if ("ant".equals(packageType)) {
                boolean containsJars = false;
                boolean containsPlugins = false;
                if (project.getPackaging().equals("pom") && project.getModules().size() > 0) {
                    for (Iterator i = collectedProjects.iterator(); i.hasNext(); ) {
                        MavenProject module = (MavenProject) i.next();
                        if (module.getPackaging().equals("maven-plugin")) {
                            containsPlugins = true;
                        } else if (!module.getPackaging().equals("pom")) {
                            containsJars = true;
                        }
                    }
                } else if (!project.getPackaging().equals("pom")) {
                    if (project.getPackaging().equals("maven-plugin")) {
                        containsPlugins = true;
                    } else if (!project.getPackaging().equals("pom")) {
                        containsJars = true;
                    }
                }
                context.put("containsJars", Boolean.valueOf(containsJars));
                context.put("containsPlugins", Boolean.valueOf(containsPlugins));

                if (project.getPackaging().equals("pom") && project.getModules().size() > 0) {
                    generateFile(context, "build.xml.vm", outputDirectory, "build.xml");
                }
                generateFile(context, "build.properties.ant.vm", outputDirectory, "build.properties");
                generateFile(context, "build-classpath.vm", outputDirectory, "build-classpath");
            } else {
                generateFile(context, "maven.properties.vm", outputDirectory, "maven.properties");
            }

            generateFile(context, controlTemplate, outputDirectory, "control");
            generateFile(context, "format.vm", new File(outputDirectory, "source"), "format");

        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    private List wrapMavenProjects(List<MavenProject> projects) {
        List<WrappedProject> wrappedProjects = new ArrayList<WrappedProject>();
        for (MavenProject aProject: projects) {
            wrappedProjects.add(new WrappedProject(this.project, aProject));
        }
        return wrappedProjects;
    }

    private void setupArtifactLocation(ListOfPOMs listOfPOMs, ListOfPOMs listOfJavadocPOMs, MavenProject mavenProject) {
        String dirRelPath = new WrappedProject(project, mavenProject).getBaseDir();

        if (! "pom".equals(mavenProject.getPackaging())) {
            String pomFile = dirRelPath + "pom.xml";
            listOfPOMs.getOrCreatePOMOptions(pomFile).setJavaLib(true);
            String extension = mavenProject.getPackaging();
            if (extension.equals("bundle")) {
                extension = "jar";
            }
            if (extension.equals("webapp")) {
                extension = "war";
            }
            if (mavenProject.getArtifact() != null && mavenProject.getArtifact().getFile() != null) {
                extension = mavenProject.getArtifact().getFile().toString();
                extension = extension.substring(extension.lastIndexOf('.') + 1);
            }
            ListOfPOMs.POMOptions pomOptions = listOfPOMs.getOrCreatePOMOptions(pomFile);
            pomOptions.setArtifact(dirRelPath + "target/" + mavenProject.getArtifactId() + "-*."
                + extension);
            if ("jar".equals(extension) && generateJavadoc && "ant".equals(packageType) && listOfJavadocPOMs != null) {
                String artifactId = mavenProject.getArtifact().getArtifactId();
                ListOfPOMs.POMOptions javadocPomOptions = listOfJavadocPOMs.getOrCreatePOMOptions(pomFile);
                javadocPomOptions.setIgnorePOM(true);
                javadocPomOptions.setArtifact(dirRelPath + "target/" + artifactId + ".javadoc.jar");
                javadocPomOptions.setClassifier("javadoc");
                javadocPomOptions.setHasPackageVersion(pomOptions.getHasPackageVersion());
            }
            pomOptions.setJavaLib(true);
            if (mavenProject.getArtifactId().matches(packageName + "\\d")) {
                pomOptions.setUsjName(packageName);
            }
        }
    }

    private Set discoverLicenses() {
        Set licenses = new TreeSet();
        for (Iterator i = project.getLicenses().iterator(); i.hasNext(); ) {
            License license = (License) i.next();
            String licenseName = "";
            if (license.getName() != null) {
                licenseName = license.getName() + " ";
            }
            String licenseUrl = "";
            if (license.getUrl() != null) {
                licenseUrl = license.getUrl();
            }
            boolean recognized = recognizeLicense(licenses, licenseName, licenseUrl);
            if (!recognized) {
                System.out.println("License " + licenseName + licenseUrl + " was not recognized, please enter a license name preferably in one of:");
                printAvailableLicenses();
                System.out.print("> ");
                String s = readLine();
                if (s.length() > 0) {
                    licenses.add(s);
                }
            }
        }

        System.out.println();
        System.out.println("Checking licenses in the upstream sources...");
        LicenseCheckResult licenseResult = new LicenseCheckResult();
        DependenciesSolver.executeProcess(new String[]{"/bin/sh", "-c", "licensecheck `find . -type f`"},
                licenseResult);
        for (Iterator i = licenseResult.getLicenses().iterator(); i.hasNext(); ) {
            String license = (String) i.next();
            boolean recognized = recognizeLicense(licenses, license, "");
            if (!recognized) {
                System.out.println("License " + license + " was not recognized, please enter a license name preferably in one of:");
                printAvailableLicenses();
                System.out.print("> ");
                String s = readLine();
                if (s.length() > 0) {
                    licenses.add(s);
                }
            }
        }

        if (licenses.isEmpty()) {
            System.out.println("License was not found, please enter a license name preferably in one of:");
            printAvailableLicenses();
            System.out.print("> ");
            String s = readLine();
            if (s.length() > 0) {
                licenses.add(s);
            }
        }
        return licenses;
    }

    private void printAvailableLicenses() {
        System.out.println("Apache-2.0 Artistic BSD FreeBSD ISC CC-BY CC-BY-SA CC-BY-ND CC-BY-NC CC-BY-NC-SA");
        System.out.println("CC-BY-NC-ND CC0 CDDL CPL Eiffel Expat GPL-2 GPL-3 LGPL-2 LGPL-2.1 LGPL-3");
        System.out.println("GFDL-1.2 GFDL-1.3 GFDL-NIV LPPL MPL Perl PSF QPL W3C-Software ZLIB Zope");
    }

    boolean recognizeLicense(Set licenses, String licenseName, String licenseUrl) {
        boolean recognized = false;
        licenseName = licenseName.toLowerCase();
        licenseUrl = licenseUrl.toLowerCase();
        if (licenseName.indexOf("mit ") >= 0 || licenseUrl.indexOf("mit-license") >= 0) {
            licenses.add("MIT");
            recognized = true;
        } else if (licenseName.indexOf("bsd ") >= 0 || licenseUrl.indexOf("bsd-license") >= 0) {
            licenses.add("BSD");
            recognized = true;
        } else if (licenseName.indexOf("artistic ") >= 0 || licenseUrl.indexOf("artistic-license") >= 0) {
            licenses.add("Artistic");
            recognized = true;
        } else if (licenseName.indexOf("apache ") >= 0 || licenseUrl.indexOf("apache") >= 0) {
            if (licenseName.indexOf("2.") >= 0 || licenseUrl.indexOf("2.") >= 0) {
                licenses.add("Apache-2.0");
                recognized = true;
            } else if (licenseName.indexOf("1.0") >= 0 || licenseUrl.indexOf("1.0") >= 0) {
                licenses.add("Apache-1.0");
                recognized = true;
            } else if (licenseName.indexOf("1.1") >= 0 || licenseUrl.indexOf("1.1") >= 0) {
                licenses.add("Apache-1.1");
                recognized = true;
            }
        } else if (licenseName.indexOf("lgpl ") >= 0 || licenseUrl.indexOf("lgpl") >= 0) {
            if (licenseName.indexOf("2.1") >= 0 || licenseUrl.indexOf("2.1") >= 0) {
                licenses.add("LGPL-2.1");
                recognized = true;
            } else if (licenseName.indexOf("2") >= 0 || licenseUrl.indexOf("2") >= 0) {
                licenses.add("LGPL-2");
                recognized = true;
            } else if (licenseName.indexOf("3") >= 0 || licenseUrl.indexOf("3") >= 0) {
                licenses.add("LGPL-2");
                recognized = true;
            }
        } else if (licenseName.indexOf("gpl ") >= 0 || licenseUrl.indexOf("gpl") >= 0) {
            if (licenseName.indexOf("2") >= 0 || licenseUrl.indexOf("2") >= 0) {
                licenses.add("GPL-2");
                recognized = true;
            } else if (licenseName.indexOf("3") >= 0 || licenseUrl.indexOf("3") >= 0) {
                licenses.add("GPL-3");
                recognized = true;
            }

        } else if (licenseUrl.indexOf("http://creativecommons.org/licenses/by-sa/3.0") >= 0) {
            licenses.add("CC-BY-SA-3.0");
            recognized = true;
        }
        return recognized;
    }

    private void generateFile(VelocityContext context, String templateName, File destDir, String fileName) throws Exception {
        destDir.mkdirs();
        FileWriter out = new FileWriter(new File(destDir, fileName));
        Velocity.mergeTemplate(templateName, "UTF8", context, out);
        out.flush();
        out.close();
    }

    private Map<String, List<String>> cacheOfSharedJars = new HashMap<String, List<String>>();
    private List<String> listSharedJars(String library) {
        if (cacheOfSharedJars.get(library) != null) {
            return cacheOfSharedJars.get(library);
        }

        final List<String> jars = new ArrayList<String>();
        if (library.indexOf("(") > 0) {
            library = library.substring(0, library.indexOf("(")).trim();
        }
        System.out.println();
        System.out.println("Looking for shared jars in package " + library + "...");
        DependenciesSolver.executeProcess(new String[]{"/usr/bin/dpkg", "--listfiles", library},
                new DependenciesSolver.OutputHandler() {

                    public void newLine(String line) {
                        if (line.startsWith("/usr/share/java/") && line.endsWith(".jar")) {
                            String jar = line.substring("/usr/share/java/".length());
                            jar = jar.substring(0, jar.length() - 4);
                            if (!line.matches(".*/.*-\\d.*")) {
                                jars.add(jar);
                                System.out.println("  Add " + jar + " to the classpath");
                            }
                        }
                    }
                });
        cacheOfSharedJars.put(library, jars);
        return jars;
    }

    private String readLine() {
        LineNumberReader consoleReader = new LineNumberReader(new InputStreamReader(System.in));
        try {
            return consoleReader.readLine().trim();
        } catch (IOException e) {
            e.printStackTrace();
            return "";
        }
    }

    private List split(String s) {
        List l = new ArrayList();
        if (s != null) {
            StringTokenizer st = new StringTokenizer(s, ",");
            while (st.hasMoreTokens()) {
                l.add(st.nextToken().trim());
            }
        }
        return l;
    }

    private void makeExecutable(String file) {
        DependenciesSolver.executeProcess(new String[]{"chmod", "+x", file}, new DependenciesSolver.NoOutputHandler());
    }

    public static class WrappedProject {
        private final MavenProject baseProject;
        private final MavenProject mavenProject;

        public WrappedProject(MavenProject baseProject, MavenProject mavenProject) {
            this.baseProject = baseProject;
            this.mavenProject = mavenProject;
        }

        public String getBaseDir() {
            String basedir = baseProject.getBasedir().getAbsolutePath();
            String dirRelPath = "";
            if (! mavenProject.getBasedir().equals(baseProject.getBasedir())) {
                dirRelPath = mavenProject.getBasedir().getAbsolutePath().substring(basedir.length() + 1) + "/";
            }
            return dirRelPath;
        }

        public String getArtifactId() {
            return mavenProject.getArtifactId();
        }

        public String getGroupId() {
            return mavenProject.getGroupId();
        }

        public String getVersion() {
            return mavenProject.getVersion();
        }

        public String getPackaging() {
            return mavenProject.getPackaging();
        }
    }

    static class LicenseCheckResult implements DependenciesSolver.OutputHandler {

        private Set licenses = new TreeSet();
        private Set copyrightOwners = new TreeSet();

        public void newLine(String line) {
            if (line.startsWith(".") && line.indexOf(":") > 0) {
                int col = line.lastIndexOf(":");
                String license = line.substring(col + 1).trim();
                if (license.indexOf("UNKNOWN") >= 0) {
                    return;
                }
                if (license.indexOf("*") >= 0) {
                    license = license.substring(license.lastIndexOf("*") + 1).trim();
                }
                licenses.add(license);
            }
        }

        public Set getLicenses() {
            return licenses;
        }

        public Set getCopyrightOwners() {
            return copyrightOwners;
        }
    }

    interface DownloadType {

        int UNKNOWN = 0;
        int SVN = 1;
        int CVS = 2;
        int TARBALL = 3;
    }
}

