#!/usr/bin/python
# This script generates the LaTeX documentation for all libraries.
#
# IMPORTANT: This script should ONLY be called from the trunk/ directory
# of the mCRL2 source tree. It is checked whether this is the case.

import os
import sys
import glob
import subprocess

################### SETTINGS ###########################################

# Path to this script from the trunk directory. This variable is used
# for the check mentioned above.
path_from_trunk = "doc/tex"

# Directory in which the PDF files will be generated.
output_dir = "build/tex"

# List of all libraries for which documentation will be generated. Every
# entry is a list of two values:
# 1. the name of the subdirectory in the output directory in which PDF
#    files for that library are placed
# 2. path to the library in the mCRL2 source tree (relative to trunk).
library_list = [
    [ 'atermpp', 'libraries/atermpp' ],
    [ 'core', 'libraries/core' ],
    [ 'data', 'libraries/data' ],
    [ 'lps', 'libraries/lps' ],
    [ 'lts', 'libraries/lts' ],
    [ 'pbes', 'libraries/pbes' ],
    [ 'process', 'libraries/process' ],
    [ 'trace', 'libraries/trace' ],
    [ 'utilities', 'libraries/utilities' ] 
    ]

# Subdirectory of every library's directory that will be checked for
# LaTeX files.
tex_dir = "doc/tex"

# Required filename pattern for LaTeX files in that directory. Only
# LaTeX files matching this pattern will be considered for PDF
# generation.
tex_file_pattern = "wiki-*.tex"

# List of file name extensions that have to be cleaned up after
# LaTeX'ing
cleanup_exts = [ '.aux', '.bbl', '.blg', '.log' ]

################### END OF SETTINGS ####################################


# Function that ensures that a path points to an existing, writable
# directory.
def ensure_dir(path) :
  if os.path.exists(path) :
    if not os.path.isdir(path) :
      sys.exit("error: " + path + " exists but is not a directory")
    if not os.access(path, os.W_OK) :
      sys.exit("error: " + path + " is not a writable directory")
  else :
    os.mkdir(path)


# First check whether we're called from trunk
if not os.path.exists(path_from_trunk + "/" + os.path.basename(sys.argv[0]) ) :
  sys.exit("error: script is not called from the trunk directory of the mCRL2 repository")

# Store the absolute path to trunk
trunk = os.getcwd()

# Setup the output directory
output_dir = os.path.abspath(output_dir)
ensure_dir(output_dir)

for lib in library_list :
  lib_path = trunk + '/' + lib[1] + '/' + tex_dir
  if os.path.exists(lib_path) :
    os.chdir(lib_path)
    tex_files = glob.glob(tex_file_pattern)
    if len(tex_files) > 0 :
      lib_output_dir = output_dir + '/' + lib[0]
      ensure_dir(lib_output_dir)
      for tex_file in tex_files :
        tex_file = tex_file.rpartition('.')[0]
        # Call: PDFLaTeX, then BiBTeX, finally PDFLaTeX twice
        if subprocess.call(['pdflatex', '-interaction=batchmode', tex_file]) != 0 : continue
        # Document may not have a bibliography, so we don't bail out if
        # BiBTeX fails
        subprocess.call(['bibtex', '-terse', tex_file])
        if subprocess.call(['pdflatex', '-interaction=batchmode', tex_file]) != 0 : continue
        if subprocess.call(['pdflatex', '-interaction=batchmode', tex_file]) != 0 : continue
        # Move output PDF file to proper location
        os.rename(tex_file + '.pdf' , lib_output_dir + '/' + tex_file + '.pdf' )
        # Cleanup other files generated by LaTeX
        for ext in cleanup_exts :
          if os.path.exists(tex_file + ext) :
            os.remove(tex_file + ext)
