/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef VerbFunctionHelp_H
#define VerbFunctionHelp_H

#include <QIcon>
#include <QString>
#include <QStringList>

#include <vector>
#include <map>

#include "MvIconParameter.h"
#include "MvIconLanguage.h"

class IconClassHelp;
class LanguageHelper;

/* ---------------------------------------------------------------------------------------------

    The classes defined here are for the purpose of storing the information required to
    give the user help for specific functions which follow the pattern of verb-parameters-values.
    For example, in the Macro languate, we have things such as:
      pcont (contour : 'on')

    and we want to be able to provide the user with in-editor help for these functions.

   --------------------------------------------------------------------------------------------- */


//-------------------------------------------------------------------------------------
// VerbFunctionHelpParam
// class to store all the information associated with a parameter
//-------------------------------------------------------------------------------------

class VerbFunctionHelpParam : public ParameterScanner
{
public:
    VerbFunctionHelpParam(const MvIconParameter&);
    ~VerbFunctionHelpParam() {}

    QString name() const;
    QString interface() const;
    QString interfaceFunction() const;
    QStringList defaultValues() const;
    int numValues() const;
    QString valueName(int i) const;

protected:
    void next(const MvIconParameter&, const char* first, const char* second);

private:
    const MvIconParameter& param_;
    QStringList values_;
};


//-------------------------------------------------------------------------------------
// VerbFunctionHelpFunction
// class to store all the information associated with a function
//-------------------------------------------------------------------------------------

class VerbFunctionHelpFunction : public MvIconLanguageScanner
{
public:
    VerbFunctionHelpFunction(QString name, IconClassHelp* iconClass = nullptr);
    ~VerbFunctionHelpFunction();

    QString name() const {return name_;}
    QIcon icon() const;
    int numParams() const { return static_cast<int>(params_.size()); }
    QString paramName(int i) const { return params_[i]->name(); }

    VerbFunctionHelpParam* paramFromName(QString name) const;
    VerbFunctionHelpParam* param(int i) const;
    int indexOfParam(VerbFunctionHelpParam*) const;
    IconClassHelp* iconClass() const {return iconClass_;}
    void loadParams();

    QString descriptionAsStr() const;
    QString descriptionAsRt() const;
    QMap<QString, QString> descriptions() const {return descriptions_;}
    void setDescription(QString group, QString desc) {descriptions_[group] = desc;}

protected:
    void next(const MvIconParameter&);

private:
    QString name_;
    IconClassHelp* iconClass_;
    bool paramsLoaded_;
    std::vector<VerbFunctionHelpParam*> params_;
    std::map<QString, VerbFunctionHelpParam*> paramMap_;
    QMap<QString, QString> descriptions_;
};

//-------------------------------------------------------------------------------------
//    The VerbFunctionHelp class allows the loading of a specially-prepared XML file with
//    all the function definitions in it (this is in its constructor). It also provides
//    the ability to query whether the given string is the name of a function for which
//    we have help available, and an accessor function to get a VerbFunctionHelpFunction
//    pointer from a given name.
//-------------------------------------------------------------------------------------

class VerbFunctionHelp
{
public:
    VerbFunctionHelp(LanguageHelper*);
    ~VerbFunctionHelp();

    void init();
    bool isEmpty() {return functionNum() == 0;}
    int functionNum() { return static_cast<int>(functions_.size()); }
    bool isHelpAvailableForFunction(QString functionName);
    VerbFunctionHelpFunction* function(QString functionName);
    QList<VerbFunctionHelpFunction*> const functions() {return functions_;}
    VerbFunctionHelpFunction* itemAt(int i) const;
    QString url(QString webGroup, QString fnName);
    QString iconUrl(VerbFunctionHelpFunction*);
    QString scopePrefix() const;

private:
    void clear();

    LanguageHelper* language_;
    std::map<QString, VerbFunctionHelpFunction*> functionMap_;
    QList<VerbFunctionHelpFunction*> functions_;
    QMap<QString, QString> webIds_;
};

#endif
