# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Models specific to the Flickr plugin.
"""

import os

from elisa.core import common
from elisa.core.components.model import Model
from elisa.core.utils import defer, caching

from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.flickr import flickr_api
from elisa.plugins.flickr.utils import get_flickr_provider

class FlickrPhotoModel(ImageModel):

    """
    A Flickr photo model contains metadata associated to a photo.

    @ivar flickr_id:   the id of the photo
    @type flickr_id:   C{unicode}
    @ivar owner:       the id of the owner of the photo
    @type owner:       C{unicode}
    @ivar secret:      the shared secret
    @type secret:      C{unicode}
    @ivar farm:        the Flickr server farm on which the photo is stored
    @type farm:        C{int}
    @ivar server:      the Flickr server on which the photo is stored
    @type server:      C{int}
    @ivar title:       the title associated to the photo
    @type title:       C{unicode}
    @ivar description: a description of the photo
    @type description: C{unicode}
    @ivar ispublic:    whether the photo is visible by anyone
    @type ispublic:    C{bool}
    @ivar isfriend:    whether the photo is visible by the owner's friends
    @type isfriend:    C{bool}
    @ivar isfamily:    whether the photo is visible by the owner's family
    @type isfamily:    C{bool}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(FlickrPhotoModel, self).__init__()
        self.flickr_id = None
        self.owner = None
        self.secret = None
        self.farm = None
        self.server = None
        self.title = None
        self.description = None
        self.ispublic = None
        self.isfriend = None
        self.isfamily = None
        self.can_rotate = False
        # we want Flickr's photos to be branded as such in the slideshow
        self.logo = "elisa.plugins.flickr.logo"

    def get_cached_data_path(self, reference_num):
        # overriden from base ImageModel since it is necessary to make a request
        # to Flickr's API to retrieve the list of available references (URLs)
        if len(self.references) > 0:
            return self._download_reference(self.references, reference_num)
        else:
            dfr = self._get_available_references()
            dfr.addCallback(self._download_reference, reference_num)
            return dfr

    def _download_reference(self, references, reference_num):
        uri = references[reference_num]
        path = caching.get_cached_image_path(uri)

        if os.path.exists(path):
            # already cached
            return defer.succeed(path)

        model, dfr = common.application.resource_manager.get(uri)
        dfr.addCallback(lambda model: caching.cache_to_file(model.data, path))
        return dfr

    def _get_available_references(self):
        # Retrieve the list of available sizes (URLs) for an image

        # Check if we have this list in the cache first, will avoid a costly call
        # to the Flickr API
        cache_dir = caching.get_cache_path(u'flickr_cache')
        cache_file = os.path.join(cache_dir, self.flickr_id)
        if os.path.exists(cache_file):
            references = caching.deserialize(cache_file)
            self.references = references
            return defer.succeed(references)

        provider = get_flickr_provider()
        authenticated = provider.auth_token is not None
        request = flickr_api.generate_call_uri(method='flickr.photos.getSizes',
                                           arguments={'photo_id': self.flickr_id},
                                           authenticated=authenticated)

        def got_available_sizes(result_model):
            references = result_model.sizes.references
            self.references = references
            caching.serialize(self.references, cache_file)
            return references

        result_model, dfr = common.application.resource_manager.get(request)
        dfr.addCallback(got_available_sizes)
        return dfr


class FlickrTagModel(Model):

    """
    A Flickr tag model contains information associated to a tag.

    @ivar label: the label of the tag
    @type label: C{unicode}
    @ivar score: a numerical score associated to the tag
    @type score: C{int}
    @ivar private: is the tag private to the user currently using the plugin?
    @type private: C{bool}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(FlickrTagModel, self).__init__()
        self.label = None
        self.score = None
        self.private = False

class FlickrPhotoSetModel(Model):
    """
    A Flickr photoset model contains information associated to a photoset.

    @ivar flickr_id:   the id of the photoset
    @type flickr_id:   C{int}
    @ivar secret:      the shared secret
    @type secret:      C{unicode}
    @ivar farm:        the Flickr server farm on which the photoset is stored
    @type farm:        C{int}
    @ivar server:      the Flickr server on which the photoset is stored
    @type server:      C{int}
    @ivar title:       the title associated to the photoset
    @type title:       C{unicode}
    @ivar description: a description of the photoset
    @type description: C{unicode}
    @ivar photosnb:    number of photos in the set
    @type photosnb:    C{int}
    @ivar primary:     whether the set is primary(?) or not
    @type primary:     C{int}
    """

    def __init__(self):
        super(FlickrPhotoSetModel, self).__init__()
        self.flickr_id = None
        self.primary = None
        self.secret = None
        self.server = None
        self.photosnb = None
        self.farm = None
        self.title = None
        self.description = None

class FlickrContactModel(Model):
    """
    A Flickr contact model contains information associated to a user contact.

    @ivar nsid:       the id of the contact
    @type nsid:       C{unicode}
    @ivar username:   username of the contact
    @type username:   C{unicode}
    @ivar iconserver: icon server of the contact
    @type iconserver: C{int}
    @ivar realname:   real name of the user
    @type realname:   C{unicode}
    @ivar is_friend:  is the contact a friend?
    @type is_friend:  C{int}
    @ivar is_family:  is the contact a family member?
    @type is_family:  C{int}
    @ivar is_ignored: is the contact being ignored?
    @type is_ignored: C{int}
    """

    def __init__(self):
        super(FlickrContactModel, self).__init__()
        self.nsid = None
        self.username = None
        self.iconserver = None
        self.realname = None
        self.is_friend = None
        self.is_family = None
        self.is_ignored = None

class FlickrResponseModel(Model):

    """
    A generic model containing a representation of the XML response returned by
    the Flickr API.
    """

    pass


class LoginModel(Model):
    def __init__(self):
        super(LoginModel, self).__init__()
        self.success = False
        self.auth_token = None
        self.username = None
        self.password = None
