//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Core
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QDebug>
#include <QFile>
#include <QTextStream>
#include <QStringList>
#include <QVariant>
#include <QFileInfo>
#include "diskfstab.h"
#include "diskdevice.h"

DiskFstab::DiskFstab() {
	QFile fstabFile("/etc/fstab");
	fstabFile.open(QIODevice::ReadOnly);
	QTextStream fstabStream(&fstabFile);
	currentFstabText = fstabStream.readAll();
	fstabStream.seek(0);
	while (!fstabStream.atEnd()) {
		QString line = fstabStream.readLine();
		if (line.contains("#"))
			continue;
		// If line starts with it - it's samba share
		if (line.startsWith("//")) {
			unknownFstabText += line + '\n';
			continue;
		}
		line.replace("\t"," ");
		QStringList list = line.split(" ");
		list.removeAll("");
		list.removeAll(" ");
		if (list.count() == 0)
			continue;
		if (list.count() != 6) {
			unknownFstabText += line + '\n';
			continue;
		}
		DiskFstabItem *diskFstabItem = new DiskFstabItem;
		diskFstabItem->itemName = list[0];
		if (diskFstabItem->itemName.contains("/dev/disk/by")) {
			diskFstabItem->itemLink = diskFstabItem->itemName;
			diskFstabItem->itemName = QFileInfo(diskFstabItem->itemName).symLinkTarget();
		}
		diskFstabItem->itemMountPoint = list[1];
		diskFstabItem->itemFileSystem = list[2];
		diskFstabItem->itemOptions = list[3];
		diskFstabItem->itemDump = QVariant(list[4]).toBool();
		diskFstabItem->itemFsck = QVariant(list[5]).toInt();
		diskFstabItems.append(diskFstabItem);
	}
	qDebug() << diskFstabItems.count() << "records in /etc/fstab were detected.";
}

DiskFstab::~DiskFstab() {
	foreach (DiskFstabItem *diskFstabItem,diskFstabItems)
		delete diskFstabItem;
}

QString DiskFstab::fstabContent() {
	return currentFstabText;
}

const QString& DiskFstab::fstabContent() const {
	return currentFstabText;
}

DiskFstabItem *DiskFstab::fstabItem(DiskDevice *device) {
	foreach (DiskFstabItem *fstabItem,diskFstabItems) {
		if (fstabItem->itemName == device->blockName()
		|| QString::compare(fstabItem->itemName,"UUID=" + device->uuid(),Qt::CaseInsensitive) == 0
		|| QString::compare(fstabItem->itemName,"LABEL=" + device->label(),Qt::CaseInsensitive) == 0)
			return fstabItem;
	}
	return 0;
}

DiskFstabItem *DiskFstab::fstabItem(const QString& blockName) {
	foreach (DiskFstabItem *fstabItem,diskFstabItems)
		if (fstabItem->itemName == blockName)
			return fstabItem;
	return 0;
}

QList<DiskFstabItem *>& DiskFstab::fstabItems() {
	return diskFstabItems;
}

const QList<DiskFstabItem *>& DiskFstab::fstabItems() const {
	return diskFstabItems;
}
		
// Return QFile::FileError
int DiskFstab::setFstabContent(const QString &newFstabContent) {
	QFile fstabFile("/etc/fstab");
	fstabFile.open(QIODevice::WriteOnly);
	if (fstabFile.error() != 0)
		return fstabFile.error();
	QTextStream fstabStream(&fstabFile);
	fstabStream << unknownFstabText;
	fstabStream << newFstabContent;
	currentFstabText = newFstabContent;
	return 0;
}
