//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QToolButton>
#include <QtGui/QTextBrowser>
#include <QtGui/QIcon>
#include <QtGui/QApplication>
#include <QtCore/QDir>
#include <QtCore/QLocale>
#include <QtCore/QTimer>
#include <QtCore/QSettings>
#include "const.h"
#include "manual.h"

const int SPEED = 15;

BrowserWithWidgets::BrowserWithWidgets() {
	timer = new QTimer;
	timer->setInterval(5);
	rollToDown = false;
	
	setOpenExternalLinks(true);
	
	buttonsFrame = new QFrame;
	buttonsFrame->setFrameShape(QFrame::Box);
	buttonsFrame->setFrameShadow(QFrame::Raised);
	buttonsFrame->setStyleSheet("QFrame {border: 1px solid gray; border-radius: 4px; background-color: rgb(200, 200, 200, 180)}");
	
	buttonsFrameLayout = new QHBoxLayout;
	buttonsFrame->setLayout(buttonsFrameLayout);
	
	showLinksButton = new QToolButton;
	showLinksButton->setIcon(QIcon(ICONS_PATH"manual.png"));
	showLinksButton->setIconSize(QSize(22, 22));
	showLinksButton->setCursor(Qt::ArrowCursor);
	showLinksButton->setAutoRaise(true);
	
	backwardButton = new QToolButton;
	backwardButton->setIcon(QIcon(ICONS_PATH"backward.png"));
	backwardButton->setIconSize(QSize(22, 22));
	backwardButton->setCursor(Qt::ArrowCursor);
	backwardButton->setAutoRaise(true);
	backwardButton->setEnabled(false);
	
	forwardButton = new QToolButton;
	forwardButton->setIcon(QIcon(ICONS_PATH"forward.png"));
	forwardButton->setIconSize(QSize(22, 22));
	forwardButton->setCursor(Qt::ArrowCursor);
	forwardButton->setAutoRaise(true);
	forwardButton->setEnabled(false);
	
	QFrame *verticalFrame = new QFrame;
	verticalFrame->setFrameStyle(QFrame::VLine | QFrame::Sunken);
	
	buttonsFrameLayout->addWidget(showLinksButton);
	buttonsFrameLayout->addWidget(verticalFrame);
	buttonsFrameLayout->addWidget(backwardButton);
	buttonsFrameLayout->addWidget(forwardButton);
	buttonsFrameLayout->setContentsMargins(0,0,0,0);
	
	buttonsFrame->setFixedSize(buttonsFrameLayout->minimumSize());
	
	linksFrame = new QFrame;
	linksFrame->setFrameShape(QFrame::Box);
	linksFrame->setFrameShadow(QFrame::Raised);
	QColor color = QApplication::palette().color(QPalette::Window);
	QString r = QVariant(color.red()).toString(); 
	QString g = QVariant(color.green()).toString();
	QString b = QVariant(color.blue()).toString();
	linksFrame->setStyleSheet(QString("QFrame {border: 1px solid gray; border-radius: 4px; background-color: rgb(%1, %2, %3, 180)}").arg(r,g,b));
	linksFrame->setMaximumHeight(0);
	
	linksFrameLayout = new QVBoxLayout;
	linksFrame->setLayout(linksFrameLayout);
	linksFrameLayout->setContentsMargins(0,0,0,0);
	
	QHBoxLayout *topLayout = new QHBoxLayout;
	topLayout->addStretch();
	topLayout->addWidget(buttonsFrame);
	topLayout->setContentsMargins(0,0,0,0);
	
	QHBoxLayout *bottomLayout = new QHBoxLayout;
	bottomLayout->addStretch();
	bottomLayout->addWidget(linksFrame);
	bottomLayout->setContentsMargins(0,0,0,0);
	
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(topLayout);
	mainLayout->addLayout(bottomLayout);
	mainLayout->addStretch();
	mainLayout->setContentsMargins(0,6,18,0);
	setLayout(mainLayout);
	
	connect(showLinksButton,SIGNAL(clicked()),this,SLOT(showLinks()));
	connect(timer,SIGNAL(timeout()),this,SLOT(updateLinksSize()));
	connect(backwardButton,SIGNAL(clicked()),this,SLOT(backward()));
	connect(forwardButton,SIGNAL(clicked()),this,SLOT(forward()));
	connect(this,SIGNAL(backwardAvailable(bool)),backwardButton,SLOT(setEnabled(bool)));
	connect(this,SIGNAL(forwardAvailable(bool)),forwardButton,SLOT(setEnabled(bool)));
}

BrowserWithWidgets::~BrowserWithWidgets() {
	foreach (Link link,links)
		delete link.linkView;
	
	delete timer;
	
	delete showLinksButton;
	delete backwardButton;
	delete forwardButton;
	
	delete buttonsFrameLayout;
	delete buttonsFrame;
	
	delete linksFrameLayout;
	delete linksFrame;
}

void BrowserWithWidgets::addLink(const QString& linkTitle,const QString& sourcePath) {
	QToolButton *linkView = new QToolButton;
	linkView->setText(linkTitle);
	linkView->setAutoRaise(true);
	linkView->setMaximumHeight(20);
	linkView->setCheckable(true);
	linksFrameLayout->addWidget(linkView);
	connect(linkView,SIGNAL(clicked()),this,SLOT(changeSource()));
	
	links << Link(linkView,sourcePath);
	if (links.count() == 1)
		setSource(sourcePath);
}

void BrowserWithWidgets::changeSource() {
	foreach (Link link,links) {
		if (link.linkView->isChecked()) {
			link.linkView->setChecked(false);
			setSource(link.linkSource);
		}
	}
	//showLinks();
}

void BrowserWithWidgets::showLinks() {
	rollToDown = !rollToDown;
	timer->start();
}

void BrowserWithWidgets::updateLinksSize() {
	if (rollToDown) {
		if (linksFrame->height() + SPEED >= linksFrame->sizeHint().height())
			linksFrame->setMaximumHeight(linksFrame->sizeHint().height());
		else
			linksFrame->setMaximumHeight(linksFrame->height() + SPEED);
		if (linksFrame->height() != linksFrame->sizeHint().height())
			timer->start();
	} else {
		if (linksFrame->height() - SPEED <= 0)
			linksFrame->setMaximumHeight(0);
		else
			linksFrame->setMaximumHeight(linksFrame->height() - SPEED);
		if (linksFrame->height() != 0)
			timer->start();
	}
	linksFrame->resize(linksFrame->sizeHint());
}

Manual::Manual(QWidget *parent) : QDialog(parent) {
	
	language = QLocale().name();
	language = language.remove(language.indexOf("_"),language.length());
	
	browser = new BrowserWithWidgets;
	
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(browser);
	mainLayout->setContentsMargins(0,0,0,0);
	
	setLayout(mainLayout);
	setWindowIcon(QIcon(ICONS_PATH"manual.png"));
	setWindowTitle(tr("Documentation"));
	
	// If documentation with user language doesn't exist - show english
	if (!QDir(QString(DOC_PATH) + language).exists())
		language = "en";
	
	browser->addLink(tr("About the program"),QString(DOC_PATH) + language + "/Information/more_about_program.html");
	browser->addLink(tr("Changelogs"),QString(DOC_PATH) + language + "/Information/changelog.html");
	browser->addLink(tr("Bugs in the program"),QString(DOC_PATH) + language + "/Information/bugs.html");
	browser->addLink(tr("Usage MountManager"),QString(DOC_PATH) + language + "/Information/usage.html");
	browser->addLink(tr("File systems"),QString(DOC_PATH) + language + "/FileSystems/index.html");
	browser->addLink(tr("General information"),QString(DOC_PATH) + language + "/General/index.html");
	browser->addLink(tr("License"),QString(DOC_PATH) + language + "/Information/license.html");
	browser->addLink(tr("Web links"),QString(DOC_PATH) + language + "/Information/links.html");
	browser->addLink(tr("Authors"),QString(DOC_PATH) + language + "/Information/authors.html");
	browser->addLink(tr("Plugins API"),QString(DOC_PATH) + language + "/Information/plugins_api.html");
	
	setMinimumSize(550,325);
}

Manual::~Manual() {
	delete browser;
}

void Manual::saveSettings() {
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	settings.setValue("documentation_dailog/size",size());
}

void Manual::loadSettings() {
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	resize(settings.value("documentation_dialog/size",QSize(800,500)).toSize());
}
