//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QTime>
#include <QPushButton>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QLabel>
#include <QCheckBox>
#include <QMenu>
#include <QSettings>
#include <QIcon>
#include "const.h"
#include "../core/diskdevice.h"
#include "messagebar.h"

MessageBar::MessageBar(QWidget *parent) : QWidget(parent) {
	
	mountAction = 0;
	unmountAction = 0;
	ignoreAction = 0;
	dontShowThisMessageAction = 0;
	
	iconLabel = new QLabel;
	textLabel = new QLabel;
	textLabel->setWordWrap(true);

	actionsMenu = new QMenu;
	
	actionsButton = new QPushButton(tr("Actions"));
	actionsButton->setMenu(actionsMenu);
	actionsButton->setVisible(false);

	hideButton = new QPushButton(tr("Hide"));

	//Connections
	connect(hideButton,SIGNAL(clicked()),this,SLOT(hide()));
	connect(actionsMenu,SIGNAL(aboutToShow()),this,SLOT(menuShown()));

	QHBoxLayout *topLayout = new QHBoxLayout;
	topLayout->addWidget(iconLabel);
	topLayout->addWidget(textLabel,1);
	topLayout->addWidget(actionsButton);
	topLayout->addWidget(hideButton);

	mainLayout = new QVBoxLayout;
	mainLayout->addLayout(topLayout);
	
	setLayout(mainLayout);

	setVisible(true);
}

MessageBar::~MessageBar() {
	if (dontShowThisMessageAction != 0)
		if (dontShowThisMessageAction->isChecked()) {
			QSettings settings(ORGANIZATION,PROGRAM_NAME);
			settings.beginGroup("MessageBars");
			QStringList lastList = settings.value("Dont_show_these_messages_bars").toStringList();
			if (!lastList.contains(textLabel->text().remove(0,8).remove("<b>").remove("</b>")))
				settings.setValue("Dont_show_these_messages_bars",lastList << textLabel->text().remove(0,8).remove("<b>").remove("</b>"));
			settings.endGroup();
		}

	delete hideButton;
	delete actionsButton;
	delete iconLabel;
	delete textLabel;
	delete mainLayout;
	
	if (mountAction != 0)
		delete mountAction;
	if (unmountAction != 0)
		delete unmountAction;
	if (ignoreAction != 0)
		delete ignoreAction;
	if (dontShowThisMessageAction != 0)
		delete dontShowThisMessageAction;
	delete actionsMenu;
}

void MessageBar::setDevice(DiskDevice *device) {
	deviceVar = device;
}

DiskDevice *MessageBar::device() const {
	return deviceVar;
}

void MessageBar::addAction(QAction *action) {
	actionsMenu->addAction(action);
	actionsButton->setVisible(true);
}

void MessageBar::addAction(int actionType) {
	switch (actionType) {
		case Mount:
			mountAction = new QAction(actionsMenu);
			if (deviceVar->isMounted())
				mountAction->setEnabled(false);
			mountAction->setText(tr("Mount"));
			mountAction->setIcon(QIcon(ICONS_PATH"mount.png"));
			connect(mountAction,SIGNAL(triggered()),this,SLOT(mountSlot()));
			actionsMenu->addAction(mountAction);
			break;
		case UnMount:
			unmountAction = new QAction(actionsMenu);
			if (!deviceVar->isMounted())
				unmountAction->setEnabled(false);
			unmountAction->setText(tr("Unmount"));
			unmountAction->setIcon(QIcon(ICONS_PATH"umount.png"));
			actionsMenu->addAction(unmountAction);
			connect(unmountAction,SIGNAL(triggered()),this,SLOT(unmountSlot()));
			break;
		case Ignore:
			ignoreAction = new QAction(actionsMenu);
			ignoreAction->setText(tr("Ignore"));
			actionsMenu->addAction(ignoreAction);
			connect(ignoreAction,SIGNAL(triggered()),this,SLOT(hide()));
			break;
		case DontShowThisMessageAgain:
			dontShowThisMessageAction = new QAction(actionsMenu);
			dontShowThisMessageAction->setText(tr("Don't show this message again"));
			dontShowThisMessageAction->setCheckable(true);
			actionsMenu->addAction(dontShowThisMessageAction);
			break;
	}
	actionsButton->setVisible(true);
}

void MessageBar::mountSlot() {
	emit(mountSignal(deviceVar));
}

void MessageBar::unmountSlot() {
	emit(unmountSignal(deviceVar));
}

void MessageBar::menuShown() {
	if (mountAction != 0)
		mountAction->setEnabled(!deviceVar->isMounted());
	if (unmountAction != 0)
		unmountAction->setEnabled(deviceVar->isMounted());
}

// Overloaded functions
void MessageBar::setMessage(const QIcon& icon,const QString& text) {
	actionsMenu->clear();
	textLabel->clear();
	iconLabel->clear();
	if (!icon.isNull())
		iconLabel->setPixmap(icon.pixmap(height(),height()));
	
	// Define time
	QString time = QTime::currentTime().toString("hh:mm");
	textLabel->setText("[" + time + "]" + " <b>" + text + "</b>");

	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	settings.beginGroup("MessageBars");
	foreach (QString label,settings.value("Dont_show_these_messages_bars").toStringList())
		if (label == text) {
			hide();
			break;
		}
	settings.endGroup();
}

void MessageBar::setMessage(int messageType,const QString& text) {
	switch (messageType) {
		case Information:
			setMessage(QIcon(ICONS_PATH"info.png"),text);
			break;
		case Warning:
			setMessage(QIcon(ICONS_PATH"warning.png"),text);
			break;
		case Error:
			setMessage(QIcon(ICONS_PATH"error.png"),text);
			break;
	}
}

void MessageBar::showEvent(QShowEvent *) {
	emit (activated());
}

void MessageBar::hideEvent(QHideEvent *) {
	emit (activated());
}