/***************************************************************************
 *   Copyright (C) 2000-20011 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "logging.h"

#include <QDebug>
#include <QTextStream>
#include <QFileInfo>
#include <QStringList>
#include <QDir>
#include "ui_loggingform.h"


QString levelStr[NUMDEBUGLEVELS]=
{
"GALLERY ",
"PARAMS  ",
"FFT     ",
"WAVIO   ",
"SYNTHES ",
"DISPATC ",
"SOUNDIO ",
"RXFUNC  ",
"TXFUNC  ",
"SYNCPRC1",
"SYNCPRC2",
"SLANTAD ",
"MODES   ",
"FTP     ",
"IMAGE   ",
"SCOPE   ",
"CAM     ",
"RX MAIN ",
"TX MAIN ",
"EDITOR  ",
"RIGCTRL ",
"PERFORM "
};




/*! class logFile
  \brief utility class to enable logging facilities

  Create an instance of this class giving the basename of the logfile.
  By default the log is disabled. call setEnabled(TRUE) to enable logging


*/
logFile logfile;


logFile::logFile()
{
#ifndef QT_NO_DEBUG
  lf=new QFile;
  auxFile=new QFile;
#endif
  logCount=0;
  savedLogEntry="";
  savedPosMask=0;
  mask.set(); //all masks set:this will enable all logfile messages
}

/*!
  creates logfile with name=logname, and opens it for writing
*/

bool logFile::open(QString logname)
{
#ifndef QT_NO_DEBUG
  lf->setFileName(QDir::homePath()+"/"+logname);
  auxFile->setFileName(QDir::homePath()+"/aux_"+logname);
  return reopen();
#endif
}


/*!
  closes the logfile
*/

logFile::~logFile()
{
 close();
}

void logFile::close()
{
#ifndef QT_NO_DEBUG
 qDebug() << "closing logfile";
 add("End of logfile",DBALL);
 add("....,",DBALL);
 delete ts;
 delete auxTs;
 lf->close();
 auxFile->close();
#endif
}

void logFile::reset()
{
  close();
  reopen();
}

bool logFile::reopen()
{
#ifndef QT_NO_DEBUG
  setEnabled(FALSE);
  QFileInfo finf(*lf);
  QFileInfo finfaux(*auxFile);
  qDebug() << "opening logfile--: " << finf.absoluteFilePath();
  if(!lf->open(QIODevice::WriteOnly))
    {
      qDebug() << "logfile creation failed";
      return FALSE;
    }
  qDebug() << "opening logfile: " << finfaux.absoluteFilePath();
  if(!auxFile->open(QIODevice::WriteOnly))
    {
      qDebug() << "auxillary file creation failed";
      lf->close();
      return FALSE;
    }
  setEnabled(TRUE);
  ts= new QTextStream( lf );
  auxTs= new QTextStream( auxFile);
  savedLogEntry="";
  logCount=0;
  timer.start();
  *ts<< "Time \tElapsed  \t  Level  \t  Count\t          Info\n";
  ts->flush();
#endif
  return TRUE;
}

/*!
  \brief Writes to the logfile

  The output is flushed after every access.Identical messages are only logged once. The count indicates the number of duplicate messages.
*/


void logFile::add(QString t,short unsigned int posMask)
{
#ifndef QT_NO_DEBUG
  if(!(posMask==DBALL)) // always show messages with DBALL
    {
      if (!mask.test(posMask)) return;
    }
  if (!enabled) return;
  mutex.lock();
  if(logCount==0)
    {
      logCount=1;
      savedLogEntry=t;
      timer.restart();
      tmp=QString("%1 ").arg(timer.elapsed(),5);
      tmp2=timer.currentTime().toString("HH:mm:ss:zzz ");
      savedPosMask=posMask;
    }
   else
    {
      if (t==savedLogEntry) logCount++;
      else
        {
          if(savedPosMask==DBALL)
            {
              *ts << tmp2<< "\t" << tmp << "\tALL     \t" << logCount << "\t" << savedLogEntry <<"\n";
            }
          else
           {
             *ts << tmp2<< "\t" << tmp << "\t" << levelStr[savedPosMask] <<"\t" << logCount << "\t" << savedLogEntry <<"\n";
           }
          tmp=QString("%1 ").arg(timer.elapsed(),5);
          tmp2 = timer.currentTime().toString("HH:mm:ss:zzz ");
          timer.restart();;
          savedLogEntry=t;
          savedPosMask=posMask;
          logCount=1;
        }
      }
    ts->flush();
    lf->flush();
    mutex.unlock();
#endif
}

void logFile::add(const char *fileName,const char *functionName, int line, QString t,short unsigned int posMask)
{
  QString s;
  s=QString(fileName)+":"+QString(functionName)+":"+QString::number(line)+" "+ t;
  add(s,posMask);
}

void logFile::addToAux(QString t)
{
  #ifndef QT_NO_DEBUG
  if (!enabled) return;
  mutex.lock();
  *auxTs << t << "\n";
  auxTs->flush();
  auxFile->flush();
  mutex.unlock();
#endif
}

/*!
  if enable=TRUE logging wil be performed
  \return previous logging state (TRUE if logging was enabled)
*/

bool logFile::setEnabled(bool enable)
{
  bool t=enabled;
  enabled=enable;
  return t;
}

void logFile::setLogMask(std::bitset<NUMDEBUGLEVELS> logMask)
{
  mask=logMask;
}

void logFile::maskSelect(QWidget *wPtr)
{
  QDialog lf(wPtr);
  Ui::loggingForm ui;
  ui.setupUi(&lf);

  ui.log0CheckBox->setText(levelStr[0]);
  ui.log1CheckBox->setText(levelStr[1]);
  ui.log2CheckBox->setText(levelStr[2]);
  ui.log3CheckBox->setText(levelStr[3]);
  ui.log4CheckBox->setText(levelStr[4]);
  ui.log5CheckBox->setText(levelStr[5]);
  ui.log6CheckBox->setText(levelStr[6]);
  ui.log7CheckBox->setText(levelStr[7]);
  ui.log8CheckBox->setText(levelStr[8]);
  ui.log9CheckBox->setText(levelStr[9]);
  ui.log10CheckBox->setText(levelStr[10]);
  ui.log11CheckBox->setText(levelStr[11]);
  ui.log12CheckBox->setText(levelStr[12]);
  ui.log13CheckBox->setText(levelStr[13]);
  ui.log14CheckBox->setText(levelStr[14]);
  ui.log15CheckBox->setText(levelStr[15]);

  ui.log16CheckBox->setText(levelStr[16]);
  ui.log17CheckBox->setText(levelStr[17]);
  ui.log18CheckBox->setText(levelStr[18]);
  ui.log19CheckBox->setText(levelStr[19]);
  ui.log20CheckBox->setText(levelStr[20]);
  ui.log21CheckBox->setText(levelStr[21]);


  ui.log0CheckBox->setChecked(mask.test(0));
  ui.log1CheckBox->setChecked(mask.test(1));
  ui.log2CheckBox->setChecked(mask.test(2));
  ui.log3CheckBox->setChecked(mask.test(3));
  ui.log4CheckBox->setChecked(mask.test(4));
  ui.log5CheckBox->setChecked(mask.test(5));
  ui.log6CheckBox->setChecked(mask.test(6));
  ui.log7CheckBox->setChecked(mask.test(7));
  ui.log8CheckBox->setChecked(mask.test(8));
  ui.log9CheckBox->setChecked(mask.test(9));
  ui.log10CheckBox->setChecked(mask.test(10));
  ui.log11CheckBox->setChecked(mask.test(11));
  ui.log12CheckBox->setChecked(mask.test(12));
  ui.log13CheckBox->setChecked(mask.test(13));
  ui.log14CheckBox->setChecked(mask.test(14));
  ui.log15CheckBox->setChecked(mask.test(15));
  ui.log16CheckBox->setChecked(mask.test(16));
  ui.log17CheckBox->setChecked(mask.test(17));
  ui.log18CheckBox->setChecked(mask.test(18));
  ui.log19CheckBox->setChecked(mask.test(19));
  ui.log20CheckBox->setChecked(mask.test(20));
  ui.log21CheckBox->setChecked(mask.test(21));

  if(lf.exec()==QDialog::Accepted)
  {
    mask.set(0,ui.log0CheckBox->isChecked());
    mask.set(1,ui.log1CheckBox->isChecked());
    mask.set(2,ui.log2CheckBox->isChecked());
    mask.set(3,ui.log3CheckBox->isChecked());
    mask.set(4,ui.log4CheckBox->isChecked());
    mask.set(5,ui.log5CheckBox->isChecked());
    mask.set(6,ui.log6CheckBox->isChecked());
    mask.set(7,ui.log7CheckBox->isChecked());
    mask.set(8,ui.log8CheckBox->isChecked());
    mask.set(9,ui.log9CheckBox->isChecked());
    mask.set(10,ui.log10CheckBox->isChecked());
    mask.set(11,ui.log11CheckBox->isChecked());
    mask.set(12,ui.log12CheckBox->isChecked());
    mask.set(13,ui.log13CheckBox->isChecked());
    mask.set(14,ui.log14CheckBox->isChecked());
    mask.set(15,ui.log15CheckBox->isChecked());
    mask.set(16,ui.log16CheckBox->isChecked());
    mask.set(17,ui.log17CheckBox->isChecked());
    mask.set(18,ui.log18CheckBox->isChecked());
    mask.set(19,ui.log19CheckBox->isChecked());
    mask.set(20,ui.log20CheckBox->isChecked());
    mask.set(21,ui.log21CheckBox->isChecked());
  }
}

void logFile::readSettings(QSettings &qSettings)
{
  qSettings.beginGroup ("logging");
  mask=qSettings.value("mask",1).toUInt();
  qSettings.endGroup();
}
void logFile::writeSettings(QSettings &qSettings)
{
  qSettings.beginGroup ("logging");
  qSettings.setValue ( "mask", (uint)mask.to_ulong());
  qSettings.endGroup();

}


