// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "third_party/blink/renderer/modules/payments/payment_request.h"

#include <ostream>  // NOLINT
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/blink/renderer/bindings/core/v8/v8_binding_for_testing.h"
#include "third_party/blink/renderer/core/dom/document.h"
#include "third_party/blink/renderer/modules/payments/payment_details_init.h"
#include "third_party/blink/renderer/modules/payments/payment_options.h"
#include "third_party/blink/renderer/modules/payments/payment_test_helper.h"
#include "third_party/blink/renderer/platform/bindings/exception_state.h"
#include "third_party/blink/renderer/platform/bindings/script_state.h"
#include "third_party/blink/renderer/platform/weborigin/security_origin.h"

namespace blink {
namespace {

class DetailsTestCase {
 public:
  DetailsTestCase(
      PaymentTestDetailToChange detail,
      PaymentTestDataToChange data,
      PaymentTestModificationType mod_type,
      const char* value_to_use,
      bool expect_exception = false,
      ESErrorType expected_exception_code = static_cast<ESErrorType>(0))
      : detail_(detail),
        data_(data),
        mod_type_(mod_type),
        value_to_use_(value_to_use),
        expect_exception_(expect_exception),
        expected_exception_code_(expected_exception_code) {}

  ~DetailsTestCase() = default;

  PaymentDetailsInit BuildDetails() const {
    return BuildPaymentDetailsInitForTest(detail_, data_, mod_type_,
                                          value_to_use_);
  }

  bool ExpectException() const { return expect_exception_; }

  ESErrorType GetExpectedExceptionCode() const {
    return expected_exception_code_;
  }

 private:
  friend std::ostream& operator<<(std::ostream&, DetailsTestCase);
  PaymentTestDetailToChange detail_;
  PaymentTestDataToChange data_;
  PaymentTestModificationType mod_type_;
  const char* value_to_use_;
  bool expect_exception_;
  ESErrorType expected_exception_code_;
};

std::ostream& operator<<(std::ostream& out, DetailsTestCase test_case) {
  if (test_case.expect_exception_)
    out << "Expecting an exception when ";
  else
    out << "Not expecting an exception when ";

  switch (test_case.detail_) {
    case kPaymentTestDetailTotal:
      out << "total ";
      break;
    case kPaymentTestDetailItem:
      out << "displayItem ";
      break;
    case kPaymentTestDetailShippingOption:
      out << "shippingOption ";
      break;
    case kPaymentTestDetailModifierTotal:
      out << "modifiers.total ";
      break;
    case kPaymentTestDetailModifierItem:
      out << "modifiers.displayItem ";
      break;
    case kPaymentTestDetailError:
      out << "error ";
      break;
    case kPaymentTestDetailNone:
      NOTREACHED();
      break;
  }

  switch (test_case.data_) {
    case kPaymentTestDataId:
      out << "id ";
      break;
    case kPaymentTestDataLabel:
      out << "label ";
      break;
    case kPaymentTestDataAmount:
      out << "amount ";
      break;
    case kPaymentTestDataCurrencyCode:
      out << "currency ";
      break;
    case kPaymentTestDataValue:
      out << "value ";
      break;
    case kPaymentTestDataNone:
      NOTREACHED();
      break;
  }

  switch (test_case.mod_type_) {
    case kPaymentTestOverwriteValue:
      out << "is overwritten by \"" << test_case.value_to_use_ << "\"";
      break;
    case kPaymentTestRemoveKey:
      out << "is removed";
      break;
  }

  return out;
}

class PaymentRequestDetailsTest
    : public testing::TestWithParam<DetailsTestCase> {};

TEST_P(PaymentRequestDetailsTest, ValidatesDetails) {
  V8TestingScope scope;
  scope.GetDocument().SetSecurityOrigin(
      SecurityOrigin::Create(KURL("https://www.example.com/")));
  scope.GetDocument().SetSecureContextStateForTesting(
      SecureContextState::kSecure);
  PaymentOptions options;
  options.setRequestShipping(true);
  PaymentRequest::Create(
      scope.GetExecutionContext(), BuildPaymentMethodDataForTest(),
      GetParam().BuildDetails(), options, scope.GetExceptionState());

  EXPECT_EQ(GetParam().ExpectException(),
            scope.GetExceptionState().HadException());
  if (GetParam().ExpectException()) {
    EXPECT_EQ(GetParam().GetExpectedExceptionCode(),
              scope.GetExceptionState().CodeAs<ESErrorType>());
  }
}

INSTANTIATE_TEST_CASE_P(
    EmptyData,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataLabel,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataLabel,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataId,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataLabel,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataLabel,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataLabel,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    false)));

INSTANTIATE_TEST_CASE_P(
    ValidCurrencyCodeFormat,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataCurrencyCode,
                                    kPaymentTestOverwriteValue,
                                    "USD"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataCurrencyCode,
                                    kPaymentTestOverwriteValue,
                                    "USD"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataCurrencyCode,
                                    kPaymentTestOverwriteValue,
                                    "USD"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataCurrencyCode,
                                    kPaymentTestOverwriteValue,
                                    "USD"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataCurrencyCode,
                                    kPaymentTestOverwriteValue,
                                    "USD")));

INSTANTIATE_TEST_CASE_P(
    ValidValueFormat,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "0"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.99"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "0"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-0"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.99"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3.00"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "0"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-0"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.99"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3.00"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-012345678901234567890123456789")));

INSTANTIATE_TEST_CASE_P(
    ValidValueFormatForModifier,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "0"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.99"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "0"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-0"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.99"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3.00"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "012345678901234567890123456789"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789.0123456789"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789012345678.9"),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-012345678901234567890123456789")));

INSTANTIATE_TEST_CASE_P(
    InvalidValueFormat,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3.00",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "notdigits",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "ALSONOTDIGITS",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    ".99",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1.0.0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1/3",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789.0123456789",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789012345678.9",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-012345678901234567890123456789",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "notdigits",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "ALSONOTDIGITS",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    ".99",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1.0.0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1/3",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "notdigits",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "ALSONOTDIGITS",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    ".99",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1.0.0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailShippingOption,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1/3",
                                    true,
                                    ESErrorType::kTypeError)));

INSTANTIATE_TEST_CASE_P(
    InvalidValueFormatForModifier,
    PaymentRequestDetailsTest,
    testing::Values(DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-3.00",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "notdigits",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "ALSONOTDIGITS",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    ".99",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1.0.0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1/3",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789.0123456789",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-01234567890123456789012345678.9",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierTotal,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-012345678901234567890123456789",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "notdigits",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "ALSONOTDIGITS",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    ".99",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "-10.",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "10-",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1-0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1.0.0",
                                    true,
                                    ESErrorType::kTypeError),
                    DetailsTestCase(kPaymentTestDetailModifierItem,
                                    kPaymentTestDataValue,
                                    kPaymentTestOverwriteValue,
                                    "1/3",
                                    true,
                                    ESErrorType::kTypeError)));

}  // namespace
}  // namespace blink
