/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#include "XMLUtils.h"
#include "RWLock.h"
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMWriter.hpp>
#include <xercesc/dom/DOMImplementationRegistry.hpp>
#include <xercesc/dom/DOMImplementationLS.hpp>
#include <xercesc/util/OutOfMemoryException.hpp>
#include <xercesc/framework/MemBufInputSource.hpp>
#include <xercesc/dom/DOMImplementation.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/util/XMLUni.hpp>
#include <xercesc/framework/MemBufFormatTarget.hpp>

XERCES_CPP_NAMESPACE_USE

namespace CimXMLUtilsNamespace
{
  static bool   _CimXMLUtilsInitialized = false;
  static Semaphore _initializationLock;
  static Semaphore _xmlStringLock;
	// local utility function
	string CIMXMLCreateErrMsg(const string& msg,
		const char* what = NULL)
	{
		string ret = msg;
		if(what)
		{
			ret.append(": \n").append(what);
		}
		return ret;
	}

	// the following utility class was copied from xerces "CreateDomDocument" sample code

	// ---------------------------------------------------------------------------
	//  This is a simple class that lets us do easy (though not terribly efficient)
	//  trancoding of char* data to XMLCh data.
	// ---------------------------------------------------------------------------
	class XStr
	{
	public :
		// -----------------------------------------------------------------------
		//  Constructors and Destructor
		// -----------------------------------------------------------------------
		XStr(const char* const toTranscode)
		{
		  // Call the private transcoding method
		  _xmlStringLock.acquire();
		  fUnicodeForm = XMLString::transcode(toTranscode);
		  _xmlStringLock.release();
		}


		~XStr()
		{
		  _xmlStringLock.acquire();
		  XMLString::release(&fUnicodeForm);
		  _xmlStringLock.release();
		}


		// -----------------------------------------------------------------------
		//  Getter methods
		// -----------------------------------------------------------------------
		const XMLCh* unicodeForm() const
		{
			return fUnicodeForm;
		}

	private :
		// -----------------------------------------------------------------------
		//  Private data members
		//
		//  fUnicodeForm
		//      This is the Unicode XMLCh format of the string.
		// -----------------------------------------------------------------------
		XMLCh*   fUnicodeForm;

	};
#define X(str) XStr(str).unicodeForm()

	// ---------------------------------------------------------------------------
	//  This is a simple class that lets us do easy (though not terribly efficient)
	//  trancoding of to XMLCh data to char* data.
	// ---------------------------------------------------------------------------
	class YStr
	{
	public :
		// -----------------------------------------------------------------------
		//  Constructors and Destructor
		// -----------------------------------------------------------------------
		YStr(const XMLCh* const toTranscode)
		{
			// Call the private transcoding method
		  _xmlStringLock.acquire();
		  fMultibyteForm = XMLString::transcode(toTranscode);
		  _xmlStringLock.release();
		}

		~YStr()
		{
		  _xmlStringLock.acquire();
		  XMLString::release(&fMultibyteForm);
		  _xmlStringLock.release();
		}


		// -----------------------------------------------------------------------
		//  Getter methods
		// -----------------------------------------------------------------------
		const char* multibyteForm() const
		{
			return fMultibyteForm;
		}

	private :
		// -----------------------------------------------------------------------
		//  Private data members
		//
		//  fMultibyteForm
		//      This is the multibyteForm  format of the string.
		// -----------------------------------------------------------------------
		char*   fMultibyteForm;
	};
#define Y(str) YStr(str).multibyteForm()

	class XMLElementImpl
	{
		friend class XMLDocImpl;
	private:
		DOMElement* domElement;
		xercesc_2_7::DOMDocument* domDoc;

	public:
		XMLElementImpl(DOMElement* element)
		{

			if (element == NULL)
			{
			    throw CIMXMLException("Failed to create XMLElement. Malformed XML\n");
			}

			domElement = element;
			try
			{
				domDoc = domDoc = domElement->getOwnerDocument();
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement", Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement",Y(e.getMessage())).c_str());
			}
		}

		XMLElementImpl(const XMLElementImpl& element)
		{
			domElement = element.domElement;
			domDoc = element.domDoc;
		}

		~XMLElementImpl()
		{
			domElement = NULL;
			domDoc = NULL;
		}

		XMLElementImpl&  operator=(const XMLElementImpl &other )
		{
			domElement = other.domElement;
			domDoc = other.domDoc;
			return *this;
		}


		XMLElementImpl* CreateChildNode(const string& nodeName,
			const string& ns,
			const string& prefix,
			const string* text = NULL)
		{
			try
			{
				string qn = !prefix.empty() ?
					prefix + ":" + nodeName : nodeName;
				DOMElement* elem = domDoc->createElementNS(
					!ns.empty() ? X(ns.c_str()) : NULL,
					X(qn.c_str()));
				domElement->appendChild(elem);
				if(text)
				{
					DOMText* val = domDoc->createTextNode(X(text->c_str()));
					elem->appendChild(val);
				}
				return new XMLElementImpl(elem);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement",Y(e.getMessage())).c_str());
			}
		}

		void CreateLeafNode(const string& nodeName,
			const string& ns,
			const string& prefix,
			const string& nodeValue)
		{
			try
			{
				string qn = !prefix.empty() ?
					prefix + ":" + nodeName : nodeName;
				DOMElement* elem = domDoc->createElementNS(
					!ns.empty() ? X(ns.c_str()) : NULL,
					X(qn.c_str()));
				domElement->appendChild(elem);
				DOMText* val = domDoc->createTextNode(X(nodeValue.c_str()));
				elem->appendChild(val);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create XMLElement",Y(e.getMessage())).c_str());
			}
		}

		void AddText(const string& nodeValue)
		{
			try
			{
				DOMText* val = domDoc->createTextNode(X(nodeValue.c_str()));
				domElement->appendChild(val);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add text node",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add text node",Y(e.getMessage())).c_str());
			}
		}

		bool HasNextSibling() const
		{
			try
			{
				DOMNode* tmp = domElement->getNextSibling();
				while(tmp != NULL && tmp->getNodeType() != DOMNode::ELEMENT_NODE)
				{
					tmp = tmp->getNextSibling();
				}
				return (tmp != NULL);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve sibling",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve sibling",Y(e.getMessage())).c_str());
			}
		}

		XMLElementImpl* GetNextSibling() const
		{
			try
			{
				DOMNode* tmp = domElement->getNextSibling();
				while(tmp != NULL && tmp->getNodeType() != DOMNode::ELEMENT_NODE)
				{
					tmp = tmp->getNextSibling();
				}
				return new XMLElementImpl((DOMElement*)tmp);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve sibling",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve sibling",Y(e.getMessage())).c_str());
			}
		}

		bool HasChildren() const
		{
			try
			{
				if(!domElement->hasChildNodes())
				{
					return false;
				}

				DOMNode* child = domElement->getFirstChild();
				while(child && child->getNodeType() != DOMNode::ELEMENT_NODE)
				{
					child = child->getNextSibling();
				}
				return (child != NULL);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve children",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve children",Y(e.getMessage())).c_str());
			}
		}

		bool IsLeafNode() const
		{
			return !HasChildren();
		}

		XMLElementImpl* GetFirstChild() const
		{
			try
			{
				DOMNode* child = domElement->getFirstChild();
				while(child && child->getNodeType() != DOMNode::ELEMENT_NODE)
				{
					child = child->getNextSibling();
				}
				return new XMLElementImpl((DOMElement*)child);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve children",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve children",Y(e.getMessage())).c_str());
			}
		}

		string GetNodeName() const
		{
			try
			{
				return Y(domElement->getLocalName());
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve name",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve name",Y(e.getMessage())).c_str());
			}
		}

		string GetNSUri() const
		{
			try
			{
				return Y(domElement->getNamespaceURI());
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve value",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve value",Y(e.getMessage())).c_str());
			}
		}

		string GetTextValue() const
		{
			try
			{
				return Y(domElement->getTextContent());
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve value",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve value",Y(e.getMessage())).c_str());
			}
		}

		string ToString(bool incRoot = false) const
		{
			if(!incRoot && IsLeafNode())
			{
				return GetTextValue();
			}
			string xml = "";
			DOMWriter* writer = NULL;
			XMLFormatTarget* dest = NULL;
			try
			{
				DOMImplementation* impl =  DOMImplementationRegistry::getDOMImplementation(X("LS"));
				writer = ((DOMImplementationLS*)impl)->createDOMWriter();
				dest = new MemBufFormatTarget();
				writer->setEncoding(X("UTF-8"));
				if(incRoot)
				{
					writer->writeNode(dest, *domElement);
					xml = (char*)((MemBufFormatTarget*)dest)->getRawBuffer();
				}
				else
				{
					for(DOMNode* child = domElement->getFirstChild(); child != NULL; child = child->getNextSibling())
					{
						if(child->getNodeType() == DOMNode::ELEMENT_NODE)
						{
							writer->writeNode(dest, *child);
						}
					}
					xml += (char*)((MemBufFormatTarget*)dest)->getRawBuffer();
				}
			}
			catch (XMLException& e)
			{
				if(writer)
				{
					writer->release();
				}
				if(dest)
				{
					delete dest;
				}
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to serialize element",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				if(writer)
				{
					writer->release();
				}
				if(dest)
				{
					delete dest;
				}
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to serialize element",Y(e.getMessage())).c_str());
			}

			if(writer)
			{
				writer->release();
			}
			if(dest)
			{
				delete dest;
			}
			return xml;
		}

		void GetAttributes(map<string, string>& attribs) const
		{
			try
			{
				if(domElement->hasAttributes())
				{
					DOMNamedNodeMap* NodeMap = domElement->getAttributes();
					for (XMLSize_t i=0; i < NodeMap->getLength(); ++i)
					{
						attribs[Y(NodeMap->item(i)->getNodeName())] =
							Y(NodeMap->item(i)->getNodeValue());
					}
				}
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve attributes",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve attributes",Y(e.getMessage())).c_str());
			}
		}

		string GetAttribValue(const string& name) const
		{
			try
			{
				string ret = "";
				if(domElement->hasAttributes())
				{
					DOMNamedNodeMap* NodeMap = domElement->getAttributes();
					for (XMLSize_t i=0; i < NodeMap->getLength(); ++i)
					{
						if(name.compare(Y(NodeMap->item(i)->getNodeName())) == 0)
						{
							ret = Y(NodeMap->item(i)->getNodeValue());
							break;
						}
					}
				}
				return ret;
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve attributes",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve attributes",Y(e.getMessage())).c_str());
			}
		}

		void AddAttribValue(const string& name, const string& value)
		{
			try
			{
				domElement->setAttribute(X(name.c_str()), X(value.c_str()));
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add attribute",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add attribute",Y(e.getMessage())).c_str());
			}
		}

		void AddNSDefinition(const string& ns, const string* prefix = NULL)
		{
			string qn = "xmlns";
			if(prefix && !prefix->empty())
			{
				qn.append(":").append(*prefix);
			}
			try
			{
				DOMAttr* att = domDoc->createAttributeNS(X("http://www.w3.org/2000/xmlns/"), X(qn.c_str()));
				att->setNodeValue(X(ns.c_str()));
				domElement->setAttributeNode(att);
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add attribute",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to add attribute",Y(e.getMessage())).c_str());
			}
		}
	};

	class XMLDocImpl
	{
	private:
		DOMDocument* doc;
		XercesDOMParser *parser;
		bool parsed;
	public:
		XMLDocImpl(const string& xml):doc(NULL), parser(NULL)
		{
			try
			{
				parsed = true;
				MemBufInputSource source((const XMLByte*) xml.c_str(),
					xml.length(),"ID");
				parser = new XercesDOMParser;
				parser->setDoNamespaces(true);
				parser->parse(source);

				doc = parser->getDocument();
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create xml document",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create xml document",Y(e.getMessage())).c_str());
			}
		}

		XMLDocImpl(const char* rootName,
			const char* uri,
			const char* prefix):doc(NULL), parser(NULL)
		{
			parsed = false;
			try
			{
				const char* ln = (rootName && strcmp(rootName, "") != 0) ? rootName : "root";
				string qn = prefix && strcmp(prefix, "") != 0 ? string(prefix).append(":").append(ln) : string(ln);
				DOMImplementation* impl =  DOMImplementationRegistry::getDOMImplementation(X("LS"));
				doc = impl->createDocument(uri && strcmp(uri, "") ? X(uri) : NULL,	X(qn.c_str()), 0);
			}

			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create xml document",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to create xml document",Y(e.getMessage())).c_str());
			}
		}

		~XMLDocImpl()
		{
			if(parsed && parser)
			{
				delete parser;
				parser = NULL;
			}
			else if(doc)
			{
				doc->release();
				doc = NULL;
			}
		}

		XMLElement GetRootNode()
		{
			try
			{
				DOMElement* rootElem = doc->getDocumentElement();
				return XMLElement(new XMLElementImpl(rootElem));
			}
			catch (XMLException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve root element",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to retrieve root element",Y(e.getMessage())).c_str());
			}
		}

		string ToString(bool incVersionStr = false)
		{
			string xml;
			DOMWriter* writer = NULL;
			XMLFormatTarget* dest = NULL;
			try
			{
				DOMImplementation* impl =  DOMImplementationRegistry::getDOMImplementation(X("LS"));
				writer = ((DOMImplementationLS*)impl)->createDOMWriter();
				dest = new MemBufFormatTarget();
				writer->setEncoding(X("UTF-8"));
				if(incVersionStr)
				{
					writer->writeNode(dest, *doc);
				}
				else
				{
					writer->writeNode(dest, *(doc->getDocumentElement()));
				}
				xml = (char*)((MemBufFormatTarget*)dest)->getRawBuffer();
			}
			catch (XMLException& e)
			{
				if(writer)
				{
					writer->release();
				}
				if(dest)
				{
					delete dest;
				}
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to serialize XML document",Y(e.getMessage())).c_str());
			}
			catch (DOMException& e)
			{
				if(writer)
				{
					writer->release();
				}
				if(dest)
				{
					delete dest;
				}
				throw CIMXMLException(CIMXMLCreateErrMsg("Failed to serialize XML document",Y(e.getMessage())).c_str());
			}
			if(writer)
			{
				writer->release();
			}
			if(dest)
			{
				delete dest;
			}
			return xml;
		}
	};
}; // CimXMLUtilsNamespace;

using namespace CimXMLUtilsNamespace;

void CimXMLUtilsNamespace::InitXMLLibrary()
{
  // threadsafety!!!
  if (_CimXMLUtilsInitialized)
    return;

  _initializationLock.acquire();
  if (!_CimXMLUtilsInitialized) {
    XMLPlatformUtils::Initialize();
    _CimXMLUtilsInitialized = true;
  }
  _initializationLock.release();
}

void CimXMLUtilsNamespace::TerminateXMLLibrary()
{
	XMLPlatformUtils::Terminate();
}

CimXMLUtilsNamespace::XMLElement::XMLElement(XMLElementImpl* i)
{
	impl = i;
}

CimXMLUtilsNamespace::XMLElement::XMLElement(const XMLElement& other)
{
	impl = new XMLElementImpl(*(other.impl));
}

CimXMLUtilsNamespace::XMLElement& XMLElement::operator=(const XMLElement &other)
{
	delete impl;
	impl = new XMLElementImpl(*(other.impl));
	return *this;
}

CimXMLUtilsNamespace::XMLElement::~XMLElement()
{
	delete impl;
	impl = NULL;
}

XMLElement CimXMLUtilsNamespace::XMLElement::CreateChildNode(const string& nodeName,
							     const string& ns,
							     const string& prefix,
							     const string* text)
{
	return XMLElement(impl->CreateChildNode(nodeName, ns, prefix, text));
}

void CimXMLUtilsNamespace::XMLElement::CreateLeafNode(const string& nodeName,
													   const string& ns,
													   const string& prefix,
													   const string& nodeValue)
{
	impl->CreateLeafNode(nodeName, ns, prefix, nodeValue);
}

void CimXMLUtilsNamespace::XMLElement::AddText(const string& nodeValue)
{
	impl->AddText(nodeValue);
}

bool CimXMLUtilsNamespace::XMLElement::HasNextSibling() const
{
	return impl->HasNextSibling();
}

XMLElement CimXMLUtilsNamespace::XMLElement::GetNextSibling() const
{
	return XMLElement(impl->GetNextSibling());
}

bool CimXMLUtilsNamespace::XMLElement::HasChildren() const
{
	return impl->HasChildren();
}

bool CimXMLUtilsNamespace::XMLElement::IsLeafNode() const
{
	return impl->IsLeafNode();
}

XMLElement CimXMLUtilsNamespace::XMLElement::GetFirstChild() const
{
	return XMLElement(impl->GetFirstChild());
}

string CimXMLUtilsNamespace::XMLElement::GetNodeName() const
{
	return impl->GetNodeName();
}

string CimXMLUtilsNamespace::XMLElement::GetTextValue() const
{
	return impl->GetTextValue();
}

string CimXMLUtilsNamespace::XMLElement::ToString(bool incRoot) const
{
	return impl->ToString(incRoot);
}

string CimXMLUtilsNamespace::XMLElement::GetNSUri() const
{
	return impl->GetNSUri();
}

void CimXMLUtilsNamespace::XMLElement::GetAttributes(map<string, string>& attribs) const
{
	impl->GetAttributes(attribs);
}

string CimXMLUtilsNamespace::XMLElement::GetAttribValue(const string& name) const
{
	return impl->GetAttribValue(name);
}

void CimXMLUtilsNamespace::XMLElement::AddAttribValue(const string& name, const string& value)
{
	impl->AddAttribValue(name, value);
}

void CimXMLUtilsNamespace::XMLElement::AddNSDefinition(const string& ns, const string* prefix)
{
	impl->AddNSDefinition(ns, prefix);
}

// XMLDoc function implementations
CimXMLUtilsNamespace::XMLDoc::XMLDoc(const string& xml)
{
	impl = new XMLDocImpl(xml);
}

CimXMLUtilsNamespace::XMLDoc::XMLDoc(const char* rootName,
									 const char* uri,
									 const char* prefix)
{
	impl = new XMLDocImpl(rootName, uri, prefix);
}

CimXMLUtilsNamespace::XMLDoc::~XMLDoc()
{
	delete impl;
}

XMLElement CimXMLUtilsNamespace::XMLDoc::GetRootNode()
{
	return impl->GetRootNode();
}

string CimXMLUtilsNamespace::XMLDoc::ToString(bool incVersionStr)
{
	return impl->ToString(incVersionStr);
}
