# Ленивая загрузка маршрутов

При использовании модульной системы, итоговая JavaScript-сборка может оказаться довольно большой, что негативно отразится на времени загрузки страницы. В некоторых случаях было бы эффективнее разделить компоненты каждого маршрута на отдельные фрагменты, и подгружать их только при переходе к соответствующему маршруту.

Совместное использование [асинхронной загрузки компонентов](https://ru.vuejs.org/v2/guide/components-dynamic-async.html#Асинхронные-компоненты) Vue и [возможностей по разделению кода](https://webpack.js.org/guides/code-splitting-async/) Webpack делает реализацию ленивой загрузки компонентов в зависимости от маршрутов тривиальной.

Во-первых, асинхронный компонент можно определить как функцию-фабрику, которая возвращает Promise (который должен разрешиться самим компонентом):

``` js
const Foo = () => Promise.resolve({ /* определение компонента */ })
```

Во-вторых, с Webpack 2 мы можем использовать синтаксис [динамических импортов](https://github.com/tc39/proposal-dynamic-import) для указания точек разделения кода:

``` js
import('./Foo.vue') // возвращает Promise
```

::: tip Примечание
Если вы используете Babel, то необходимо добавить плагин [syntax-dynamic-import](https://babeljs.io/docs/plugins/syntax-dynamic-import/), чтобы Babel смог корректно обработать синтаксис.
:::

Эти два пункта — всё необходимое, чтобы определить асинхронный компонент, который Webpack автоматически выделит в отдельный фрагмент:

``` js
const Foo = () => import('./Foo.vue')
```

В конфигурации маршрута ничего менять не нужно, можно использовать `Foo` как обычно:

``` js
const router = new VueRouter({
  routes: [
    { path: '/foo', component: Foo }
  ]
})
```

## Группировка компонентов в одном фрагменте

Иногда может понадобиться объединить в одном фрагменте все компоненты, расположенные по определённому маршруту. Для этого можно указывать [имена фрагментов Webpack](https://webpack.js.org/guides/code-splitting-async/#chunk-names), используя специальный синтаксис комментариев (в версиях Webpack > 2.4):

``` js
const Foo = () => import(/* webpackChunkName: "group-foo" */ './Foo.vue')
const Bar = () => import(/* webpackChunkName: "group-foo" */ './Bar.vue')
const Baz = () => import(/* webpackChunkName: "group-foo" */ './Baz.vue')
```

Webpack сгруппирует все одноимённые асинхронные модули в одном фрагменте.
