/******************************************************************************
 * program:     rasimg library 0.18                                           *
 * function:    Object set for handling vector images.                        *
 * modul:       vec_image.cc                                                  *
 * licency:     GPL or LGPL                                                   *
 * Copyright: (c) 2018-2019 Jaroslav Fojtik                                   *
 ******************************************************************************/
#include <stdlib.h>
#include <math.h>

#include "vecimage.h"
#include "../images.h"


static void FixPsAccent(string & s, const char *TexAccent, const char *PsAccent);


////////////////////////////////////

vecPen::vecPen(void)
{
  LineStyle = 1;
  PenWidth = -1;
}


void vecPen::prepExport(PS_State *PSS) const
{
  if(PSS)
  {
    if(memcmp(&LineColor,&PSS->LineColor,sizeof(RGB_Record)) != 0) 
    {
      memcpy(&PSS->LineColor,&LineColor,sizeof(RGB_Record));
      PSS->dirty |= PSS_LineColor;      
    }
    if(PSS->LineStyle != LineStyle)
    {
      PSS->LineStyle = LineStyle;
      PSS->dirty |= PSS_LineStyle;
    }
    if(PenWidth>=0 && PSS->LineWidth!=PenWidth)
    {      
      PSS->LineWidth = PenWidth;
      PSS->dirty |= PSS_LineWidth;
    }
  }
}


void vecPen::AttribFromPSS(const PS_State &PSS)
{
  memcpy(&LineColor, &PSS.LineColor, sizeof(PSS.LineColor));
  LineStyle = PSS.LineStyle;
  PenWidth = PSS.LineWidth;
}


vecBrush::vecBrush(const PS_State &PSS)
{
  BrushStyle = PSS.FillPattern;
  memcpy(&FillColor, &PSS.FillColor, sizeof(PSS.FillColor));
}


void vecBrush::prepExport(PS_State *PSS) const
{
  if(PSS)
  {
    if(memcmp(&FillColor,&PSS->FillColor,sizeof(RGB_Record)) != 0) 
    {
      memcpy(&PSS->FillColor,&FillColor,sizeof(RGB_Record));
      PSS->dirty |= PSS_FillColor;
    }
    if(BrushStyle != PSS->FillPattern)
        PSS->FillPattern = BrushStyle;
  }
}


void vecBrush::AttribFromPSS(const PS_State &PSS)
{
  memcpy(&FillColor, &PSS.FillColor, sizeof(FillColor));
  BrushStyle = PSS.FillPattern;  
}


vecFont::vecFont()
{
  ConvertCpg = NULL;
  Weight = 0;
  Italic = 0;
  memset(&TextColor,0,sizeof(TextColor));
}


vecFont::vecFont(const PS_State &PSS)
{
  ConvertCpg = PSS.ConvertCpg;
  FontSize = PSS.FontSize;
  FontSizeW = PSS.FontSizeW;
  Weight = PSS.FontWeight;
  Italic = PSS.FontItallic;
  memcpy(&TextColor, &PSS.TextColor, sizeof(TextColor));
}


void vecFont::prepExport(PS_State *PSS) const
{
  if(PSS)
  {
    if(ConvertCpg!=PSS->ConvertCpg)
    {
      PSS->ConvertCpg = ConvertCpg;
    }
    if(FontSize!=PSS->FontSize)
    {
      PSS->FontSize = FontSize;
    }
    if(Weight != PSS->FontWeight)
    {
      PSS->FontWeight = Weight;
    }
    if(Italic != PSS->FontItallic)
    {
      PSS->FontItallic = Italic;
    }
  }
}

////////////////////////////////////

vecTransform::vecTransform(void)
{
  CenterX = CenterY = TranslateX = TranslateY = 0;
  RotAngle = 0;
  ScaleX = ScaleY = 1;
}


bool vecTransform::ApplyTransform(string &s)
{
bool ContextSaved = false;

  if(fabs(RotAngle)>1e-10)
  {
    s.cat_printf("\ngsave\n%2.2f %2.2f translate\n%2.2f rotate\n%2.2f %2.2f translate",
		 CenterX,CenterY, RotAngle, -CenterX,-CenterY);
    ContextSaved = true;
  }

  if(fabs(TranslateX) || fabs(TranslateY)>1e-10)
  {
    if(!ContextSaved)
    {
      s += "\ngsave";
      ContextSaved = true;
    }
    s.cat_printf("\n%2.2f %2.2f translate", TranslateX, TranslateY);
  }

  if(fabs(1-ScaleX)>1e-10 || fabs(1-ScaleY)>1e-10)
  {
    if(!ContextSaved)
    {
      s += "\ngsave";
      ContextSaved = true;
    }
    s.cat_printf("\n%2.2f %2.2f scale", ScaleX, ScaleY);
  }

  return ContextSaved;
}


////////////////////////////////////

VectorEllipse::VectorEllipse(float iniBottomRect, float iniTopRect, float iniRightRect, float iniLeftRect)
{
 Tx = NULL;
 BottomRect = iniBottomRect;
 TopRect = iniTopRect;
 RightRect = iniRightRect;
 LeftRect = iniLeftRect; 
 bAngle = 0;
 eAngle = 360;
}


VectorEllipse::~VectorEllipse()
{
  if(Tx)
      {delete(Tx); Tx=NULL;}
}


string VectorEllipse::Export2EPS(PS_State *PSS) const
{
string str;

  if(PSS != NULL)
  {
    vecPen::prepExport(PSS);    
    vecBrush::prepExport(PSS);
    PS_Attr(str,PSS);
  }

  bool ContextSaved = false;
  if(Tx) 
    ContextSaved = Tx->ApplyTransform(str);

  str.cat_printf("\nnewpath"
                 "\n%2.2f %2.2f %2.2f %2.2f %d %d DrawEllipse",
			 (RightRect+LeftRect)/2, (TopRect+BottomRect)/2,
			 fabs(RightRect-LeftRect)/2, fabs(TopRect-BottomRect)/2,
			 bAngle, eAngle);

  FillObjectPS(LeftRect,RightRect,BottomRect,TopRect,str,PSS);

  if(ContextSaved) str += "\ngrestore";

return str;
}


void VectorEllipse::Transform(const AbstractTransformXY &Tx)
{
  Tx.ApplyTransform(LeftRect, BottomRect);
  Tx.ApplyTransform(RightRect, TopRect);

  if(TopRect < BottomRect)
  {
    const float f = TopRect;
    TopRect = BottomRect;
    BottomRect = f;
  }
  if(RightRect < LeftRect)
  {
    const float f = RightRect;
    RightRect = LeftRect;
    LeftRect = f;
  }
}



/////////////////////////////////////////////////////////////


VectorRectangle::VectorRectangle(float iniBottomRect, float iniTopRect, float iniRightRect, float iniLeftRect)
{
 Tx = NULL;
 BottomRect = iniBottomRect;
 TopRect = iniTopRect;
 RightRect = iniRightRect;
 LeftRect = iniLeftRect;
}


VectorRectangle::~VectorRectangle()
{
  if(Tx)
      {delete(Tx); Tx=NULL;}
}


string VectorRectangle::Export2EPS(PS_State *PSS) const
{
string str;

  if(PSS != NULL)
  {
    vecBrush::prepExport(PSS);    
    vecPen::prepExport(PSS);
    PS_Attr(str,PSS);
  }

  bool ContextSaved = false;
  if(Tx) 
    ContextSaved = Tx->ApplyTransform(str);
  
  str.cat_printf("\nnewpath\n%2.2f %2.2f moveto", LeftRect, BottomRect);
  str.cat_printf("\n%2.2f %2.2f lineto", LeftRect, TopRect);
  str.cat_printf("\n%2.2f %2.2f lineto", RightRect, TopRect);
  str.cat_printf("\n%2.2f %2.2f lineto", RightRect, BottomRect);
  str.cat_printf("\n%2.2f %2.2f lineto", LeftRect, BottomRect);

  FillObjectPS(LeftRect,RightRect,BottomRect,TopRect,str,PSS);

  if(ContextSaved) str += "\ngrestore";

return str;
}


void VectorRectangle::Transform(const AbstractTransformXY &Tx)
{
  Tx.ApplyTransform(LeftRect, BottomRect);
  Tx.ApplyTransform(RightRect, TopRect);

  if(TopRect < BottomRect)
  {
    const float f = TopRect;
    TopRect = BottomRect;
    BottomRect = f;
  }
  if(RightRect < LeftRect)
  {
    const float f = RightRect;
    RightRect = LeftRect;
    LeftRect = f;
  }
}


/////////////////////////////////////////////////////////////

VectorRectangleArc::VectorRectangleArc(float iniBottomRect, float iniTopRect, 
                    float iniRightRect, float iniLeftRect, float iniHradius, float iniVradius)
                   : VectorRectangle(iniBottomRect,iniTopRect,iniRightRect,iniLeftRect)
{
  Hradius = iniHradius;
  Vradius = iniVradius;
}


void VectorRectangleArc::Transform(const AbstractTransformXY &Tx)
{
  VectorRectangle::Transform(Tx);
  Tx.ApplyTransform(Hradius, Vradius);
  if(Hradius<0) Hradius=-Hradius;
  if(Vradius<0) Vradius=-Vradius;
}


string VectorRectangleArc::Export2EPS(PS_State *PSS) const
{
float Ty = 1;
bool ContextSaved = false;

  if(fabs(Hradius)<1e-5 || fabs(Vradius)<1e-5)
      return VectorRectangle::Export2EPS(PSS);

  string PSData;

  if(PSS != NULL)
  {
    vecBrush::prepExport(PSS);    
    vecPen::prepExport(PSS);
    PS_Attr(PSData, PSS);
  }

  if(Tx) 
    ContextSaved = Tx->ApplyTransform(PSData);
  
  float Y_ll = BottomRect;
  float Y_ur = TopRect;
  if(fabs(Hradius-Vradius) > 1e-5)
  {	//Horizontal and Vertical radiuses are different - scale canvas
    Ty = Vradius/Hradius;
    if(!ContextSaved) PSData+="\ngsave";
    PSData.cat_printf("\n0 %2.2f translate\n1 %2.3f scale", Y_ll, 1.0/Ty);
    Y_ur -= Y_ll;
    Y_ll = 0;
    ContextSaved = true;
 }

 PSData += "\nnewpath";
 PSData.cat_printf("\n%2.2f %2.2f moveto", LeftRect+Hradius, Ty*Y_ll);
 PSData.cat_printf("\n%2.2f %2.2f %2.2f 270 0 arc", LeftRect-Hradius, Ty*Y_ll+Hradius , Hradius);
 PSData.cat_printf("\n%2.2f %2.2f %2.2f 0 90 arc", LeftRect-Hradius, Ty*Y_ur-Hradius, Hradius);
 PSData.cat_printf("\n%2.2f %2.2f %2.2f 90 180 arc", RightRect+Hradius, Ty*Y_ur-Hradius, Hradius);
 PSData.cat_printf("\n%2.2f %2.2f %2.2f 180 270 arc", RightRect+Hradius, Ty*Y_ll+Hradius, Hradius);

 PSData+="\nclosepath";
 FillObjectPS(RightRect, LeftRect, Ty*Y_ll, Ty*Y_ur,PSData,PSS);
 if(ContextSaved) PSData+="\ngrestore";
 return PSData;
}


/////////////////////////////////////////////////////////////


VectorPolygon::VectorPolygon(float *iniPoints, int iniPointCount)
{
  Points = iniPoints;
  CountPoints = iniPointCount;
  Close = false;
  Outline = true;  
}


VectorPolygon::~VectorPolygon()
{
  if(Points!=NULL)
  {
    free(Points);
    Points = NULL;
  }  
  CountPoints = 0;  
}


string VectorPolygon::Export2EPS(PS_State *PSS) const
{
string str;
  if(Points==NULL || CountPoints<=0) return str;

  if(PSS != NULL)
  {
    vecBrush::prepExport(PSS);
    vecPen::prepExport(PSS);    
    PS_Attr(str,PSS);
  }
  
  const float *pEnd = Points + 2*CountPoints;
  float MinPolyX=65537,MaxPolyX=-32767,MinPolyY=65537,MaxPolyY=-32767;

  for(float *p=Points; p<pEnd; p+=2)
  {
    const float x = *p;
    const float y = p[1];
    if(x<MinPolyX) MinPolyX=x;
    if(x>MaxPolyX) MaxPolyX=x;
    if(y<MinPolyY) MinPolyY=y;
    if(y>MaxPolyY) MaxPolyY=y;

    str.cat_printf(p==Points?"\nnewpath\n%2.2f %2.2f moveto":"\n%2.2f %2.2f lineto",
		    x, y);
  }
  if(Close) str += "\nclosepath";
  
  FillObjectPS(MinPolyX,MaxPolyX,MinPolyY,MaxPolyY,str,PSS);
  
  return str;
}


void VectorPolygon::Transform(const AbstractTransformXY &Tx)
{
  if(Points==NULL || CountPoints<=0) return;

  const float *pEnd = Points + 2*CountPoints;
  for(float *p=Points; p<pEnd; p+=2)
  {  
    Tx.ApplyTransform(p[0], p[1]);
  }
}


/////////////////////////////////////////////////////////////


VectorLine::VectorLine(float *iniPoints, int iniPointCount)
{
  Points = iniPoints;
  CountPoints = iniPointCount;
  Close = false;
}


VectorLine::~VectorLine()
{
  if(Points!=NULL)
  {
    free(Points);
    Points = NULL;
  }  
  CountPoints = 0;  
}


string VectorLine::Export2EPS(PS_State *PSS) const
{
string str;
	// There must be at least two points for the line.
  if(Points==NULL || CountPoints<=1) return str;

  if(PSS != NULL)
  {    
    vecPen::prepExport(PSS);    
    PS_Attr(str,PSS);
  }

  str.cat_printf("\nnewpath\n%2.2f %2.2f moveto", *Points, Points[1]);

  float *p;
  const float *pEnd = Points + 2*CountPoints;
  for(p=Points+2; p<pEnd; p+=2)
  {
    str.cat_printf("\n%2.2f %2.2f lineto", *p, p[1]);
  }
  if(Close) str += "\nclosepath";
  str += "\nstroke"; 
  return str;
}


void VectorLine::Transform(const AbstractTransformXY &Tx)
{
  if(Points==NULL || CountPoints<=0) return;

  const float *pEnd = Points + 2*CountPoints;
  for(float *p=Points; p<pEnd; p+=2)
  {  
    Tx.ApplyTransform(p[0], p[1]);
  }
}


/////////////////////////////////////////////////////////////


VectorList::VectorList(void)
{
  Objects = NULL;
  VectorObjects =0;
}


VectorList::~VectorList()
{
  if(Objects!=NULL)
  {
    for(int i=0; i<VectorObjects; i++)
    {
      if(Objects[i] != NULL)
      {
        delete(Objects[i]);
        Objects[i] = NULL;
      }
    }
    free(Objects);
    Objects = NULL;
  }  
  VectorObjects = 0;
}


string VectorList::Export2EPS(PS_State *PSS) const
{
string str;

  if(Objects!=NULL && VectorObjects>0)
  {    
    for(int i=0; i<VectorObjects; i++)
    {
      if(Objects[i]!=NULL)
      {
        str += Objects[i]->Export2EPS(PSS);
      }
    }
  }
  return str;
}


void VectorList::Transform(const AbstractTransformXY &Tx)
{
  if(Objects!=NULL && VectorObjects>0)
  {    
    for(int i=0; i<VectorObjects; i++)
    {
      if(Objects[i]!=NULL)
      {
        Objects[i]->Transform(Tx);
      }
    }
  }
}


void VectorList::AddObject(VectorObject *NewObj)
{
  if(NewObj==NULL || NewObj==this) return;

  if(Objects==NULL || VectorObjects<=0)
  {
    Objects = (VectorObject**)malloc(sizeof(VectorObject*));
    if(Objects==NULL) return;
    Objects[0] = NewObj;    
    VectorObjects = 1;
    return;
  }

  VectorObjects++;
  Objects = (VectorObject**)realloc(Objects, (VectorObjects)*sizeof(VectorObject*));
  Objects[VectorObjects-1] = NewObj;
}


/////////////////////////////////////////////////////////////


/// Symbol PostScript font, see https://en.wikipedia.org/wiki/Symbol_%28typeface%29
static const char *GreekText[] = 
    {"$\\ALPHA$", "$\\alpha$",		//A
     "$\\BETA$", "$\\beta$",		//B
     "$\\CHI$", "$\\chi$",		//C
     "$\\Delta$", "$\\delta$",		//D
     "$\\EPSILON$", "$\\epsilon$",	//E
     "$\\Phi$", "$\\phi$",		//F
     "$\\Gamma$", "$\\gamma$",		//G
     "$\\ETA$", "$\\eta$",		//H
     "$\\IOTA$", "$\\iota$",		//I
     "$\\vartheta$", "$\\varphi$",	//J
     "$\\KAPPA$", "$\\kappa$",		//K
     "$\\Lambda$", "$\\lambda$",	//L
     "$\\MU$", "$\\mu$",		//M
     "$\\NU$", "$\\nu$",		//N
     "$\\OMICRON$", "$\\omicron$",	//O
     "$\\Pi$", "$\\pi$",		//P
     "$\\Theta$", "$\\theta$",		//Q
     "$\\RHO$", "$\\rho$",		//R
     "$\\Sigma$", "$\\sigma$",		//S
     "$\\TAU$", "$\\tau$",		//T
     "$\\textupsilon$", "$\\upsilon$",	//U
     "$\\varsigma$", "$\\textcloseomega$", //V ??
     "$\\Omega$", "$\\omega$",		//W
     "$\\Xi$", "$\\xi$",		//X
     "$\\Psi$", "$\\psi$",		//Y					
     "$\\ZETA$", "$\\zeta$"		//Z
     };

static const char PsText[] = 
    {'A', 'a',
     'B', 'b',
     'C', 'c',
     'D', 'd',
     'E', 'e',
     'F', 'f',
     'G', 'g',
     'H', 'h',
     'I', 'i',
     'J', 'J',
     'K', 'k',
     'L', 'l',
     'M', 'm',
     'N', 'n',
     'O', 'o',
     'P', 'p',
     'Q', 'q',
     'R', 'r',
     'S', 's',
     'T', 't',
     'U', 'u',
     'V', 'v',
     'W', 'w',
     'X', 'x',
     'Y', 'y',
     'Z', 'x',
    };



TextContainer::TextContainer()
{
  Text = NULL;
  TextObjects = 0;
}


TextContainer::~TextContainer()
{
  if(Text!=NULL)
  {
    for(int i=0; i<TextObjects; i++)
    {
      if(Text[i] != NULL)
      {
        delete(Text[i]);
        Text[i] = NULL;
      }
    }
    free(Text);
    Text = NULL;
  }  
  TextObjects = 0;
}


string TextContainer::Export2EPS(PS_State *PSS) const
{
string str;

  if(Text!=NULL && TextObjects>0)
  {
    if(PSS!=NULL)
    {
      if(memcmp(&TextColor,&PSS->LineColor,sizeof(RGB_Record)) != 0) 
      {			// EPS does not handle separatelly text color and line color.
        PSS->dirty |= PSS_LineColor;
        memcpy(&PSS->LineColor,&TextColor,sizeof(RGB_Record));
      }    
      PS_Attr(str,PSS);
    }

    for(int i=0; i<TextObjects; i++)
    {
      if(Text[i]!=NULL)
      {
        FixPsAccent(Text[i]->contents,"\\'{","(\\302)");	// acute
        FixPsAccent(Text[i]->contents,"\\v{","(\\317)");	// charon
        FixPsAccent(Text[i]->contents,"\\r{","(\\312)");	// ring
        FixPsAccent(Text[i]->contents,"\\`{","(\\301)");	// grave

        if(i>0) str += "\nshow";
        str.cat_printf("\n/%s findfont %.2f scalefont setfont",
            Text[i]->TargetFont(), Text[i]->size*2.66f);

        if(i==0)
            str.cat_printf("\nnewpath %2.2f %2.2f moveto", PosX, PosY);

        str += "\n(";
        str += Text[i]->contents + ')';
      }      
    }
    str += "\nshow";
  }

return str;
}


void TextContainer::Transform(const AbstractTransformXY &Tx)
{
  Tx.ApplyTransform(PosX, PosY);
}


void TextContainer::AddText(temp_string contents, const PS_State &PSS)
{
  const char *PsFontName;  

  if(PSS.FontItallic)
    PsFontName = (PSS.FontWeight>=500) ? "Times-BoldItalic" : "Times-Italic";
  else
    PsFontName = (PSS.FontWeight>=500) ? "Times-Bold" : "Times-Roman";

  for(int i=0; i<sizeof(GreekText)/sizeof(char*); i++)
    {
    if(!StrCmp(contents,GreekText[i]))
      {      
      AddText(PsText[i], "Symbol", PSS);
      return;
      }    
    }
  
  AddText(contents, PsFontName, PSS);
}


void TextContainer::AddText(temp_string contents, const char *font, const PS_State &PSS)
{
TextObject *pTobj;
  if(contents.length()<=0) return;
  
  if(Text==NULL || TextObjects<=0)
  {
    Text = (TextObject**)malloc(sizeof(TextObject*));
    pTobj = Text[0] = new TextObject;
    pTobj->TargetFont = font;
    pTobj->contents = contents;
    pTobj->size = PSS.FontSize;
    //pTobj->Weight = Weight;
    TextObjects = 1;
    return;
  }

  pTobj = Text[TextObjects-1];
  if(pTobj->TargetFont==font && fabs(pTobj->size-PSS.FontSize)<1e-4)	//&& pTobj->Weight==Weight
  {
    pTobj->contents += contents;
    return;
  }

  TextObjects++;
  Text = (TextObject**)realloc(Text, (TextObjects)*sizeof(TextObject*));
  pTobj = Text[TextObjects-1] = new TextObject;
  pTobj->TargetFont = font;
  pTobj->contents = contents;
  pTobj->size = PSS.FontSize;
  //pTobj->Weight = Weight;
}


bool TextContainer::isEmpty(void) const
{
  if(Text==NULL || TextObjects<=0) return true;
  for(int i=0; i<TextObjects; i++)
    {
    if(Text[i]!=NULL && Text[i]->contents.length()>0) 
      return false;
    }
  return true;
}

/////////////////////////////////////////////////////////////

#include "typedfs.h"
#include "raster.h"


void Color2PS(string & Text, const RGB_Record & TextColor);


/** Transfer accent to PostSctipt text.
 * @param[in,out] s		String with text.
 * @param[in]	TexAccent	Accent prefix in LeTex.
 * @param[in]   PsAccent	Accent equivalent in PS.*/
static void FixPsAccent(string & s, const char *TexAccent, const char *PsAccent)
{
string Text2;
const char *pos;

 while((pos=strstr(s(),TexAccent)) != NULL)
   {
   bool CharIsUpper = false;
   if(pos>s())
     {
     if(pos[-1]=='(')
       Text2 = copy(s,0,pos-s()-1);  // Discard empty text before
     else
       {
       Text2 = copy(s,0,pos-s());
       Text2 += ") show ";	// Close previous text path
       }
     }

   if(Text2.length()<=0 && PsAccent!=NULL && PsAccent[0]=='(')
     Text2 += PsAccent + 1;
   else
     Text2 += PsAccent;
   Text2 += "(";
   pos += 3;
   while(*pos!='}')
     {
     if(*pos==0) break;
     if(isupper(*pos)) CharIsUpper = true;
     Text2 += *pos++;
   }
   if(CharIsUpper) Text2 += ") ACCENTSHOW";
              else Text2 += ") accentshow";
   Text2 += " (";
   if(*pos!=0)
     Text2 += pos+1;		// Append a rest of the string.

   s = Text2;
   }
}


int LineCountTxt(const char *Txt)
{
int LineCount = 1;

  while((Txt=strchr(Txt,'\n')) != NULL)
  {
    Txt++;
    LineCount++;
  }
  return LineCount;
}


/** Convert accents to PS equivalents. */
static void FixPsSymbol(string & s, const char *TeXSymb, const char PSsymb,
                        const string &CurrentFont, const string &SymbolFont)
{
string Text2;
const char *pos;

 while((pos=strstr(s(),TeXSymb)) != NULL)
   {
   if(pos>s())
     {
     if(pos[-1]=='(')
       Text2 = copy(s,0,pos-s()-1);  // Discard empty text before
     else
       {
       Text2 = copy(s,0,pos-s());
       Text2 += ") show ";	// Close previous text path
       }
     }
   else
     Text2 = ") show ";

   Text2 += SymbolFont;
   Text2 += "(";
   Text2 += PSsymb;
   Text2 += ") show\n";

   Text2 += CurrentFont;
   pos += StrLen(TeXSymb);

   Text2 += " (";   
   if(*pos!=0)
     Text2 += pos;		// Append a rest of the string.

   s = Text2;
   }
}


/** Convert plain text in PS sequence of operations.
 @param[in,out] PSData	Whole postscript image.
 @param[in] UpRightX	Boundind box right top x boundary in PS units.
 @param[in] UpRightY     Boundind box right top y boundary in PS units.
 @param[in] LowLeftX	Boundind box left bottom x boundary in PS units.
 @param[in] LowLeftY	Boundind box left bottom y boundary in PS units.
 @param[in] FontSize	Size of font [mm]. */
void StoreText2PS(string & PSData, string & Text, PS_State *PSS, WORD RotAngle,
		float UpRightX, float UpRightY, float LowLeftX, float LowLeftY,
		float FontSize)
{
float AdjY;

  if(Text=="") return;
  if(RotAngle>0)
	  {
	  float Xs,Ys;
	  Xs = (UpRightX+LowLeftX)/2;
	  Ys = (UpRightY+LowLeftY)/2;
	  PSData.cat_printf("\ngsave\n%2.2f %2.2f translate\n%d rotate\n%2.2f %2.2f translate",
		  Xs,Ys, RotAngle, -Xs,-Ys);
	  }

  string CurrentFont, SymbolFont;
  Font2PS(CurrentFont, FontSize*2.66f);			// Convert mm to points.
  Font2PS(SymbolFont, FontSize*2.66f, "Symbol");
  PSData += '\n';
  PSData += CurrentFont;

  AdjY = mm2PSu(FontSize);
  AdjY = LowLeftY+(UpRightY-LowLeftY)/2.0 + AdjY*LineCountTxt(Text())/2.0 - AdjY;
  //AdjY = UpRightY - AdjY;
  PSData.cat_printf("\nnewpath %2.2f %2.2f moveto\n", LowLeftX,AdjY);
  Color2PS(PSData,PSS->TextColor);
  PSS->dirty |= PSS_LineColor;

  FixPsAccent(Text,"\\'{","(\\302)");	// acute
  FixPsAccent(Text,"\\v{","(\\317)");	// charon
  FixPsAccent(Text,"\\r{","(\\312)");	// ring
  FixPsAccent(Text,"\\`{","(\\301)");	// grave

	// Convert \n to newline in PS
  const char *pos = Text();
  while((pos=strchr(pos,'\n')) != NULL)
  {
    string TextBeg = copy(Text,0,pos-Text());
    TextBeg += ") show\n";
    AdjY -= 1.05 * mm2PSu(FontSize);	// /50 [pt] /2.66 [mm]
    TextBeg.cat_printf("%2.2f %2.2f moveto\n(",  LowLeftX, AdjY);
    Text = TextBeg + (pos+1);
    pos = Text() + TextBeg.length();
  }

  for(int i=0; i<sizeof(GreekText)/sizeof(char*); i++)
    FixPsSymbol(Text, GreekText[i],PsText[i], CurrentFont,SymbolFont);
  
  if(!strncmp(Text(),") show",6))
    {					// Remove relict from FixPsSymbol
    PSData += '\n';
    PSData += Text() + 6;
    }
  else
    {
    PSData += "\n(";
    PSData += Text;
    }
  
  if(length(PSData)>=2 && PSData[length(PSData)-1]=='(' && PSData[length(PSData)-2]!='\\')
    {
    PSData[length(PSData)-1] = '\n';
    }
  else
    PSData += ")\nshow";

  if(RotAngle>0) PSData+="\ngrestore";
}
